<?php
/**
 * Mailbox model file.
 *
 * @package Model
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

use App\Encryption;
use App\Mail\Account;
use App\Mail\Account\Entity\AccountPrivateEntity;
use App\Mail\Account\Entity\Enum\Status;
use App\Mail\Account\Repository\AccountPrivateRepository;
use App\Mail\Server;
use App\User;

class Users_Mailbox_Model extends \App\Base
{
	private array $editFields = [
		'name', 'mail_server_id', 'login', 'password', 'status', 'folders',  'scanner_actions', 'scanner_fields', 'email_exceptions'
	];

	private AccountPrivateRepository $repository;
	private AccountPrivateEntity $entity;

	private int $id;

	public function __construct(int $recordId)
	{
		$this->id = $recordId;
		$this->repository = new AccountPrivateRepository();
		$this->entity = $this->repository->getByUserId($recordId);
		if (!$this->entity->getId()) {
			$this->entity->set('owner_id', User::getCurrentUserRealId());
		}
	}

	public function isExists(): bool
	{
		return (bool) $this->entity->getId();
	}

	public function getOAuthUrl(): string
	{
		return "index.php?module=Users&action=OAuth&record={$this->getId()}";
	}

	public function getId()
	{
		return $this->id;
	}

	public function getRedirectUrl(): string
	{
		$reloadPage = ($this->entity->getServer()?->isOAuth() && array_intersect_key($this->entity->getChanges(), array_flip(['password', 'mail_server_id', 'login']))) || (isset($this->entity->getChanges()['status']) && Status::STATUS_ACTIVE === $this->entity->getStatus());

		return $reloadPage ? $this->getOAuthUrl() : '';
	}

	/**
	 * Function to set the value for a given key.
	 *
	 * @param string $key
	 * @param mixed  $value
	 */
	public function set($key, $value)
	{
		$value = $this->sanitize($key, $value);
		if ($this->entity->get($key) !== $value) {
			$this->entity->set($key, $value);
		}

		return $this;
	}

	public function sanitize($key, $value)
	{
		if ('mail_server_id' === $key) {
			$value = (int) $value;
		} elseif ('password' === $key) {
			$value = Encryption::getInstance()->decrypt($value);
		}

		return $value;
	}

	public function getAccount(): Account
	{
		return new Account($this->entity, $this->repository);
	}

	public function getFieldsForEdit(): array
	{
		$fields = [];
		foreach ($this->editFields as $fieldName) {
			$fields[$fieldName] = $this->getFieldInstanceByName($fieldName);
		}
		return $fields;
	}

	public function save()
	{
		return $this->repository->save($this->entity);
	}

	public function getStructure(bool $onlyField = false): array
	{
		$structure = [];

		foreach ($this->getFieldsForEdit() as $fieldName => $fieldModel) {
			if ($this->entity->has($fieldName)) {
				$fieldModel->set('fieldvalue', $this->entity->get($fieldName));
			} else {
				$defaultValue = $fieldModel->get('defaultvalue');
				$fieldModel->set('fieldvalue', $defaultValue ?? '');
			}
			if ($onlyField) {
				$structure[$fieldName] = $fieldModel;
			} else {
				$block = $fieldModel->get('blockLabel') ?: '';
				$structure[$block][$fieldName] = $fieldModel;
			}
		}

		return $structure;
	}

	/**
	 * Get fields instance by name.
	 *
	 * @param string $name
	 *
	 * @return \Vtiger_Field_Model|null
	 */
	public function getFieldInstanceByName(string $name): ?Vtiger_Field_Model
	{
		$moduleName = 'Users';
		$params = [];
		switch ($name) {
			case 'name':
				$params = [
					'name' => $name,
					'label' => 'FL_MAILBOX_SUBJECT',
					'uitype' => 1,
					'typeofdata' => 'V~M',
					'maximumlength' => '100',
					'purifyType' => \App\Purifier::TEXT,
					'blockLabel' => 'BL_MAILBOX_BASE',
					'defaultvalue' => User::getCurrentUserModel()->getDetail('email1'),
				];
				break;
			case 'login':
				$params = [
					'name' => $name,
					'label' => 'FL_MAILBOX_LOGIN',
					'uitype' => 1,
					'typeofdata' => 'V~M',
					'maximumlength' => '255',
					'purifyType' => \App\Purifier::TEXT,
					'blockLabel' => 'BL_MAILBOX_BASE',
					'defaultvalue' => User::getCurrentUserModel()->getDetail('email1'),
				];
				break;
			case 'status':
				$params = [
					'name' => $name,
					'label' => 'FL_MAILBOX_STATUS',
					'uitype' => 16,
					'typeofdata' => 'V~M',
					'maximumlength' => '255',
					'purifyType' => \App\Purifier::TEXT,
					'blockLabel' => 'BL_MAILBOX_BASE',
					'defaultvalue' => 'PLL_ACTIVE',
					'picklistValues' => [
						'PLL_ACTIVE' => \App\Language::translate('PLL_ACTIVE', $moduleName),
						'PLL_INACTIVE' => \App\Language::translate('PLL_INACTIVE', $moduleName),
						'PLL_LOCKED' => \App\Language::translate('PLL_LOCKED', $moduleName)
					]
				];
				break;
			case 'password':
				$params = [
					'name' => $name,
					'label' => 'FL_MAILBOX_PASSWORD',
					'uitype' => 99,
					'typeofdata' => 'V~O',
					'maximumlength' => '255',
					'purifyType' => 'raw',
					'blockLabel' => 'BL_MAILBOX_BASE',
					'fromOutsideList' => true,
					'field'
				];
				$moduleName = 'OSSMailScanner';
				break;
			case 'mail_server_id':
				$params = [
					'name' => $name,
					'label' => 'FL_MAILBOX_SERVER',
					'uitype' => 16,
					'typeofdata' => 'I~M',
					'maximumlength' => '128',
					'purifyType' => \App\Purifier::INTEGER,
					'blockLabel' => 'BL_MAILBOX_BASE',
				];
				$params['picklistValues'] = array_map(fn ($server) => $server['name'], Server::getAll(Server::STATUS_ACTIVE));
				break;
			case 'scanner_actions':
				$params = [
					'name' => $name,
					'label' => 'FL_SCANNER_ACTIONS',
					'uitype' => 322,
					'typeofdata' => 'V~O',
					'maximumlength' => '255',
					'purifyType' => \App\Purifier::TEXT,
					'blockLabel' => 'BL_MAILBOX_SCANNER',
					'tdClass' => 'text-wrap',
					'defaultvalue' => '',
				];
				break;
			case 'folders':
				$params = [
					'name' => $name,
					'label' => 'FL_MAILBOX_FOLDERS',
					'uitype' => 89,
					'typeofdata' => 'V~O',
					'maximumlength' => '65535',
					'purifyType' => \App\Purifier::TEXT,
					'blockLabel' => 'BL_MAILBOX_SCANNER',
					'tooltip' => 'LBL_MAILBOX_FOLDERS_DESC',
					'defaultvalue' => '',
				];
				break;
			case 'logs':
				$params = [
					'name' => $name,
					'label' => 'FL_LOGS',
					'uitype' => 21,
					'typeofdata' => 'V~O',
					'maximumlength' => '65535',
					'purifyType' => \App\Purifier::TEXT,
					'blockLabel' => 'BL_SMTP',
					'defaultvalue' => '',
				];
				break;
			case 'scanner_fields':
				$params = [
					'name' => $name,
					'label' => 'FL_SCANNER_FIELDS',
					'uitype' => 323,
					'typeofdata' => 'V~O',
					'maximumlength' => '65535',
					'purifyType' => \App\Purifier::TEXT,
					'blockLabel' => 'BL_MAILBOX_SCANNER',
					'defaultvalue' => '',
				];
				break;
			case 'email_exceptions':
				$params = [
					'name' => $name,
					'label' => 'FL_EMAIL_EXCEPTIONS',
					'uitype' => 314,
					'typeofdata' => 'V~O',
					'maximumlength' => '65535',
					'purifyType' => \App\Purifier::TEXT,
					'blockLabel' => 'BL_MAILBOX_SCANNER',
					'defaultvalue' => '',
					'fieldparams' => '{"consenticon":false}',
				];
				break;
			default:
				break;
		}

		return $params ? \Vtiger_Field_Model::init($moduleName, $params, $name) : null;
	}
}
