<?php

/**
 * Inventory Currency Field Class.
 *
 * @package   InventoryField
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
class Vtiger_Currency_InventoryField extends Vtiger_Basic_InventoryField
{
	protected $type = 'Currency';
	protected $defaultLabel = 'LBL_CURRENCY';
	protected $columnName = 'currency';
	protected $dbType = [\yii\db\Schema::TYPE_INTEGER, 11];
	protected $customColumn = [
		'currencyparam' => [\yii\db\Schema::TYPE_STRING, 1024],
	];
	protected $blocks = [0];
	protected $maximumLength = '-2147483648,2147483647';
	protected $customMaximumLength = [
		'currencyparam' => 1024
	];
	protected $purifyType = \App\Purifier::INTEGER;
	protected $customPurifyType = [
		'currencyparam' => App\Purifier::TEXT
	];
	/** {@inheritdoc} */
	protected $params = ['reset_currency', 'cit_exchange'];

	/** {@inheritdoc} */
	public function getDisplayValue($value, array $rowData = [], bool $rawText = false)
	{
		if (empty($value)) {
			return '';
		}
		return \App\Fields\Currency::getById($value)['currency_name'];
	}

	/**
	 * Gets currency param.
	 *
	 * @param array  $currencies
	 * @param string $param
	 * @param string $date
	 *
	 * @throws \App\Exceptions\AppException
	 *
	 * @return array
	 */
	public function getCurrencyParam(array $currencies, string $param = '', string $date = '')
	{
		$params = [];
		if ($param) {
			$params = \App\Json::decode($param);
		}
		foreach ($currencies as $currency) {
			if (!isset($params[$currency['id']])) {
				$params[$currency['id']] = vtlib\Functions::getConversionRateInfo($currency['id'], $date);
			}
		}
		return $params;
	}

	/**
	 * Returns currency param object representation
	 *
	 * @param string $jsonParam
	 * @return \App\Inventory\Currency\Param
	 * @throws \App\Exceptions\AppException
	 */
	public function createCurrencyParamObject(string $jsonParam): \App\Inventory\Currency\Param
	{
		return \App\Inventory\Currency\Param::fromJson($jsonParam);
	}

	/** {@inheritdoc} */
	public function getDBValue($value, ?string $name = '')
	{
		if ($name === $this->getColumnName()) {
			$value = (int) $value;
		}
		return $value;
	}

	/** {@inheritdoc} */
	public function validate($value, string $columnName, bool $isUserFormat, $originalValue = null)
	{
		if ($columnName === $this->getColumnName()) {
			if (!is_numeric($value) || !isset(\App\Fields\Currency::getAll()[$value])) {
				throw new \App\Exceptions\Security("ERR_ILLEGAL_FIELD_VALUE||$columnName||" . print_r($value, true), 406);
			}
		} elseif (!\is_array($value) && \App\TextUtils::getTextLength($value) > $this->customMaximumLength[$columnName]) {
			throw new \App\Exceptions\Security("ERR_VALUE_IS_TOO_LONG||$columnName||" . print_r($value, true), 406);
		}
	}

	/** {@inheritdoc} */
	public function compare($value, $prevValue, string $column): bool
	{
		return $column === $this->getColumnName() ? (int) $value === (int) $prevValue : parent::compare($value, $prevValue, $column);
	}

	/** {@inheritdoc} */
	public function getConfigFieldsData(): array
	{
		$data = parent::getConfigFieldsData();
		$data['reset_currency'] = [
			'name' => 'reset_currency',
			'label' => 'LBL_INV_CURRENCY_RESET',
			'uitype' => 56,
			'maximumlength' => '1',
			'typeofdata' => 'C~O',
			'purifyType' => \App\Purifier::INTEGER,
			'defaultvalue' => 0,
			'tooltip' => 'LBL_INV_CURRENCY_RESET_DESC'
		];
		$data['cit_exchange'] = [
			'name' => 'cit_exchange',
			'label' => 'LBL_INV_ADDITIONAL_CIT_EXCHANGE',
			'uitype' => 56,
			'maximumlength' => '1',
			'typeofdata' => 'C~O',
			'purifyType' => \App\Purifier::INTEGER,
			'defaultvalue' => 0,
			'tooltip' => 'LBL_INV_ADDITIONAL_CIT_EXCHANGE_DESC'
		];

		return $data;
	}

	/**
	 * Returns currency conversation summary grouped by tax groups.
	 *
	 * @param Vtiger_Record_Model $model
	 *
	 * @return array
	 */
	public function getCurrencyConversationSummary(Vtiger_Record_Model $model): array
	{
		$inventoryModel = Vtiger_Inventory_Model::getInstance($this->getModuleName());
		$currency = $inventoryModel->getCurrency($model);
		$defaultCurrency = \App\Fields\Currency::getDefault();

		$taxSummary = static::getInstance($this->getModuleName(), 'Tax')->getTaxSummary($model);
		if (
			$currency['id'] === $defaultCurrency['id']
			|| !$taxSummary
		) {
			return [];
		}

		$inventory = $model->getInventoryData();
		$row = key($inventory);

		$currencyParam = $this->getCurrencyParam([], $inventory[$row]['currencyparam']);
		$conversionRate = (float) $currencyParam[$defaultCurrency['id']]['conversion'] / (float) $currencyParam[(int) $inventory[$row]['currency']]['conversion'];

		$groups = [];
		$total = 0;
		foreach ($taxSummary['groups'] as $taxGroup => $value) {
			$result = $this->roundDecimal($value * $conversionRate);
			$groups[$taxGroup] = $result;
			$total += $result;
		}

		return [
			'groups' => $groups,
			'total' => $this->roundDecimal($total),
		];
	}
}
