<?php
/**
 * Sanitize fields
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Leszek Koziatek <l.koziatek@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Report\Sanitizer;

use App\FieldCoordinatorTransformer\QueryGeneratorFieldTransformer;
use App\Fields\Currency;
use App\Module;
use App\Purifier;
use App\Record;
use App\Report\Factory\FieldFactory;
use App\Report\Model\Expression;
use App\Report\Model\Field;
use App\Report\Model\Query;
use Vtiger_Module_Model;
use Vtiger_Basic_InventoryField;
use Throwable;

/** ReportFieldSanitizer class */
final class ReportFieldSanitizer implements ReportSanitizer
{
	public function __construct(
		private readonly FieldFactory $fieldFactory,
	) {
	}

	/** {@inheritDoc} */
	public function sanitize(Query $query, array $data = []): array
	{
		$result = [];

		foreach ($data as $i => $row) {
			$result[$i] = [];
			foreach ($row as $fieldCoordinate => $value) {
				try {
					$field = $this->fieldFactory->create($query, $fieldCoordinate);
					$result[$i][$fieldCoordinate] = $this->prepareToXlsExport($field, $value);
				} catch (Throwable $throwable) {
					$result[$i][$fieldCoordinate] = $this->getNonFieldValue($fieldCoordinate, $value, $query);
				}
			}
		}

		return $result;
	}

	/** Get display data for expression not related with Yetiforce fields */
	private function getNonFieldValue(string $fieldCoordinate, mixed $value, Query $query): mixed
	{
		$fieldParts = QueryGeneratorFieldTransformer::extract($fieldCoordinate);

		$modulePK = '';
		$fieldName = $fieldParts['fieldName'];

		if (null !== $moduleName = $fieldParts['moduleName']) {
			$moduleName = preg_replace('/_.*/', '', $moduleName);

			if (true === Module::isModuleActive($moduleName)) {
				$moduleModel = Vtiger_Module_Model::getInstance($moduleName);
				$modulePK = $moduleModel->basetableid;
			}
		}

		return ('id' === $fieldName || $modulePK === $fieldName) && false === $this->isFunctionField($query, $fieldParts)
			? Record::getLabel($value)
			: $value;
	}

	/** Check if field is related with any functions */
	private function isFunctionField(Query $query, array $fieldParts): bool
	{
		/** @var array<Expression> $relatedExpression */
		$relatedExpression = array_values(array_filter(
			$query->getExpressions(),
			static fn(Expression $expression): bool => QueryGeneratorFieldTransformer::extract($expression->getAlias()) === $fieldParts,
		));

		return count($relatedExpression) > 0 && null !== $relatedExpression[0]->getFunction();
	}

	/** Get data in format prepared for xls file */
	private function prepareToXlsExport(Field $fieldModel, mixed $value): int|string|float
	{
		/** @var \Vtiger_Field_Model|Vtiger_Basic_InventoryField $nativeModel */
		$nativeModel = $fieldModel->getNativeModel();

		return match ($nativeModel->getFieldDataType()) {
			'integer', 'double' => $nativeModel->getDisplayValue($value, recordModel: true, rawText: true),
			'currency', 'currencyInventory' => number_format((float) $value, 2, '.', ''),
			'date', 'datetime' => false === is_numeric($value) ? $nativeModel->getDisplayValue($value, true) : $value,
			'phone' => $nativeModel->getDisplayValue($value, false, false, true),
			'text' => trim(Purifier::purify(preg_replace("/\s+/", " ", $value))),
			'image', 'multiImage' => '',
			'categoryMultipicklist' => null === $value ? '' : $nativeModel->getDisplayValue($value, false, true, true),
			'inventory' => $this->parseInventoryData($nativeModel, $value),
			default => ($nativeModel instanceof Vtiger_Basic_InventoryField)
				? $nativeModel->getDisplayValue($value, [], true)
				: $nativeModel->getDisplayValue($value, false, false, true),
		};
	}

	/** Parse data from inventory field */
	private function parseInventoryData(Vtiger_Basic_InventoryField $fieldType, $value): int|string
	{
		return match ($fieldType->getType()) {
			'Margin', 'TotalPrice' => number_format((float) $value, 2, '.', ''),
			'Currency' => (Currency::getById($value))['currency_code'],
			'Date', 'Name', 'Reference', 'Value', 'Unit', 'Boolean', 'Comment', 'Picklist', 'PicklistField',
			'DiscountMode', 'TaxMode' => $fieldType->getDisplayValue($value, rawText: true),
			default => $fieldType->getDisplayValue($value, rawText: true),
		};
	}
}
