<?php

/**
 * Password login provider.
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

namespace App\Authenticator;

/**
 * Abstract class for password login provider.
 */
class Password extends AbstractProvider
{
	/** @var \App\User User model */
	protected $userModel;

	/**
	 * Set user model.
	 *
	 * @param \App\User $userModel
	 *
	 * @return $this
	 */
	public function setUserModel(\App\User $userModel): self
	{
		$this->userModel = $userModel;
		return $this;
	}

	/** {@inheritdoc} */
	public function verify(#[\SensitiveParameter] string &$password, string $userName): bool
	{
		$userModel = $this->userModel;
		$pwd = $password;
		$password = '[REDACTED]';

		$fail = true;
		if (empty($pwd)) {
			\App\Log::warning('Password not provided', 'UserAuthentication');
		} elseif (!$userModel->getDetail('user_name') || mb_strtolower($userModel->getDetail('user_name')) !== mb_strtolower($userName)) {
			\App\Log::warning('User not found: ' . $userName, 'UserAuthentication');
		} elseif (!$userModel->isActive()) {
			\App\Log::warning('Inactive user :' . $userName, 'UserAuthentication');
		} else {
			$fail = false;
		}

		return $this->verifyPassword($pwd, $fail) && !$fail;
	}

	/**
	 * Verifies that a password matches a hash.
	 *
	 * @param string $password Reference added for anonymization as SensitiveParameter is not supported for PHP < 8.2
	 * @param bool   $falsify  Slower logon for security purposes
	 *
	 * @return bool
	 */
	public function verifyPassword(#[\SensitiveParameter] string &$password, bool $falsify = false): bool
	{
		$pwd = $password;
		$password = '[REDACTED]';

		$password1 = '-'.$pwd;
		$hash = $falsify ? $this->encryptPassword($password1) : $this->userModel->getDetail('user_password');
		return password_verify($pwd, $hash);
	}

	/**
	 * Encrypt user password.
	 *
	 * @param string $password User password. Reference added for anonymization as SensitiveParameter is not supported for PHP < 8.2
	 *
	 * @return string Encrypted password
	 */
	public function encryptPassword(#[\SensitiveParameter] string &$password): string
	{
		$pwd = $password;
		$password = '[REDACTED]';

		return password_hash($pwd, PASSWORD_BCRYPT, ['cost' => \App\Config::security('USER_ENCRYPT_PASSWORD_COST')]);
	}

	/** {@inheritdoc} */
	public function isMFA(): bool
	{
		return \Users_Totp_Authmethod::isActive($this->userModel->getId()) && !\in_array(\App\RequestUtil::getRemoteIP(true), \App\Config::security('whitelistIp2fa', [])) && !empty($this->userModel->getDetail('authy_secret_totp'));
	}

	/** {@inheritdoc} */
	public function verifyMFA(string $code): bool
	{
		try {
			$userId = $this->userModel->getId();
			$authMethod = new \Users_Totp_Authmethod($userId);
			$result = $authMethod->verifyCode($this->userModel->getDetail('authy_secret_totp'), $code);
		} catch (\Throwable $th) {
			\App\Log::error($th->getMessage(), 'UserAuthentication');
			$result = false;
		}
		return $result;
	}

	/** {@inheritdoc} */
	public function preProcess()
	{
		\App\Session::set('UserAuthMethod', 'PASSWORD');
	}

	/** {@inheritdoc} */
	public function logIn()
	{
		\App\Session::set('authenticated_user_id', $this->userModel->getId());
	}

	/** {@inheritdoc} */
	public function postProcess()
	{
		\Users_Module_Model::getInstance('Users')->saveLoginHistory($this->userModel->getDetail('user_name'), 'Signed in');
		if ('PASSWORD' === \App\Session::get('UserAuthMethod') && empty($this->userModel->getDetail('authy_secret_totp')) && \Users_Totp_Authmethod::isActive($this->userModel->getId())) {
			\App\Process::addEvent([
				'name' => 'ShowAuthy2faModal',
				'priority' => 7,
				'execution' => 'TOTP_OPTIONAL' === \App\Config::security('USER_AUTHY_MODE') ? 'once' : 'constant',
				'type' => 'modal',
				'url' => 'index.php?module=Users&view=TwoFactorAuthenticationModal&record=' . $this->userModel->getId(),
			]);
		}
		if ($this->userModel->isAdmin() && \App\Config::security('askAdminAboutVisitPurpose', true)) {
			\App\Process::addEvent([
				'name' => 'showVisitPurpose',
				'type' => 'modal',
				'url' => 'index.php?module=Users&view=VisitPurpose',
			]);
		}
		if (\App\YetiForce\Shop::verify()) {
			\App\Process::addEvent([
				'name' => 'YetiForceShopAlert',
				'type' => 'modal',
				'execution' => 'once',
				'url' => 'index.php?module=Users&view=YetiForce&mode=shop',
			]);
		}
		if (\App\YetiForce\Register::shouldEnforceRegistration()) {
			if ($this->userModel->isAdmin()) {
				\App\Process::addEvent(\Settings_Companies_EditModal_View::MODAL_EVENT);
			} else {
				\App\Process::addEvent(\Users_NoAccessModal_View::MODAL_EVENT);
			}
		}

		if (!\App\YetiForce\Register::isPreRegistered()) {
			if ($this->userModel->isAdmin()) {
				\App\Process::addEvent(\Settings_Companies_EmailVerificationModal_View::MODAL_EVENT);
			} else {
				\App\Process::addEvent(\Users_NoAccessModal_View::MODAL_EVENT);
			}
		}
	}

	/** {@inheritDoc} */
	public function getFieldInstanceByName(string $name): ?\Vtiger_Field_Model
	{
		return null;
	}
}
