<?php

namespace App\Conditions\QueryFields;

use App\Db\Query;
use App\Fields\Owner;
use App\Layout;
use App\PrivilegeUtil;
use App\User;
use yii\db\Expression;

/**
 * Shared Owner Query Field Class.
 *
 * @package UIType
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
class SharedOwnerField extends BaseField
{
	/** {@inheritdoc} */
	public function getColumnName(): string
	{
		if ($this->fullColumnName) {
			return $this->fullColumnName;
		}

		if ($this->related) {
			$relatedField = $this->getRelated();
			$relatedField['relatedField'] = 'id';
			$tableName = $this->queryGenerator->resolveRelTableName($relatedField) . $this->getRelTablePostfix();
			$tableIndex = $this->queryGenerator->resolveRelTableIndex($relatedField);
			return $this->fullColumnName = \sprintf('%s.%s', $tableName, $tableIndex);
		}

		$focus = $this->queryGenerator->getEntityModel();
		return $this->fullColumnName = "{$focus->table_name}.{$focus->table_index}";
	}

	public function addJoin()
	{
		if ($this->related) {
			$queryGenerator = $this->queryGenerator;
			$sourceFieldName = $this->getRelated('sourceField');

			$sourceFieldModel = $queryGenerator->getModuleField($sourceFieldName);
			$sourceTableName = $sourceFieldModel->getModule()->getBaseTableName();
			$sourceTableIndex = $sourceFieldModel->getColumnName();

			$relatedField = $this->getRelated();
			$relatedField['relatedField'] = 'id';
			$relTableName = $queryGenerator->resolveRelTableName($relatedField);
			$relTableIndex = $queryGenerator->resolveRelTableIndex($relatedField);
			$relTableAlias = \sprintf('%s%s', $relTableName, $sourceFieldName);

			$queryGenerator->addJoin([
				'LEFT JOIN',
				\sprintf('%s %s', $relTableName, $relTableAlias),
				\sprintf('%s.%s=%s.%s', $sourceTableName, $sourceTableIndex, $relTableAlias, $relTableIndex)
			]);
		}
	}

	/** {@inheritdoc} */
	public function operatorE(): array
	{
		if (empty($this->value)) {
			return [];
		}
		return [$this->getColumnName() => (new Query())->select(['crmid'])->from('u_#__crmentity_showners')->where(['userid' => explode('##', $this->value)])];
	}

	/** {@inheritdoc} */
	public function operatorN(): array
	{
		if (empty($this->value)) {
			return [];
		}
		return ['NOT IN', $this->getColumnName(), (new Query())->select(['crmid'])->from('u_#__crmentity_showners')->where(['userid' => explode('##', $this->value)])];
	}

	/**
	 * Currently logged user.
	 *
	 * @return array
	 */
	public function operatorOm()
	{
		$this->value = User::getCurrentUserId();
		return $this->operatorE();
	}

	/**
	 * Currently logged-in user groups.
	 *
	 * @return array
	 */
	public function operatorOgr(): array
	{
		$this->value = implode('##', array_keys(Owner::getInstance($this->getModuleName())->getGroups(false, 'private')));
		return $this->operatorE();
	}

	/**
	 * Users who belong to the same group as the currently logged in user.
	 *
	 * @return array
	 */
	public function operatorOgu(): array
	{
		$groups = Owner::getInstance($this->getModuleName())->getGroups(false, 'private');
		if ($groups) {
			$where = ['or'];
			$query = (new Query())->select(['crmid'])->from('u_#__crmentity_showners');
			foreach (array_keys($groups) as $groupId) {
				$where[] = ['u_#__crmentity_showners.userid' => (new Query())->select(['userid'])->from(["condition_groups_{$groupId}_" . Layout::getUniqueId() => PrivilegeUtil::getQueryToUsersByGroup((int) $groupId)])];
			}
			$condition = [$this->getColumnName() => $query->where($where)];
		} else {
			$condition = [$this->getColumnName() => (new Expression('0=1'))];
		}
		return $condition;
	}

	/** {@inheritdoc} */
	public function operatorC(): array
	{
		return $this->operatorE();
	}

	/**
	 * Is not empty operator.
	 *
	 * @return array
	 */
	public function operatorNy(): array
	{
		return [$this->getColumnName() => $this->getSubQuery()];
	}

	/**
	 * Is empty operator.
	 *
	 * @return array
	 */
	public function operatorY(): array
	{
		return ['NOT IN', $this->getColumnName(), $this->getSubQuery()];
	}

	public function getSelectQuery(): Query
	{
		$focus = $this->queryGenerator->getEntityModel();
		$columnName = 'u_#__crmentity_showners.userid';
		$expression = new Expression("GROUP_CONCAT({$columnName} SEPARATOR ',')");
		$query = (new Query())->select([$expression])->from('u_#__crmentity_showners');

		if ($this->related) {
			$query->where(['u_#__crmentity_showners.crmid' => new Expression($this->getColumnName())]);
		} else {
			$focus = $this->queryGenerator->getEntityModel();
			$query->where(['u_#__crmentity_showners.crmid' => new Expression("{$focus->table_name}.{$focus->table_index}")]);
		}

		return $query;
	}

	/** {@inheritDoc} */
	protected function getRelTablePostfix(): string
	{
		$sourceFieldName = $this->getRelated('sourceField');
		$type = (string) $this->getRelated('advancedType');

		return $sourceFieldName . $type;
	}

	private function getSubQuery(): Query
	{
		if ($this->related) {
			$fieldModel = $this->queryGenerator->getModuleField($this->related['sourceField']);
			$query = (new Query())->select(['crmid'])->from('u_#__crmentity_showners')
				->innerJoin($fieldModel->getTableName(), "u_#__crmentity_showners.crmid={$fieldModel->getTableName()}.{$fieldModel->getColumnName()}");
		} else {
			$focus = $this->queryGenerator->getEntityModel();
			$query = (new Query())->select(['crmid'])->from('u_#__crmentity_showners')
				->innerJoin($focus->table_name, "u_#__crmentity_showners.crmid={$focus->table_name}.{$focus->table_index}");
		}

		return $query;
	}
}
