<?php

/**
 * Main file that includes basic operations on relations.
 *
 * @package   Relation
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 */

use App\Condition;
use App\Db;
use App\Db\Query;
use App\QueryGenerator;
use App\Relation\RelationAbstraction;

/**
 * Documents_GetAttachments_Relation class.
 */
class Documents_GetAttachments_Relation extends RelationAbstraction
{
	/**
	 * @var string Name of the table that stores relations.
	 */
	public const TABLE_NAME = 'vtiger_senotesrel';

	/** {@inheritdoc} */
	public function getRelationType(): int
	{
		return Vtiger_Relation_Model::RELATION_M2M;
	}

	/** {@inheritdoc} */
	public function getQuery()
	{
		$queryGenerator = $this->relationModel->getQueryGenerator();
		$queryGenerator->setCustomColumn('vtiger_notes.filetype');
		$tableName = self::TABLE_NAME;
		$queryGenerator->addJoin(['INNER JOIN', $tableName, "{$tableName}.notesid= vtiger_notes.notesid"]);
		$queryGenerator->addJoin(['INNER JOIN', 'vtiger_crmentity crm2', "crm2.crmid = {$tableName}.crmid"]);
		$queryGenerator->addNativeCondition(['crm2.crmid' => $this->relationModel->get('parentRecord')->getId()]);
		$queryGenerator->setOrder('id', 'DESC');
	}

	public function getQueryForReport(QueryGenerator $queryGenerator, array $relField): void
	{
		$tableName = static::TABLE_NAME;
		$query = $queryGenerator->getQueryRelatedField($relField);
		$relTableName = $queryGenerator->resolveRelTableName($relField);
		$relatedTableIndex = $queryGenerator->resolveRelTableIndex($relField);
		$relTableAlias = $query->getTableName();

		$queryGenerator->addJoin(['LEFT JOIN', $tableName, \sprintf('%s.crmid = vtiger_crmentity.crmid', $tableName)]);
		$queryGenerator->addJoin(['LEFT JOIN', 'vtiger_crmentity crm2', \sprintf('crm2.crmid = %s.notesid', $tableName)]);
		$queryGenerator->addJoin(
			[
				'LEFT JOIN',
				\sprintf('%s %s', $relTableName, $relTableAlias),
				\sprintf('crm2.crmid = %s.%s', $relTableAlias, $relatedTableIndex),
			],
		);
	}

	/**
	 * Load advanced conditions for filtering related records.
	 *
	 * @param QueryGenerator $queryGenerator QueryGenerator for the list of records to be tapered based on the relationship
	 * @param array          $searchParam
	 *
	 * @return void
	 */
	public function loadAdvancedConditionsByRelationId(QueryGenerator $queryGenerator): void
	{
		$tableName = static::TABLE_NAME;
		$advancedConditions = $queryGenerator->getAdvancedConditions();
		$relationQueryGenerator = $this->relationModel->getQueryGenerator();
		$relationQueryGenerator->setStateCondition($queryGenerator->getState());
		$relationQueryGenerator->setFields(['id']);
		if (!empty($advancedConditions['relationConditions'])) {
			$relationQueryGenerator->setConditions(Condition::getConditionsFromRequest($advancedConditions['relationConditions']));
		}
		$query = $relationQueryGenerator->createQuery();
		$queryGenerator->addJoin(['INNER JOIN', $tableName, "{$tableName}.crmid = vtiger_crmentity.crmid"])
			->addNativeCondition(["{$tableName}.notesid" => $query]);
	}

	/**
	 * Load advanced conditions relationship by custom column.
	 *
	 * @param QueryGenerator $queryGenerator QueryGenerator for the list of records to be tapered based on the relationship
	 * @param array          $searchParam    Related record for which we are filtering the list of records
	 *
	 * @return void
	 */
	public function loadAdvancedConditionsByColumns(QueryGenerator $queryGenerator, array $searchParam): void
	{
		$tableName = static::TABLE_NAME;
		$queryGenerator->addJoin(['INNER JOIN', $tableName, "{$tableName}.crmid = vtiger_crmentity.crmid"])
			->addNativeCondition(["{$tableName}.notesid" => $searchParam['value']]);
	}

	/** {@inheritdoc} */
	public function delete(int $sourceRecordId, int $destinationRecordId): bool
	{
		$data = ['notesid' => $destinationRecordId, 'crmid' => $sourceRecordId];
		if ($this->relationModel && 'Accounts' === $this->relationModel->getParentModuleModel()->getName()) {
			$subQuery = (new Query())->select(['contactid'])->from('vtiger_contactdetails')->where(['parentid' => $sourceRecordId]);
			$data = ['or', $data, ['crmid' => $subQuery] + $data];
		}
		return (bool) Db::getInstance()->createCommand()->delete(self::TABLE_NAME, $data)->execute();
	}

	/** {@inheritdoc} */
	public function create(int $sourceRecordId, int $destinationRecordId): bool
	{
		$result = false;
		$data = ['notesid' => $destinationRecordId, 'crmid' => $sourceRecordId];
		if (!(new Query())->from(self::TABLE_NAME)->where($data)->exists()) {
			$result = Db::getInstance()->createCommand()->insert(self::TABLE_NAME, $data)->execute();
		}
		return (bool) $result;
	}

	/** {@inheritdoc} */
	public function transfer(int $relatedRecordId, int $fromRecordId, int $toRecordId): bool
	{
		return (bool) Db::getInstance()->createCommand()->update(
			self::TABLE_NAME,
			['crmid' => $toRecordId],
			['crmid' => $fromRecordId, 'notesid' => $relatedRecordId]
		)->execute();
	}
}
