<?php
/**
 * Basic transfer ownership model file.
 *
 * @package Model
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 */

use App\Base;
use App\Db\Query;
use App\Privilege;
use App\Record;

/**
 * Basic transfer ownership model class.
 */
class Vtiger_TransferOwnership_Model extends Base
{
	/** @var string[] List of modules to skip during ownership transfer */
	protected $skipModules = [];

	/**
	 * Get the list of modules to skip during ownership transfer.
	 *
	 * @return string[]
	 */
	public function getSkipModules(): array
	{
		return $this->skipModules;
	}

	/**
	 * Get related module record IDs based on relation type.
	 *
	 * @param string $basicModule
	 * @param array  $recordIds
	 * @param array  $relModData
	 *
	 * @return array
	 */
	public function getRelatedModuleRecordIds(string $basicModule, array $recordIds, array $relModData): array
	{
		$parentModuleModel = Vtiger_Module_Model::getInstance($basicModule);
		$relatedIds = [];
		$relatedModule = $relModData[0];
		$type = $relModData[1];
		switch ($type) {
			case 0:
				$field = $relModData[2];
				foreach ($recordIds as $recordId) {
					$recordModel = Vtiger_Record_Model::getInstanceById($recordId, $basicModule);
					if (0 != $recordModel->get($field) && Record::getType($recordModel->get($field)) == $relatedModule) {
						$relatedIds[] = $recordModel->get($field);
					}
				}
				break;
			case 1:
				$tablename = Vtiger_Relation_Model::getInstance($parentModuleModel, Vtiger_Module_Model::getInstance($relatedModule))->getRelationField()->get('table');
				$tabIndex = CRMEntity::getInstance($relatedModule)->table_index;
				$relIndex = $this->getRelatedColumnName($relatedModule, $basicModule);
				if (!$relIndex) {
					break;
				}
				$relatedIds = (new Query())->select([$tabIndex])->from($tablename)->where([$relIndex => $recordIds])->column();
				break;
			case 2:
				foreach ($recordIds as $recordId) {
					$recordModel = Vtiger_Record_Model::getInstanceById($recordId, $basicModule);
					$relationListView = Vtiger_RelationListView_Model::getInstance($recordModel, $relatedModule);
					$relatedIds = $relationListView->getRelationQuery()->select(['vtiger_crmentity.crmid'])
						->distinct()
						->column();
				}
				break;
			default:
				break;
		}
		return array_unique($relatedIds);
	}

	/**
	 * Transfer ownership of records to another user.
	 *
	 * @param string $module
	 * @param int    $transferOwnerId
	 * @param array  $relatedModuleRecordIds
	 *
	 * @return void
	 */
	public function transferRecordsOwnership(string $module, int $transferOwnerId, array $relatedModuleRecordIds): void
	{
		foreach ($relatedModuleRecordIds as $record) {
			$recordModel = Vtiger_Record_Model::getInstanceById($record, $module);
			if ($recordModel->isEditable()) {
				$recordModel->set('assigned_user_id', $transferOwnerId);
				$recordModel->save();
			}
		}
	}

	/**
	 * Get instance of the transfer ownership model for a module.
	 *
	 * @param string $module
	 *
	 * @return static
	 */
	public static function getInstance(string $module): static
	{
		$instance = Vtiger_Cache::get('transferOwnership', $module);
		if (!$instance) {
			$modelClassName = Vtiger_Loader::getComponentClassName('Model', 'TransferOwnership', $module);
			$instance = new $modelClassName();
			$instance->set('module', $module);
			Vtiger_Cache::set('transferOwnership', $module, $instance);
		}
		return $instance;
	}

	/**
	 * Get related modules by reference fields.
	 *
	 * @return array
	 */
	public function getRelationsByFields(): array
	{
		$module = $this->get('module');
		$moduleModel = Vtiger_Module_Model::getInstance($module);
		$relatedModelFields = $moduleModel->getFields();

		$relatedModules = [];
		foreach ($relatedModelFields as $fieldName => $fieldModel) {
			if ($fieldModel->isReferenceField()) {
				$referenceList = $fieldModel->getReferenceList();
				foreach ($referenceList as $relation) {
					if (Privilege::isPermitted($relation, 'EditView')) {
						$relatedModules[] = ['name' => $relation, 'field' => $fieldName, 'label' => $fieldModel->getLabel()];
					}
				}
			}
		}
		return $relatedModules;
	}

	/**
	 * Get related modules by related list.
	 *
	 * @return array
	 */
	public function getRelationsByRelatedList(): array
	{
		$module = $this->get('module');
		$moduleModel = Vtiger_Module_Model::getInstance($module);
		$relatedModules = [];
		foreach ($moduleModel->getRelations() as $relation) {
			$relationModule = $relation->getRelationModuleName();
			if (Privilege::isPermitted($relationModule, 'EditView')) {
				$relatedModules[] = [
					'name' => $relationModule,
					'type' => $relation->getRelationType(),
				];
			}
		}
		return $relatedModules;
	}

	/**
	 * Get related column name for a module.
	 *
	 * @param string $relatedModule
	 * @param string $findModule
	 *
	 * @return string|null
	 */
	protected function getRelatedColumnName($relatedModule, $findModule): ?string
	{
		$relatedModuleModel = Vtiger_Module_Model::getInstance($relatedModule);
		$relatedModelFields = $relatedModuleModel->getFields();
		foreach ($relatedModelFields as $fieldModel) {
			if ($fieldModel->isReferenceField()) {
				$referenceList = $fieldModel->getReferenceList();
				if (\in_array($findModule, $referenceList)) {
					return $fieldModel->get('column');
				}
			}
		}
		return null;
	}
}
