<?php
/**
 * File generation strategy class
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Leszek Koziatek <l.koziatek@yetiforce.com>
 */
declare(strict_types=1);

namespace App\DocumentBuilder;

use App\Report\Model\Template;

/** FileGenerator class */
final class FileGenerator
{
	/** @var string directory for storing generated files. Files will be removed from directory after adding to Document module */
	public const DOCUMENT_DIR = 'cache/reports';

	/** @var FileGeneratorInterface used generator */
	private FileGeneratorInterface $fileGenerator;

	/** @var array data set used in generated file */
	private array $data = [];

	/** Set file generator */
	public function setFileGenerator(FileGeneratorInterface $fileGenerator): self
	{
		$this->fileGenerator = $fileGenerator;

		return $this;
	}

	/** Add more data to final data set */
	public function addData(array $dataset): self
	{
		$this->data = array_merge($this->data, $dataset);

		return $this;
	}

	/** Get current data set */
	public function getData(): array
	{
		return $this->data;
	}

	/**
	 * Generate file from current data
	 *
	 * @param Template $template
	 * @param array $headers - names of data inserted into first row
	 * @param array $groupedBy
	 *
	 * @return void
	 */
	public function generateFile(Template $template, array $headers = [], array $groupedBy = []): void
	{
		$filename = $this->getReportFilename($template);

		$this->fileGenerator->generate($filename, $this->data, $headers, $groupedBy);
	}

	/** Get filepath for generated file */
	public function getFilePath(string $filename): string
	{
		return $this->fileGenerator->getFilePath($filename);
	}

	/** Generate filename from used report */
	public function getReportFilename(Template $report): string
	{
		return $report->getName();
	}
}
