<?php

/**
 * Comarch abstract inventory map file.
 *
 * The file is part of the paid functionality. Using the file is allowed only after purchasing a subscription.
 * File modification allowed only with the consent of the system producer.
 *
 * @package Integration
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 */

namespace App\Integrations\Comarch;

/**
 * Comarch abstract inventory map class.
 */
abstract class InventoryMap extends Map
{
	/** @var Map The parent map instance. */
	public $parentMap;
	/** @var array Inventory mapped fields. */
	protected $invFieldMap = [];
	/** @var array Inventory data from YetiForce. */
	protected $invDataYf;
	/** {@inheritdoc} */
	protected $defaultDataYf = [
		'invFieldMap' => [
			'discountmode' => 1,
			'discountparam' => '{"aggregationType":"individual","individualDiscountType":"amount","individualDiscount":0}',
		]
	];
	/** {@inheritdoc} */
	protected $defaultDataApi = [];

	/**
	 * Get inventory data from/for YetiForce.
	 *
	 * @param bool $cache
	 *
	 * @return array
	 */
	public function getInvDataYf(bool $cache = true): array
	{
		if (null !== $this->invDataYf && true === $cache) {
			return $this->invDataYf;
		}
		$this->invDataYf = [];
		foreach ($this->dataApi['pozycje'] as $item) {
			$self = clone $this;
			$self->parentMap = $this;
			$self->setDataApi($item);
			$invDataYf = $self->getDataYf('invFieldMap');
			if (empty($this->dataYf['currency_id']) && !empty($invDataYf['currency'])) {
				$this->dataYf['currency_id'] = $invDataYf['currency'];
			}
			$this->invDataYf[] = $invDataYf;
		}
		return $this->invDataYf;
	}

	/**
	 * Load additional data.
	 *
	 * @return void
	 */
	public function loadAdditionalData(): void
	{
		if ($this->skip) {
			return;
		}
		if ($invDataYf = $this->getInvDataYf()) {
			if ($invRecord = $this->recordModel->getInventoryData()) {
				foreach ($invDataYf as &$newRowValue) {
					foreach ($invRecord as $keyRecord => $valueRecord) {
						if ($valueRecord['name'] == $newRowValue['name']) {
							$newRowValue['id'] = $valueRecord['id'];
							$newRowValue['crmid'] = $valueRecord['crmid'];
							unset($invRecord[$keyRecord]);
							break;
						}
					}
				}
			}
			$this->recordModel->initInventoryData($invDataYf, false);
		}
	}

	/** {@inheritdoc} */
	public function getDataApi(string $type = 'fieldMap', bool $mapped = true): array
	{
		if ($mapped) {
			parent::getDataApi($type, $mapped);
			if ('fieldMap' === $type) {
				$this->dataApi['Pozycje'] = $this->getInvDataApi();
			}
		}
		return $this->dataApi;
	}

	/**
	 * Get inventory data from/for API.
	 *
	 * @return array
	 */
	public function getInvDataApi(): array
	{
		$invDataApi = [];
		foreach ($this->recordModel->getInventoryData() as $row) {
			$self = clone $this;
			$self->setDataApi([]);
			$self->setDataYf($row);
			$invDataApi[] = $self->getDataApi('invFieldMap');
		}
		return $invDataApi;
	}

	/**
	 * Convert payment date to system YF format.
	 *
	 * @param mixed $value
	 * @param array $field
	 * @param bool  $fromApi
	 *
	 * @return string
	 */
	protected function convertPaymentDate($value, array $field, bool $fromApi): string
	{
		if ($fromApi) {
			return $value;
		}
		$paymentDiff = 0;
		if ($value && $this->dataYf['issue_time']) {
			$paymentDiff = \App\Fields\DateTime::getDiff($value, $this->dataYf['issue_time'], 'days');
		}
		return $paymentDiff;
	}

	/**
	 * Convert inventory tax in YF/API.
	 *
	 * @param mixed $value
	 * @param array $field
	 * @param bool  $fromApi
	 *
	 * @return float|string string (YF) or string (API)
	 */
	protected function convertInvTax($value, array $field, bool $fromApi)
	{
		$newValue = $this->findBySynchronizer($value, $field, $fromApi);
		if ($fromApi) {
			$this->dataYf['taxmode'] = 1;
			return \App\Json::encode([
				'aggregationType' => 'global',
				'globalTax' => $newValue['value'],
				'globalId' => $newValue['id'],
			]);
		}
		$this->dataApi['StawkaPodatku'] = (int) $newValue['value'];
		return $newValue['id'];
	}

	/**
	 * Convert inventory description in YF/API.
	 *
	 * @param mixed $value
	 * @param array $field
	 * @param bool  $fromApi
	 *
	 * @return string
	 */
	protected function convertInvDesc($value, array $field, bool $fromApi): string
	{
		if (empty($value)) {
			return '';
		}
		if ($fromApi) {
			return nl2br(trim($value));
		}
		return strip_tags(str_replace('<br>', PHP_EOL, $value));
	}

	/**
	 * Convert address in YF/API.
	 *
	 * @param mixed $value
	 * @param array $field
	 * @param bool  $fromApi
	 *
	 * @return int|string|null string (YF) or int (API)
	 */
	protected function convertAddress($value, array $field, bool $fromApi)
	{
		if ($fromApi) {
			$return = $this->convertAddressFromApi($value, $field);
		} else {
			$return = $this->convertAddressToApi($value, $field);
		}
		if (null === $return && (!\array_key_exists('optional', $field) || empty($field['optional']))) {
			if (!$fromApi) {
				$this->skip = true;
			}
			$this->synchronizer->controller->log(
				($fromApi ? '[API>YF]' : '[YF>API]') .
				"Skip value: {$value} (Field: {$field['fieldCrm']} ,Address type: {$field['addressType']})",
				['fieldConfig' => $field, 'data' => $this->dataApi],
				null,
				true
			);
			$this->setErrorLog([
				$field['fieldCrm'] => ['field' => $value, 'message' => 'ERR_REQUIRED_VALUE_MISSING']
			]);
			if (null !== $this->recordModel && !$fromApi) {
				$this->recordModel->save();
			}
		}
		return $return;
	}

	/**
	 * Convert address from API.
	 *
	 * @param mixed $value
	 * @param array $field
	 *
	 * @return string
	 */
	protected function convertAddressFromApi($value, array $field): ?string
	{
		$synchronizer = $this->synchronizer->controller->getSync('AccountAddress');
		$return = null;
		foreach ($synchronizer->getAddress($this->dataApi[$field['accountKey']]) as $id => $address) {
			if ($value === $id) {
				foreach ($address as $name => $val) {
					$this->dataYf[$name . $field['addressType']] = $val;
					if ('addresslevel5' === $name) {
						$return = $val;
					}
				}
				break;
			}
		}
		return $return;
	}

	/**
	 * Convert address to API.
	 *
	 * @param mixed $value
	 * @param array $field
	 *
	 * @return int|null
	 */
	protected function convertAddressToApi($value, array $field): ?int
	{
		if (empty($this->dataApi['Akronim']) || $this->addressYfIsEmpty($field)) {
			return 0;
		}
		$addressId = null;
		$synchronizer = $this->synchronizer->controller->getSync('AccountAddress');
		foreach ($synchronizer->getAddress($this->dataApi['Akronim']) as $id => $address) {
			if ($this->compareAddress($address, $field)) {
				$addressId = $id;
				break;
			}
		}
		if (null === $addressId) {
			$addressId = $synchronizer->createAddress($this->getYfAddress($field));
		}
		return $addressId;
	}

	/**
	 * Compare two addresses to see if they are identical.
	 *
	 * @param array $address
	 * @param array $field
	 *
	 * @return bool
	 */
	protected function compareAddress(array $address, array $field): bool
	{
		$status = true;
		foreach ($field['addressFields'] as $addressName) {
			$fieldName = $addressName . $field['addressType'];
			if ($this->dataYf[$fieldName] != $address[$addressName]) {
				$status = false;
			}
		}
		return $status;
	}

	/**
	 * Get address to create.
	 *
	 * @param array $field
	 *
	 * @return array
	 */
	protected function getYfAddress(array $field): array
	{
		$address = ['account_id' => $this->dataYf[$field['accountField']]];
		foreach ($field['addressFields'] as $addressName) {
			$fieldName = $addressName . $field['addressType'];
			$address[$addressName] = $this->dataYf[$fieldName];
		}
		return $address;
	}

	/**
	 * Check if the address is empty.
	 *
	 * @param array $field
	 *
	 * @return bool
	 */
	protected function addressYfIsEmpty(array $field): bool
	{
		$status = true;
		foreach ($field['addressFields'] as $addressName) {
			if (!empty($this->dataYf[$addressName . $field['addressType']])) {
				$status = false;
			}
		}
		return $status;
	}
}
