<?php
/**
 * Connector to find routing. Connector based on service GraphHopper.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 *
 * @see       https://developers.google.com/maps/documentation/routes/reference/rest/v2/TopLevel/computeRoutes
 */

namespace App\Map\Routing;

/**
 * Connector for service GraphHopper to get routing.
 */
class GoogleRoutes extends Base
{
	/**
	 * API URL.
	 */
	private const API_URL = 'https://routes.googleapis.com/directions/v2:computeRoutes';

	/**
	 * @inheritdoc
	 */
	protected string $label = 'LBL_ROUTING_GOOGLE_ROUTES';

	/**
	 * @inheritdoc
	 */
	protected string $docUrl = 'https://developers.google.com/maps/documentation/routes/reference/rest/v2/TopLevel/computeRoutes';

	/**
	 * @inheritdoc
	 */
	protected array $formFields = [
		'api_key' => [
			'validator' => [['name' => 'AlphaNumeric']],
			'uitype' => 99,
			'label' => 'LBL_API_KEY',
			'purifyType' => \App\Purifier::ALNUM_EXTENDED,
			'maximumlength' => '200',
			'typeofdata' => 'V~M',
		],
		'travelMode' => [
			'validator' => [['name' => 'AlphaNumeric']],
			'uitype' => 16,
			'label' => 'LBL_TRAVEL_MODE',
			'purifyType' => \App\Purifier::TEXT,
			'maximumlength' => '11',
			'typeofdata' => 'V~0',
			'defaultvalue' => 'DRIVE',
			'doc' => 'https://developers.google.com/maps/documentation/routes/reference/rest/v2/RouteTravelMode',
			'picklistValues' => [
				'DRIVE' => 'DRIVE',
				'TRANSIT' => 'TRANSIT',
				'BICYCLE' => 'BICYCLE',
				'WALK' => 'WALK',
				'TWO_WHEELER' => 'TWO_WHEELER',
			],
		],
		'routingPreference' => [
			'validator' => [['name' => 'AlphaNumeric']],
			'uitype' => 16,
			'label' => 'LBL_ROUTING_PREFERENCE',
			'purifyType' => \App\Purifier::TEXT,
			'maximumlength' => '21',
			'typeofdata' => 'V~O',
			'defaultvalue' => 'TRAFFIC_UNAWARE',
			'doc' => 'https://developers.google.com/maps/documentation/routes/reference/rest/v2/RoutingPreference',
			'picklistValues' => [
				'TRAFFIC_UNAWARE' => 'TRAFFIC_UNAWARE',
				'TRAFFIC_AWARE' => 'TRAFFIC_AWARE',
				'TRAFFIC_AWARE_OPTIMAL' => 'TRAFFIC_AWARE_OPTIMAL',
			],
		],
	];

	/** {@inheritdoc} */
	public function calculate(): void
	{
		if (!\App\RequestUtil::isNetConnection()) {
			return;
		}
		$options = [
			'timeout' => 120,
			'http_errors' => false,
			'headers' => [
				'X-Goog-Api-Key' => $this->getConfig()['api_key'],
				'X-Goog-FieldMask' => $this->getFieldMask(),
			],
			'json' => array_merge([
				'polylineEncoding' => 'GEO_JSON_LINESTRING',
				'computeAlternativeRoutes' => false,
				'optimizeWaypointOrder' => true,
				'languageCode' => \App\Language::getLanguage(),
			], $this->getConfig()['params'], $this->parsePoints()),
		];
		$url = self::API_URL;
		\App\Log::beginProfile("POST|GoogleRoutes::calculate|{$url}", __NAMESPACE__);
		$response = \App\RequestHttp::getClient()->post($url, $options);
		\App\Log::endProfile("POST|GoogleRoutes::calculate|{$url}", __NAMESPACE__);

		$body = \App\Json::decode($response->getBody()->getContents());

		if (200 !== $response->getStatusCode() || !empty($body['error'])) {
			\App\Log::error(
				'Error: ' . $url . ' | ' . ($body['error']['message'] ?? $response->getReasonPhrase()),
				__CLASS__
			);
			return;
		}
		$this->parseResponse($body);
	}

	/** {@inheritdoc} */
	public function parsePoints(): array
	{
		$points = [
			'origin' => $this->getWaypoint($this->start)
		];
		if (!empty($this->indirectPoints)) {
			foreach ($this->indirectPoints as $tempLon) {
				$points['intermediates'][] = $this->getWaypoint($tempLon);
			}
		}
		$points['destination'] = $this->getWaypoint($this->end);
		return $points;
	}

	/**
	 * Parse response from API.
	 *
	 * @param array $body
	 *
	 * @return void
	 */
	private function parseResponse(array $body): void
	{
		$coordinates = [];
		$description = '';
		if (!empty($body['routes'])) {
			foreach ($body['routes'] as $route) {
				$this->distance += $route['distanceMeters'];
				if ($this->returnDetails['duration']) {
					$this->travelTime += \App\Fields\Time::formatToSeconds($route['duration']);
				}
				if ($this->returnDetails['polyline']) {
					$coordinates = array_merge($coordinates, $route['polyline']['geoJsonLinestring']['coordinates']);
				}
				if ($this->returnDetails['navigationInstruction']) {
					foreach ($route['legs'] as $leg) {
						foreach ($leg['steps'] as $step) {
							$description .= $step['navigationInstruction']['instructions'];
							$description .= ($step['localizedValues'] ? ' (' . $step['localizedValues']['distance']['text'] . ')' : '');
							$description .= '<br>';
						}
					}
				}
			}
		}
		$this->geoJson = [
			'type' => 'LineString',
			'coordinates' => $coordinates,
		];
		$this->description = $description;
		$this->distance = round($this->distance/1000, 1);
	}

	/**
	 * Get location waypoint.
	 *
	 * @param array $point `['lat' => 0, 'lon' => 0]`
	 *
	 * @return array
	 */
	private function getWaypoint(array $point): array
	{
		return [
			'location' => [
				'latLng' => [
					'latitude' => $point['lat'],
					'longitude' => $point['lon']
				]
			]
		];
	}

	/**
	 * Get field mask.
	 *
	 * @return string
	 */
	private function getFieldMask(): string
	{
		$fields = ['routes.distanceMeters', 'routes.optimized_intermediate_waypoint_index'];
		if ($this->returnDetails['duration']) {
			$fields[] = 'routes.duration';
		}
		if ($this->returnDetails['polyline']) {
			$fields[] = 'routes.polyline.geoJsonLinestring';
		}
		if ($this->returnDetails['navigationInstruction']) {
			$fields[] = 'routes.legs.steps.navigationInstruction';
			$fields[] = 'routes.legs.steps.localizedValues';
		}
		return implode(',', $fields);
	}
}
