<?php
/**
 * Interface file for ModTracker repository.
 *
 * @package \App\ModTracker
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mateusz Slominski <m.slominski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\ModTracker\Repository;

/**
 * Modtracker repository interface class.
 */
interface ModTrackerRepositoryInterface
{
	/**
	 * Creates ModTracker entry.
	 *
	 * @param array $entry
	 * @param array $changeDetails
	 *
	 * @return string|int
	 */
	public function createEntry(array $entry, array $changeDetails = []): string|int;

	/**
	 * Creates ModTracker relation entry.
	 *
	 * @param array $entry
	 *
	 * @return string|int
	 */
	public function createRelationEntry(array $entry): string|int;

	/**
	 * Creates ModTracker inventory entry.
	 *
	 * @param array $entry
	 *
	 * @return string
	 */
	public function createInventoryEntry(array $entry): string;

	/**
	 * Returns ModTracker relations entries for given base entry id.
	 *
	 * @param string|int $id
	 *
	 * @return array
	 */
	public function findRelationById(string|int $id): array;

	/**
	 * Returns ModTracker relations entries for given base entries ids.
	 *
	 * @param array $ids
	 *
	 * @return array
	 */
	public function getRelationsByIds(array $ids): array;

	/**
	 * Sets "user reviewed" marker to last change entry.
	 *
	 * @param int $userId
	 * @param int $recordId
	 * @param int $status
	 *
	 * @return mixed
	 */
	public function setLastReviewed(int $userId, int $recordId, int $status);

	/**
	 * Removes "user reviewed" marker from last matching ModTracker entry.
	 *
	 * @param int        $userId
	 * @param int        $recordId
	 * @param int        $statusException
	 * @param int|string $entryException
	 *
	 * @return mixed
	 */
	public function unsetReviewed(int $userId, int $recordId, int $statusException, int|string $entryException);

	/**
	 * Updates "user reviewed" marker for given ModTracker entry.
	 *
	 * @param int|string $id
	 * @param array      $lastReviewedUsers
	 *
	 * @return mixed
	 */
	public function setReviewed(int|string $id, array $lastReviewedUsers);

	/**
	 * @param int $crmId
	 * @param int $status
	 *
	 * @return mixed
	 */
	public function getReviewChanges(int $crmId, int $status);

	/**
	 * Returns last_reviewed_users for last ModTracker entry.
	 *
	 * @param int $recordId
	 * @param int $status
	 *
	 * @return mixed
	 */
	public function getLastReviewedUsers(int $recordId, int $status);

	/**
	 * Returns total change entries for given record.
	 *
	 * @param int         $recordId
	 * @param string|null $type
	 *
	 * @return mixed
	 */
	public function getTotalRecordCount(int $recordId, ?string $type = null);

	/**
	 * Returns matching change entries.
	 *
	 * @param int                  $parentRecordId
	 * @param \Vtiger_Paging_Model $pagingModel
	 * @param string               $type
	 * @param int|null             $startWith
	 *
	 * @return mixed
	 */
	public function getUpdates(int $parentRecordId, \Vtiger_Paging_Model $pagingModel, string $type, ?int $startWith = null);

	/**
	 * Get field history for given field.
	 *
	 * @param int    $record
	 * @param string $fieldName
	 *
	 * @return array
	 */
	public function getFieldHistory(int $record, string $fieldName): array;

	/**
	 * Returns changes details.
	 *
	 * @param int|string $id
	 *
	 * @return array
	 */
	public function getDetail(int|string $id): array;

	/**
	 * Returns inventory changes for given modtracker entry.
	 *
	 * @param int|string $id
	 *
	 * @return array
	 */
	public function getInventoryChanges(int|string $id): array;

	/**
	 * Performs change of entry owners.
	 *
	 * @param int $oldOwnerId
	 * @param int $newOwnerId
	 */
	public function changeEntryOwner(int $oldOwnerId, int $newOwnerId): void;

	/**
	 * Returns recent activity for given record.
	 *
	 * @param int    $recordId
	 * @param string $startDateTime
	 * @param array  $statuses
	 * @param int    $authorIdExclusion
	 * @param int    $limit
	 *
	 * @return array
	 */
	public function getRecentActivityForRecord(int $recordId, string $startDateTime, array $statuses, int $authorIdExclusion, int $limit = 5): array;

	/**
	 * Returns activity history for given record.
	 *
	 * @param int         $recordId
	 * @param array       $statusesExclusion
	 * @param int         $limit
	 * @param int         $offset
	 * @param string|null $startingIndex
	 *
	 * @return array
	 */
	public function getActivityForRecord(int $recordId, array $statusesExclusion, int $limit, int $offset, ?string $startingIndex = null): array;

	/**
	 * Returns recent changes for whole crm.
	 *
	 * @param int        $limit
	 * @param int        $offset
	 * @param array|null $modulesExclusion
	 *
	 * @return mixed
	 */
	public function getHistory(int $limit, int $offset, ?array $modulesExclusion = null, ?array $modulesInclusion = null);

	/**
	 * Returns unreviewed changes count for given user and record.
	 *
	 * @param int|int[] $recordId
	 * @param array     $statuses
	 * @param int       $userId
	 * @param bool      $sort
	 *
	 * @return array
	 */
	public function getUnreviewedChangesCount($recordId, array $statuses, int $userId, bool $sort): array;

	/**
	 *  Gets updates.
	 *
	 * @param string               $moduleName
	 * @param array                $actions
	 * @param array                $dateRange
	 * @param int|null             $owner
	 * @param int|null             $historyOwner
	 * @param \Vtiger_Paging_Model $pagingModel
	 *
	 * @return array
	 */
	public function getUpdatesForWidget(string $moduleName, array $actions, array $dateRange, ?int $owner, ?int $historyOwner, \Vtiger_Paging_Model $pagingModel): array;

	/**
	 * Gets summary.
	 *
	 * @param array    $modules
	 * @param array    $actions
	 * @param array    $dateRange
	 * @param int|null $owner
	 * @param int|null $historyOwner
	 *
	 * @return array
	 */
	public function getSummary(array $modules, array $actions, array $dateRange, ?int $owner, ?int $historyOwner): array;
}
