<?php
/**
 * ModTracker data access implementation for MongoDB file.
 *
 * @package \App\ModTracker
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mateusz Slominski <m.slominski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\ModTracker\Repository\Mongo;

use App\Db\Mongo;
use App\Db\Mongo\Query;
use App\Exceptions\AppException;
use App\ModTracker\Repository\ModTrackerRepositoryInterface;

/**
 * ModTracker data access implementation for MongoDB class.
 */
class ModTrackerRepository implements ModTrackerRepositoryInterface
{
	/**
	 * Constructor.
	 *
	 * @param Mongo $db
	 */
	public function __construct(
		private Mongo $db
	) {
	}

	/**
	 * {@inheritDoc}
	 */
	public function createEntry(array $entry, array $changeDetails = []): string
	{
		$entry['detail'] = $changeDetails;

		$objectId = $this->db->createCommand()->insert('vtiger_modtracker_basic', $entry);

		return (string) $objectId;
	}

	/**
	 * {@inheritDoc}
	 */
	public function createRelationEntry(array $entry): string
	{
		$objectId = $this->db->createCommand()->insert('vtiger_modtracker_relations', $entry);

		return (string) $objectId;
	}

	/**
	 * {@inheritDoc}
	 */
	public function createInventoryEntry(array $entry): string
	{
		$objectId = $this->db->createCommand()->insert('u_#__modtracker_inv', $entry);

		return (string) $objectId;
	}

	/**
	 * {@inheritDoc}
	 */
	public function getUpdates(int $parentRecordId, \Vtiger_Paging_Model $pagingModel, string $type, ?int $startWith = null)
	{
		$startIndex = $pagingModel->getStartIndex();
		$pageLimit = $pagingModel->getPageLimit();
		$where = $this->getConditionByType($type);
		$query = (new Query())
			->from('vtiger_modtracker_basic')
			->where(['crmid' => $parentRecordId])
			->andWhere($where)
			->limit($pageLimit)
			->offset($startIndex)
			->orderBy(['changedon' => Query::SORT_DESC]);
		if (!empty($startWith)) {
			$query->andWhere(['>=', '_id', $startWith]);
		}

		return $query->buildCursor();
	}

	/**
	 * {@inheritDoc}
	 */
	public function setLastReviewed(int $userId, int $recordId, int $status)
	{
		$row = (new Query())->select(['last_reviewed_users', '_id'])
			->from('vtiger_modtracker_basic')
			->where(['crmid' => $recordId])
			->andWhere(['not', 'status', $status])
			->orderBy(['changedon' => Query::SORT_DESC, '_id' => Query::SORT_DESC])
			->limit(1)
			->one();

		if ($row) {
			$lastReviewedUsers = $row['last_reviewed_users'];
			$lastReviewedUsers[] = $userId;

			$this->setReviewed($row['_id'], $lastReviewedUsers);

			return $row['_id'];
		}

		return false;
	}

	/**
	 * {@inheritDoc}
	 */
	public function unsetReviewed($userId, $recordId, $statusException, $entryException)
	{
		$query = new Query();
		$query->select(['last_reviewed_users', '_id'])
			->from('vtiger_modtracker_basic')
			->where(['crmid' => $recordId])
			->andWhere(['not', 'status', $statusException])
			->andWhere(['in', 'last_reviewed_users', $userId])
			->orderBy(['changedon' => Query::SORT_DESC, '_id' => Query::SORT_DESC])
			->limit(1);

		if ($entryException) {
			$query->andWhere(['not', '_id', $entryException]);
		}

		$row = $query->one();

		if ($row) {
			$lastReviewedUsers = array_filter($row['last_reviewed_users']);
			$key = array_search($userId, $lastReviewedUsers);
			unset($lastReviewedUsers[$key]);

			$this->setReviewed($row['_id'], $lastReviewedUsers);

			return $row['_id'];
		}

		return false;
	}

	/**
	 * {@inheritDoc}
	 */
	public function findRelationById(string|int $id): array
	{
		return (new Query())->from('vtiger_modtracker_relations')->where(['id' => $id])->one();
	}

	/**
	 * {@inheritDoc}
	 */
	public function getRelationsByIds(array $ids): array
	{
		return (new Query())->from('vtiger_modtracker_relations')->where(['in', 'id', $ids])->all();
	}

	/**
	 * {@inheritDoc}
	 */
	public function getLastReviewedUsers(int $recordId, int $status): array
	{
		$lastReviewedUsers = (new Query())->select(['last_reviewed_users'])->from('vtiger_modtracker_basic')
			->where(['crmid' => $recordId])
			->andWhere(['not', 'status', $status])
			->orderBy(['changedon' => Query::SORT_DESC, '_id' => Query::SORT_DESC])
			->one();

		return \is_array($lastReviewedUsers['last_reviewed_users']) ? $lastReviewedUsers['last_reviewed_users'] : [];
	}

	/**
	 * {@inheritDoc}
	 */
	public function getTotalRecordCount($recordId, ?string $type = null): int
	{
		return (new Query())
			->from('vtiger_modtracker_basic')
			->where(['crmid' => $recordId])
			->andWhere($this->getConditionByType($type))
			->count();
	}

	/**
	 * {@inheritDoc}
	 */
	public function getFieldHistory(int $record, string $fieldName): array
	{
		$query = (new Query())
			->select(['changedon', "detail.$fieldName.prevalue", "detail.$fieldName.postvalue"])
			->from('vtiger_modtracker_basic')
			->where(['crmid' => $record, "detail.$fieldName" => ['$exists' => true]])
			->orderBy(['_id' => Query::SORT_ASC]);

		return $query->all();
	}

	/**
	 * {@inheritDoc}
	 */
	public function setReviewed($id, array $lastReviewedUsers)
	{
		return $this->db->createCommand()->update('vtiger_modtracker_basic', ['_id' => $id], ['last_reviewed_users' => array_filter($lastReviewedUsers)]);
	}

	/**
	 * {@inheritDoc}
	 */
	public function getReviewChanges(int $crmId, int $status)
	{
		$query = (new Query())
			->select(['last_reviewed_users', '_id', 'changedon'])
			->from('vtiger_modtracker_basic')
			->where(['crmid' => $crmId])
			->andWhere(['not', 'status', $status])
			->orderBy(['changedon' => Query::SORT_DESC, '_id' => Query::SORT_DESC]);

		return $query->buildCursor();
	}

	/**
	 * {@inheritDoc}
	 */
	public function getDetail(int|string $id): array
	{
		return (new Query())->select(['detail.*'])->from('vtiger_modtracker_basic')->where(['_id' => $id])->all();
	}

	/**
	 * {@inheritDoc}
	 */
	public function getInventoryChanges(int|string $id): array
	{
		$result = (new Query())->select(['changes'])->from('u_#__modtracker_inv')->where(['id' => $id])->one();

		return \is_array($result) ? $result['changes'] : [];
	}

	/**
	 * {@inheritDoc}
	 */
	public function changeEntryOwner(int $oldOwnerId, int $newOwnerId): void
	{
		$this->db->createCommand()->update('vtiger_modtracker_basic', ['whodid' => $oldOwnerId], ['whodid' => $newOwnerId]);
	}

	/**
	 * {@inheritDoc}
	 */
	public function getRecentActivityForRecord(int $recordId, string $startDateTime, array $statuses, int $authorIdExclusion, int $limit = 5): array
	{
		$query = (new Query())
			->from('vtiger_modtracker_basic')
			->where(['crmid' => $recordId])
			->andWhere(['>=', 'changedon', $startDateTime])
			->andWhere(['not', 'whodid', $authorIdExclusion])
			->andWhere(['status' => $statuses])
			->orderBy(['_id' => \App\Db\Mongo\Query::SORT_ASC])->limit($limit);

		return $query->all();
	}

	/**
	 * {@inheritDoc}
	 */
	public function getActivityForRecord(int $recordId, array $statusesExclusion, int $limit, int $offset, ?string $startingIndex = null): array
	{
		$query = (new Query())
			->from('vtiger_modtracker_basic')
			->where(['crmid' => $recordId])
			->andWhere(['not in', 'status', $statusesExclusion])
			->limit($limit)
			->offset($offset)
			->orderBy(['changedon' => SORT_DESC]);
		if ($startingIndex) {
			$query->andWhere(['>=', '_id', $startingIndex]);
		}

		return $query->all();
	}

	/**
	 * {@inheritDoc}
	 */
	public function getUnreviewedChangesCount($recordId, array $statuses, int $userId, bool $sort): array
	{
		$query = (new Query())
			->select(['crmid', 'last_reviewed_users', 'status'])
			->from('vtiger_modtracker_basic')
			->where(['crmid' => $recordId])
			->andWhere(['not in', 'status', $statuses])
			->orderBy(['crmid' => Query::SORT_ASC, '_id' => Query::SORT_DESC]);

		$modTrackerRecords = $query->all();

		if ($sort) {
			$trackerIds = array_map(fn ($id) => (string) $id, array_column($modTrackerRecords, '_id'));
			$relations = $this->getRelationsByIds($trackerIds);
			$relationsIndexed = [];
			foreach ($relations as $relation) {
				$relationsIndexed[$relation['id']] = $relation['targetid'];
			}

			$ossmailView = (new \App\Db\Query())
				->select(['vtiger_ossmailview.type', 'vtiger_ossmailview.ossmailviewid'])
				->from('vtiger_ossmailview')
				->where(['vtiger_ossmailview.ossmailviewid' => array_values($relationsIndexed)])
				->indexBy('ossmailviewid')
				->createCommand()
				->queryAll();
		}

		$changes = [];
		foreach ($modTrackerRecords as $record) {
			$id = (string) $record['_id'];
			$details = $record;
			if ($sort) {
				$details['ossmailview'] = isset($relationsIndexed[$id]) && isset($ossmailView[$relationsIndexed[$id]])
					? $ossmailView[$relationsIndexed[$id]]
					: null;
			}
			$changes[$record['crmid']][] = $details;
		}

		$unreviewed = [];
		foreach ($changes as $crmId => $rows) {
			$all = $mails = 0;
			foreach ($rows as $row) {
				if (\is_array($row['last_reviewed_users']) && \in_array($userId, $row['last_reviewed_users'])) {
					break;
				}
				if (isset($row['ossmailview']['type']) && 1 === (int) $row['ossmailview']['type']) {
					++$mails;
				} elseif (!isset($row['ossmailview']['type'])) {
					++$all;
				}
			}
			$unreviewed[$crmId]['a'] = $all;
			$unreviewed[$crmId]['m'] = $mails;
		}
		return $unreviewed;
	}

	/**
	 * {@inheritDoc}
	 */
	public function getHistory(int $limit, int $offset, ?array $modulesExclusion = null, ?array $modulesInclusion = null)
	{
		// not implemented due to difficulties in data aggregation between two databases
		throw new AppException("Not implemented for MongoDB storage");
	}

	/**
	 * {@inheritDoc}
	 */
	public function getUpdatesForWidget(string $moduleName, array $actions, array $dateRange, ?int $owner, ?int $historyOwner, \Vtiger_Paging_Model $pagingModel): array
	{
		// not implemented due to difficulties in data aggregation between two databases
		throw new AppException("Not implemented for MongoDB storage");
	}

	/**
	 * {@inheritDoc}
	 */
	public function getSummary(array $modules, array $actions, array $dateRange, ?int $owner, ?int $historyOwner): array
	{
		// not implemented due to difficulties in data aggregation between two databases
		throw new AppException("Not implemented for MongoDB storage");
	}

	/**
	 * Returns statuses query part for given entry type.
	 *
	 * @param string $type
	 *
	 * @return array|array[]
	 */
	private function getConditionByType(string $type): array
	{
		return match ($type) {
			'changes' => ['not in', 'status', [\ModTracker_Record_Model::DISPLAYED, \ModTracker_Record_Model::SHOW_HIDDEN_DATA]],
			'review' => ['status' => [\ModTracker_Record_Model::DISPLAYED, \ModTracker_Record_Model::SHOW_HIDDEN_DATA]],
			default => [],
		};
	}
}
