<?php

/**
 * Report Query model.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Leszek Koziatek <l.koziatek@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Report\Model;

/** Query class */
final class Query
{
	public function __construct(
		private int $id,
		private string $moduleName,
		/** @var array<Expression> */
		private array $expressions = [],
		/** @var array<Join> */
		private array $joins = [],
		private ?FilterGroup $filterGroup = null,
	) {}

	/** Get Query id */
	public function getId(): int
	{
		return $this->id;
	}

	/** Get related module name */
	public function getModuleName(): string
	{
		return $this->moduleName;
	}

	/** @return array<Expression> */
	public function getExpressions(): array
	{
		return $this->expressions;
	}

	/** Add Expression to Query */
	public function addExpression(Expression $expression): void
	{
		$id = $expression->getId();

		if (false === \array_key_exists($id, $this->expressions)) {
			$this->expressions[$id] = $expression;
		}
	}

	/** Add Join to Query */
	public function addJoin(Join $join): void
	{
		$id = $join->getId();

		if (false === \array_key_exists($id, $this->joins)) {
			$this->joins[$id] = $join;
		}
	}

	/** Get Filter Group for query */
	public function getFilterGroup(?int $id = null): ?FilterGroup
	{
		if (null === $this->filterGroup || null === $id || $this->filterGroup->getId() === $id) {
			return $this->filterGroup;
		}

		return $this->filterGroup->getChild($id);
	}

	public function getFilterGroupIds()
	{
		$ids = [];
		if (null !== $this->filterGroup) {
			$ids[] = $this->filterGroup->getId();
			$ids += $this->filterGroup->getChildrenIds(true);
		}

		return $ids;
	}

	/** Set Filter Group for Query */
	public function setFilterGroup(FilterGroup $filterGroup): void
	{
		$this->filterGroup = $filterGroup;
	}

	/** Get standard fields for Query */
	public function getStandardFields(): array
	{
		$result = [];

		foreach ($this->expressions as $expression) {
			if (null === $expression->getFunction() && false === $expression->isFormula()) {
				$result[$expression->getAlias()] = $expression->getField();
			}
		}

		return $result;
	}

	/** Get custom fields for Query */
	public function getCustomFields(): array
	{
		$result = [];

		foreach ($this->expressions as $expression) {
			if (null !== $expression->getFunction() && false === $expression->isFormula()) {
				$result[$expression->getAlias()] = [
					$expression->getField(),
					$expression->getFunctionField()
				];
			}
		}

		return $result;
	}

	/**
	 * @return array<Expression>
	 */
	public function getFormulaExpressions(): array
	{
		return array_filter(
			$this->expressions,
			static fn (Expression $expression): bool => $expression->isFormula(),
		);
	}

	/** Get Group expression in Query */
	public function getGroupExpressions(): array
	{
		return array_filter(
			$this->expressions,
			static fn (Expression $expression): bool => false === $expression->isAggregate() && false === $expression->isFormula(),
		);
	}

	/** Get expression for specific field */
	public function getExpressionByField(string $field): ?Expression
	{
		return array_values(array_filter(
			$this->expressions,
			static fn (Expression $expression): bool => $expression->getField() === $field,
		))[0] ?? null;
	}
}
