<?php

/**
 * QueryGeneratorFieldsProvider class.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Leszek Koziatek <l.koziatek@yetiforce.com>
 * @author Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Report\Provider;

use App\FieldCoordinatorTransformer\QueryGeneratorFieldTransformer;
use App\Language;
use App\Report\Builder\TemplateBuilder;
use App\Report\Model\Expression;
use App\Report\Model\Template;
use App\Report\Repository\FieldRepository;
use App\Report\Repository\TemplateRepository;

/** QueryGeneratorFieldsProvider class */
final class QueryGeneratorFieldsProvider implements FieldsProvider
{
	public function __construct(
		private readonly TemplateRepository $templateRepository,
		private readonly FieldRepository $fieldRepository,
		private readonly TemplateBuilder $builder,
	) {}

	/** {@inheritDoc} */
	public function provide(int $id, array $dataSample = []): array
	{
		$template = $this->templateRepository->find($id);
		$headers = $this->generateByTemplate($template);

		return \count($dataSample) > 0 ? $this->recheckHeadersSort($headers, $dataSample) : $headers;
	}

	/** {@inheritDoc} */
	public function provideByData(string $moduleName, array $expressions, array $dataSample = []): array
	{
		$query = $this->builder->buildQuery($moduleName);
		$template = $this->builder->build($query, $expressions);
		$headers = $this->generateByTemplate($template);

		return \count($dataSample) > 0 ? $this->recheckHeadersSort($headers, $dataSample) : $headers;
	}

	/** {@inheritDoc} */
	public function wrappedFields(int $id): array
	{
		// @TODO: implement method which will return array of columns where cells can be merged
		return [];
	}

	private function generateByTemplate(Template $template): array
	{
		$query = $template->getQuery();
		$baseModule = $query->getModuleName();

		$headers = [];
		foreach ($query->getExpressions() as $expression) {
			$alias = $expression->getAlias();
			if (true === $expression->isFormula()) {
				$headers[$alias] = $this->getFormulaLabel($expression, $baseModule);
				continue;
			}

			$fieldParams = QueryGeneratorFieldTransformer::extract($expression->getField());
			$fieldName = $fieldParams['fieldName'];
			$moduleName = null === $fieldParams['moduleName']
				? \Vtiger_Module_Model::getInstance($baseModule)->getName()
				: \Vtiger_Module_Model::getInstance($fieldParams['moduleName'])->getName();

			$field = ($moduleName === $baseModule) && QueryGeneratorFieldTransformer::isInventory($expression->getField())
				? $this->fieldRepository->getInventoryField($fieldName, $moduleName)
				: $this->fieldRepository->getFieldStraight($fieldName, $moduleName);

			if (null === $field) {
				$field = $this->getLabelForModuleKey($fieldName, $moduleName);
			}

			if (null === $field) {
				$headers[$alias] = $fieldName;
				continue;
			}

			$fieldLabel = $moduleName !== $baseModule
				? \sprintf(
					'[%s] %s',
					Language::translate($field['moduleName'], $moduleName),
					Language::translate($field['label'], $moduleName),
				)
				: Language::translate($field['label'], $moduleName);
			$headers[$alias] = null === ($functionType = $expression->getFunction())
				? $fieldLabel
				: \sprintf('%s(%s)', strtoupper($functionType->value), $fieldLabel);
		}

		return $headers;
	}

	/**
	 * Set headers orders as same as on first line of data.
	 *
	 * @param array $headers
	 * @param array $data
	 *
	 * @return array<string> - ordered and translated headers names
	 */
	private function recheckHeadersSort(array $headers, array $data): array
	{
		if (empty($data)) {
			return array_values($headers);
		}

		$orderedHeaders = [];
		foreach ($data as $alias => $value) {
			$orderedHeaders[] = $headers[$alias];
		}

		return $orderedHeaders;
	}

	/** Get translated label for formula field */
	private function getFormulaLabel(Expression $expression, string $baseModule): string
	{
		$translatedFields = [];
		foreach ($expression->getFormulaField() as $formulaField) {
			$fieldParams = QueryGeneratorFieldTransformer::extract($formulaField);
			$moduleName = $fieldParams['moduleName'] ?? $baseModule;
			$field = $this->getFieldArray(
				$fieldParams['fieldName'],
				$moduleName,
				$baseModule,
				QueryGeneratorFieldTransformer::isInventory($formulaField),
			);

			$translatedFields[] = Language::translate($field['label'], $moduleName);
		}

		return \sprintf(
			'%s(%s)',
			strtoupper($expression->getFunction()->value),
			implode(', ', $translatedFields),
		);
	}

	/**
	 * Get info array for field.
	 *
	 * @param string $fieldName
	 * @param string $moduleName
	 * @param string $baseModule
	 * @param bool   $isInventory
	 *
	 * @return array{label: string, moduleName: string}|null
	 */
	private function getFieldArray(string $fieldName, string $moduleName, string $baseModule, bool $isInventory): ?array
	{
		if (false === $isInventory) {
			$field = $this->fieldRepository->getFieldStraight($fieldName, $moduleName);
		} elseif ($baseModule === $moduleName) {
			$field = $this->fieldRepository->getInventoryField($fieldName, $moduleName);
		} else {
			$field = $this->fieldRepository->getFieldStraight($fieldName, $moduleName);
		}

		if (null !== $field) {
			return $field;
		}

		$module = \Vtiger_Module_Model::getInstance($moduleName);

		return false !== $module && ('id' === $fieldName || $module->basetableid === $fieldName)
			? ['label' => $moduleName, 'moduleName' => $moduleName]
			: $field;
	}

	/**
	 * Check if field is primary key for column.
	 *
	 * @param mixed $fieldName
	 * @param mixed $moduleName
	 *
	 * @return array{label: string, moduleName: string}|null
	 */
	private function getLabelForModuleKey($fieldName, $moduleName): ?array
	{
		$module = \Vtiger_Module_Model::getInstance($moduleName);

		if (false === $module) {
			return null;
		}

		return $fieldName === $module->basetableid || 'id' === $fieldName
			? ['label' => $moduleName, 'moduleName' => $moduleName]
			: null;
	}
}
