<?php
/**
 * Gross amount in words file.
 *
 * @package TextParser
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mateusz Slominski <m.slominski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\TextParser;

/**
 * GrossAmountInWords class.
 *
 * Returns inventory gross amount in words
 * Example output:
 * EN 399.75zł -> three hundred ninety-nine zł 75/100
 * PL 399.75zł -> trzysta dziewięćdziesiąt dziewięć zł 75/100
 */
class GrossAmountInWords extends Base
{
	/** @var string */
	public $name = 'LBL_GROSS_AMOUNT_IN_WORDS';

	/** @var mixed Parser type */
	public $type = 'pdf';

	/**
	 * Process.
	 *
	 * @return string
	 */
	public function process()
	{
		if (!$this->textParser->recordModel || !$this->textParser->recordModel->getModule()->isInventory()) {
			return '';
		}

		$inventory = \Vtiger_Inventory_Model::getInstance($this->textParser->moduleName);
		$currencySymbol = $inventory->getCurrency($this->textParser->recordModel)['currency_symbol'] ?? '';

		$userModel = \App\User::getCurrentUserModel();
		$numberOfDecimals = $userModel->getDetail('no_of_currency_decimals');

		$formatted = number_format(
			(float)$this->textParser->recordModel->get('sum_gross'),
			(int)$numberOfDecimals,
			'.',
			''
		);
		[$integer, $fraction] = array_pad(explode('.', $formatted), 2, 0);

		$decimalString = (new \NumberFormatter(\App\Language::getShortLanguageName(), \NumberFormatter::SPELLOUT))->format($integer);
		$fractionString = $numberOfDecimals > 0 ? sprintf(' %d/%d', $fraction, 10 ** $numberOfDecimals) : '';

		return $decimalString . ' ' . $currencySymbol . $fractionString;
	}
}
