<?php

/**
 * Save geographical coordinates handler file.
 *
 * @package Handler
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Tomasz Kur <t.kur@yetiforce.com>
 * @author Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 */

use App\Fields\MapCoordinates;
use App\Json;
use App\Map\Coordinates;

/**
 *  Save geographical coordinates handler class.
 */
class OpenStreetMap_OpenStreetMapHandler_Handler
{
	/** @var string[] Type of address. */
	const FIELDS_ADDRESS_TYPE = ['addresslevel', 'buildingnumber', 'localnumber', 'pobox'];

	/**
	 * EntityAfterSave handler function.
	 *
	 * @param App\EventHandler $eventHandler
	 *
	 * @return void
	 */
	public function entityAfterSave(App\EventHandler $eventHandler): void
	{
		$recordModel = $eventHandler->getRecordModel();
		self::updateByAddressFields($recordModel);
		self::updateByMapCoordinatesFields($recordModel);
	}

	/**
	 * Update coordinates from address fields.
	 *
	 * @param Vtiger_Record_Model $recordModel
	 *
	 * @return void
	 */
	public static function updateByAddressFields(Vtiger_Record_Model $recordModel): void
	{
		if (!\in_array($recordModel->getModuleName(), App\Config::module('OpenStreetMap', 'mapModules', []))) {
			return;
		}

		$typeAddressToUpdate = [];

		if (!$recordModel->isNew()) {
			$deltaFields = array_keys($recordModel->getPreviousValue());
			foreach ($deltaFields as &$deltaField) {
				if ($recordModel->getPreviousValue($deltaField) !== $recordModel->get($deltaField)) {
					foreach (self::FIELDS_ADDRESS_TYPE as $field) {
						if (str_contains($deltaField, $field)) {
							$typeAddressToUpdate[] = substr($deltaField, -1);
						}
					}
				}
			}
		}

		$typeAddressToUpdate = array_unique($typeAddressToUpdate);

		foreach (Coordinates::TYPE_ADDRESS as $typeAddress) {
			$address = Coordinates::getAddressParams($recordModel, $typeAddress);
			if (
				($recordModel->isNew() || \in_array($typeAddress, $typeAddressToUpdate))
				&& !empty(array_filter($address))
			) {
				Coordinates::saveUpdater(
					$recordModel->getId(),
					$typeAddress,
					$address
				);
			} elseif (
				!$recordModel->isNew()
				&& empty(array_filter($address))
				&& \in_array($typeAddress, $typeAddressToUpdate)
			) {
				Coordinates::deleteAll(
					$recordModel->getId(),
					$typeAddress,
				);
			}
		}
	}

	/**
	 * Update coordinates from fields of type `mapCoordinates`.
	 *
	 * @param Vtiger_Record_Model $recordModel
	 *
	 * @throws \App\Exceptions\AppException
	 *
	 * @return void
	 */
	public static function updateByMapCoordinatesFields(Vtiger_Record_Model $recordModel): void
	{
		$isNew = $recordModel->isNew();
		$fields = $recordModel->getModule()->getFieldsByType('mapCoordinates');

		foreach ($fields as $fieldName => $fieldModel) {
			if (
				($isNew && !$recordModel->isEmpty($fieldName))
				|| (!$isNew && false !== $recordModel->getPreviousValue($fieldName))
			) {
				$coordinates = Json::decode($recordModel->get($fieldName));

				if ('decimal' !== $coordinates['type']) {
					$value = MapCoordinates::convert($coordinates['type'], 'decimal', $coordinates['value']);
				} else {
					$value = $coordinates['value'];
				}

				Coordinates::saveCoordinates($recordModel->getId(), $fieldName, $value);
			}
		}
	}
}
