<?php

namespace MLocati\IDNA;

class IdnaMap
{
    /**
     * Valid, never excluded.
     *
     * @var int
     */
    const EXCLUDE_NO = 1;

    /**
     * Range is excluded by IDNA2008 from all domain names for all versions of Unicode.
     *
     * @var int
     */
    const EXCLUDE_ALWAYS = 2;

    /**
     * Range is excluded by IDNA2008 for the current version of Unicode.
     *
     * @var int
     */
    const EXCLUDE_CURRENT = 3;

    protected static $deviations = array(
        223 => array(115, 115),
        962 => array(963),
        8204 => array(),
        8205 => array(),
    );

    /**
     * Get the IDNA2003 deviation from IDNA2008 for a specific code point.
     *
     * @param int $codepoint
     *
     * @return int[]|null
     */
    public static function getDeviation($codepoint)
    {
        return isset(static::$deviations[$codepoint]) ? static::$deviations[$codepoint] : null;
    }

    protected static $ignoredSingle = array(
        173,
        847,
        6155,
        6156,
        6157,
        6159,
        8203,
        8288,
        8292,
        65024,
        65025,
        65026,
        65027,
        65028,
        65029,
        65030,
        65031,
        65032,
        65033,
        65034,
        65035,
        65036,
        65037,
        65038,
        65039,
        65279,
        113824,
        113825,
        113826,
        113827,
    );

    /**
     *Check if a codepoint is ignored.
     *
     *@param int $codepoint
     *
     *@return bool
     */
    public static function isIgnored($codepoint)
    {
        $result = false;
        if (
            in_array($codepoint, static::$ignoredSingle, true)
            || ($codepoint >= 917760 && $codepoint <= 917999)
        ) {
            $result = true;
        }

        return $result;
    }

    protected static $mapped = array(
        65 => array(97),
        66 => array(98),
        67 => array(99),
        68 => array(100),
        69 => array(101),
        70 => array(102),
        71 => array(103),
        72 => array(104),
        73 => array(105),
        74 => array(106),
        75 => array(107),
        76 => array(108),
        77 => array(109),
        78 => array(110),
        79 => array(111),
        80 => array(112),
        81 => array(113),
        82 => array(114),
        83 => array(115),
        84 => array(116),
        85 => array(117),
        86 => array(118),
        87 => array(119),
        88 => array(120),
        89 => array(121),
        90 => array(122),
        170 => array(97),
        178 => array(50),
        179 => array(51),
        181 => array(956),
        185 => array(49),
        186 => array(111),
        188 => array(49, 8260, 52),
        189 => array(49, 8260, 50),
        190 => array(51, 8260, 52),
        192 => array(224),
        193 => array(225),
        194 => array(226),
        195 => array(227),
        196 => array(228),
        197 => array(229),
        198 => array(230),
        199 => array(231),
        200 => array(232),
        201 => array(233),
        202 => array(234),
        203 => array(235),
        204 => array(236),
        205 => array(237),
        206 => array(238),
        207 => array(239),
        208 => array(240),
        209 => array(241),
        210 => array(242),
        211 => array(243),
        212 => array(244),
        213 => array(245),
        214 => array(246),
        216 => array(248),
        217 => array(249),
        218 => array(250),
        219 => array(251),
        220 => array(252),
        221 => array(253),
        222 => array(254),
        256 => array(257),
        258 => array(259),
        260 => array(261),
        262 => array(263),
        264 => array(265),
        266 => array(267),
        268 => array(269),
        270 => array(271),
        272 => array(273),
        274 => array(275),
        276 => array(277),
        278 => array(279),
        280 => array(281),
        282 => array(283),
        284 => array(285),
        286 => array(287),
        288 => array(289),
        290 => array(291),
        292 => array(293),
        294 => array(295),
        296 => array(297),
        298 => array(299),
        300 => array(301),
        302 => array(303),
        304 => array(105, 775),
        306 => array(105, 106),
        307 => array(105, 106),
        308 => array(309),
        310 => array(311),
        313 => array(314),
        315 => array(316),
        317 => array(318),
        319 => array(108, 183),
        320 => array(108, 183),
        321 => array(322),
        323 => array(324),
        325 => array(326),
        327 => array(328),
        329 => array(700, 110),
        330 => array(331),
        332 => array(333),
        334 => array(335),
        336 => array(337),
        338 => array(339),
        340 => array(341),
        342 => array(343),
        344 => array(345),
        346 => array(347),
        348 => array(349),
        350 => array(351),
        352 => array(353),
        354 => array(355),
        356 => array(357),
        358 => array(359),
        360 => array(361),
        362 => array(363),
        364 => array(365),
        366 => array(367),
        368 => array(369),
        370 => array(371),
        372 => array(373),
        374 => array(375),
        376 => array(255),
        377 => array(378),
        379 => array(380),
        381 => array(382),
        383 => array(115),
        385 => array(595),
        386 => array(387),
        388 => array(389),
        390 => array(596),
        391 => array(392),
        393 => array(598),
        394 => array(599),
        395 => array(396),
        398 => array(477),
        399 => array(601),
        400 => array(603),
        401 => array(402),
        403 => array(608),
        404 => array(611),
        406 => array(617),
        407 => array(616),
        408 => array(409),
        412 => array(623),
        413 => array(626),
        415 => array(629),
        416 => array(417),
        418 => array(419),
        420 => array(421),
        422 => array(640),
        423 => array(424),
        425 => array(643),
        428 => array(429),
        430 => array(648),
        431 => array(432),
        433 => array(650),
        434 => array(651),
        435 => array(436),
        437 => array(438),
        439 => array(658),
        440 => array(441),
        444 => array(445),
        452 => array(100, 382),
        453 => array(100, 382),
        454 => array(100, 382),
        455 => array(108, 106),
        456 => array(108, 106),
        457 => array(108, 106),
        458 => array(110, 106),
        459 => array(110, 106),
        460 => array(110, 106),
        461 => array(462),
        463 => array(464),
        465 => array(466),
        467 => array(468),
        469 => array(470),
        471 => array(472),
        473 => array(474),
        475 => array(476),
        478 => array(479),
        480 => array(481),
        482 => array(483),
        484 => array(485),
        486 => array(487),
        488 => array(489),
        490 => array(491),
        492 => array(493),
        494 => array(495),
        497 => array(100, 122),
        498 => array(100, 122),
        499 => array(100, 122),
        500 => array(501),
        502 => array(405),
        503 => array(447),
        504 => array(505),
        506 => array(507),
        508 => array(509),
        510 => array(511),
        512 => array(513),
        514 => array(515),
        516 => array(517),
        518 => array(519),
        520 => array(521),
        522 => array(523),
        524 => array(525),
        526 => array(527),
        528 => array(529),
        530 => array(531),
        532 => array(533),
        534 => array(535),
        536 => array(537),
        538 => array(539),
        540 => array(541),
        542 => array(543),
        544 => array(414),
        546 => array(547),
        548 => array(549),
        550 => array(551),
        552 => array(553),
        554 => array(555),
        556 => array(557),
        558 => array(559),
        560 => array(561),
        562 => array(563),
        570 => array(11365),
        571 => array(572),
        573 => array(410),
        574 => array(11366),
        577 => array(578),
        579 => array(384),
        580 => array(649),
        581 => array(652),
        582 => array(583),
        584 => array(585),
        586 => array(587),
        588 => array(589),
        590 => array(591),
        688 => array(104),
        689 => array(614),
        690 => array(106),
        691 => array(114),
        692 => array(633),
        693 => array(635),
        694 => array(641),
        695 => array(119),
        696 => array(121),
        736 => array(611),
        737 => array(108),
        738 => array(115),
        739 => array(120),
        740 => array(661),
        832 => array(768),
        833 => array(769),
        835 => array(787),
        836 => array(776, 769),
        837 => array(953),
        880 => array(881),
        882 => array(883),
        884 => array(697),
        886 => array(887),
        895 => array(1011),
        902 => array(940),
        903 => array(183),
        904 => array(941),
        905 => array(942),
        906 => array(943),
        908 => array(972),
        910 => array(973),
        911 => array(974),
        913 => array(945),
        914 => array(946),
        915 => array(947),
        916 => array(948),
        917 => array(949),
        918 => array(950),
        919 => array(951),
        920 => array(952),
        921 => array(953),
        922 => array(954),
        923 => array(955),
        924 => array(956),
        925 => array(957),
        926 => array(958),
        927 => array(959),
        928 => array(960),
        929 => array(961),
        931 => array(963),
        932 => array(964),
        933 => array(965),
        934 => array(966),
        935 => array(967),
        936 => array(968),
        937 => array(969),
        938 => array(970),
        939 => array(971),
        975 => array(983),
        976 => array(946),
        977 => array(952),
        978 => array(965),
        979 => array(973),
        980 => array(971),
        981 => array(966),
        982 => array(960),
        984 => array(985),
        986 => array(987),
        988 => array(989),
        990 => array(991),
        992 => array(993),
        994 => array(995),
        996 => array(997),
        998 => array(999),
        1000 => array(1001),
        1002 => array(1003),
        1004 => array(1005),
        1006 => array(1007),
        1008 => array(954),
        1009 => array(961),
        1010 => array(963),
        1012 => array(952),
        1013 => array(949),
        1015 => array(1016),
        1017 => array(963),
        1018 => array(1019),
        1021 => array(891),
        1022 => array(892),
        1023 => array(893),
        1024 => array(1104),
        1025 => array(1105),
        1026 => array(1106),
        1027 => array(1107),
        1028 => array(1108),
        1029 => array(1109),
        1030 => array(1110),
        1031 => array(1111),
        1032 => array(1112),
        1033 => array(1113),
        1034 => array(1114),
        1035 => array(1115),
        1036 => array(1116),
        1037 => array(1117),
        1038 => array(1118),
        1039 => array(1119),
        1040 => array(1072),
        1041 => array(1073),
        1042 => array(1074),
        1043 => array(1075),
        1044 => array(1076),
        1045 => array(1077),
        1046 => array(1078),
        1047 => array(1079),
        1048 => array(1080),
        1049 => array(1081),
        1050 => array(1082),
        1051 => array(1083),
        1052 => array(1084),
        1053 => array(1085),
        1054 => array(1086),
        1055 => array(1087),
        1056 => array(1088),
        1057 => array(1089),
        1058 => array(1090),
        1059 => array(1091),
        1060 => array(1092),
        1061 => array(1093),
        1062 => array(1094),
        1063 => array(1095),
        1064 => array(1096),
        1065 => array(1097),
        1066 => array(1098),
        1067 => array(1099),
        1068 => array(1100),
        1069 => array(1101),
        1070 => array(1102),
        1071 => array(1103),
        1120 => array(1121),
        1122 => array(1123),
        1124 => array(1125),
        1126 => array(1127),
        1128 => array(1129),
        1130 => array(1131),
        1132 => array(1133),
        1134 => array(1135),
        1136 => array(1137),
        1138 => array(1139),
        1140 => array(1141),
        1142 => array(1143),
        1144 => array(1145),
        1146 => array(1147),
        1148 => array(1149),
        1150 => array(1151),
        1152 => array(1153),
        1162 => array(1163),
        1164 => array(1165),
        1166 => array(1167),
        1168 => array(1169),
        1170 => array(1171),
        1172 => array(1173),
        1174 => array(1175),
        1176 => array(1177),
        1178 => array(1179),
        1180 => array(1181),
        1182 => array(1183),
        1184 => array(1185),
        1186 => array(1187),
        1188 => array(1189),
        1190 => array(1191),
        1192 => array(1193),
        1194 => array(1195),
        1196 => array(1197),
        1198 => array(1199),
        1200 => array(1201),
        1202 => array(1203),
        1204 => array(1205),
        1206 => array(1207),
        1208 => array(1209),
        1210 => array(1211),
        1212 => array(1213),
        1214 => array(1215),
        1217 => array(1218),
        1219 => array(1220),
        1221 => array(1222),
        1223 => array(1224),
        1225 => array(1226),
        1227 => array(1228),
        1229 => array(1230),
        1232 => array(1233),
        1234 => array(1235),
        1236 => array(1237),
        1238 => array(1239),
        1240 => array(1241),
        1242 => array(1243),
        1244 => array(1245),
        1246 => array(1247),
        1248 => array(1249),
        1250 => array(1251),
        1252 => array(1253),
        1254 => array(1255),
        1256 => array(1257),
        1258 => array(1259),
        1260 => array(1261),
        1262 => array(1263),
        1264 => array(1265),
        1266 => array(1267),
        1268 => array(1269),
        1270 => array(1271),
        1272 => array(1273),
        1274 => array(1275),
        1276 => array(1277),
        1278 => array(1279),
        1280 => array(1281),
        1282 => array(1283),
        1284 => array(1285),
        1286 => array(1287),
        1288 => array(1289),
        1290 => array(1291),
        1292 => array(1293),
        1294 => array(1295),
        1296 => array(1297),
        1298 => array(1299),
        1300 => array(1301),
        1302 => array(1303),
        1304 => array(1305),
        1306 => array(1307),
        1308 => array(1309),
        1310 => array(1311),
        1312 => array(1313),
        1314 => array(1315),
        1316 => array(1317),
        1318 => array(1319),
        1320 => array(1321),
        1322 => array(1323),
        1324 => array(1325),
        1326 => array(1327),
        1329 => array(1377),
        1330 => array(1378),
        1331 => array(1379),
        1332 => array(1380),
        1333 => array(1381),
        1334 => array(1382),
        1335 => array(1383),
        1336 => array(1384),
        1337 => array(1385),
        1338 => array(1386),
        1339 => array(1387),
        1340 => array(1388),
        1341 => array(1389),
        1342 => array(1390),
        1343 => array(1391),
        1344 => array(1392),
        1345 => array(1393),
        1346 => array(1394),
        1347 => array(1395),
        1348 => array(1396),
        1349 => array(1397),
        1350 => array(1398),
        1351 => array(1399),
        1352 => array(1400),
        1353 => array(1401),
        1354 => array(1402),
        1355 => array(1403),
        1356 => array(1404),
        1357 => array(1405),
        1358 => array(1406),
        1359 => array(1407),
        1360 => array(1408),
        1361 => array(1409),
        1362 => array(1410),
        1363 => array(1411),
        1364 => array(1412),
        1365 => array(1413),
        1366 => array(1414),
        1415 => array(1381, 1410),
        1653 => array(1575, 1652),
        1654 => array(1608, 1652),
        1655 => array(1735, 1652),
        1656 => array(1610, 1652),
        2392 => array(2325, 2364),
        2393 => array(2326, 2364),
        2394 => array(2327, 2364),
        2395 => array(2332, 2364),
        2396 => array(2337, 2364),
        2397 => array(2338, 2364),
        2398 => array(2347, 2364),
        2399 => array(2351, 2364),
        2524 => array(2465, 2492),
        2525 => array(2466, 2492),
        2527 => array(2479, 2492),
        2611 => array(2610, 2620),
        2614 => array(2616, 2620),
        2649 => array(2582, 2620),
        2650 => array(2583, 2620),
        2651 => array(2588, 2620),
        2654 => array(2603, 2620),
        2908 => array(2849, 2876),
        2909 => array(2850, 2876),
        3635 => array(3661, 3634),
        3763 => array(3789, 3762),
        3804 => array(3755, 3737),
        3805 => array(3755, 3745),
        3852 => array(3851),
        3907 => array(3906, 4023),
        3917 => array(3916, 4023),
        3922 => array(3921, 4023),
        3927 => array(3926, 4023),
        3932 => array(3931, 4023),
        3945 => array(3904, 4021),
        3955 => array(3953, 3954),
        3957 => array(3953, 3956),
        3958 => array(4018, 3968),
        3959 => array(4018, 3953, 3968),
        3960 => array(4019, 3968),
        3961 => array(4019, 3953, 3968),
        3969 => array(3953, 3968),
        3987 => array(3986, 4023),
        3997 => array(3996, 4023),
        4002 => array(4001, 4023),
        4007 => array(4006, 4023),
        4012 => array(4011, 4023),
        4025 => array(3984, 4021),
        4295 => array(11559),
        4301 => array(11565),
        4348 => array(4316),
        5112 => array(5104),
        5113 => array(5105),
        5114 => array(5106),
        5115 => array(5107),
        5116 => array(5108),
        5117 => array(5109),
        7296 => array(1074),
        7297 => array(1076),
        7298 => array(1086),
        7299 => array(1089),
        7300 => array(1090),
        7301 => array(1090),
        7302 => array(1098),
        7303 => array(1123),
        7304 => array(42571),
        7312 => array(4304),
        7313 => array(4305),
        7314 => array(4306),
        7315 => array(4307),
        7316 => array(4308),
        7317 => array(4309),
        7318 => array(4310),
        7319 => array(4311),
        7320 => array(4312),
        7321 => array(4313),
        7322 => array(4314),
        7323 => array(4315),
        7324 => array(4316),
        7325 => array(4317),
        7326 => array(4318),
        7327 => array(4319),
        7328 => array(4320),
        7329 => array(4321),
        7330 => array(4322),
        7331 => array(4323),
        7332 => array(4324),
        7333 => array(4325),
        7334 => array(4326),
        7335 => array(4327),
        7336 => array(4328),
        7337 => array(4329),
        7338 => array(4330),
        7339 => array(4331),
        7340 => array(4332),
        7341 => array(4333),
        7342 => array(4334),
        7343 => array(4335),
        7344 => array(4336),
        7345 => array(4337),
        7346 => array(4338),
        7347 => array(4339),
        7348 => array(4340),
        7349 => array(4341),
        7350 => array(4342),
        7351 => array(4343),
        7352 => array(4344),
        7353 => array(4345),
        7354 => array(4346),
        7357 => array(4349),
        7358 => array(4350),
        7359 => array(4351),
        7468 => array(97),
        7469 => array(230),
        7470 => array(98),
        7472 => array(100),
        7473 => array(101),
        7474 => array(477),
        7475 => array(103),
        7476 => array(104),
        7477 => array(105),
        7478 => array(106),
        7479 => array(107),
        7480 => array(108),
        7481 => array(109),
        7482 => array(110),
        7484 => array(111),
        7485 => array(547),
        7486 => array(112),
        7487 => array(114),
        7488 => array(116),
        7489 => array(117),
        7490 => array(119),
        7491 => array(97),
        7492 => array(592),
        7493 => array(593),
        7494 => array(7426),
        7495 => array(98),
        7496 => array(100),
        7497 => array(101),
        7498 => array(601),
        7499 => array(603),
        7500 => array(604),
        7501 => array(103),
        7503 => array(107),
        7504 => array(109),
        7505 => array(331),
        7506 => array(111),
        7507 => array(596),
        7508 => array(7446),
        7509 => array(7447),
        7510 => array(112),
        7511 => array(116),
        7512 => array(117),
        7513 => array(7453),
        7514 => array(623),
        7515 => array(118),
        7516 => array(7461),
        7517 => array(946),
        7518 => array(947),
        7519 => array(948),
        7520 => array(966),
        7521 => array(967),
        7522 => array(105),
        7523 => array(114),
        7524 => array(117),
        7525 => array(118),
        7526 => array(946),
        7527 => array(947),
        7528 => array(961),
        7529 => array(966),
        7530 => array(967),
        7544 => array(1085),
        7579 => array(594),
        7580 => array(99),
        7581 => array(597),
        7582 => array(240),
        7583 => array(604),
        7584 => array(102),
        7585 => array(607),
        7586 => array(609),
        7587 => array(613),
        7588 => array(616),
        7589 => array(617),
        7590 => array(618),
        7591 => array(7547),
        7592 => array(669),
        7593 => array(621),
        7594 => array(7557),
        7595 => array(671),
        7596 => array(625),
        7597 => array(624),
        7598 => array(626),
        7599 => array(627),
        7600 => array(628),
        7601 => array(629),
        7602 => array(632),
        7603 => array(642),
        7604 => array(643),
        7605 => array(427),
        7606 => array(649),
        7607 => array(650),
        7608 => array(7452),
        7609 => array(651),
        7610 => array(652),
        7611 => array(122),
        7612 => array(656),
        7613 => array(657),
        7614 => array(658),
        7615 => array(952),
        7680 => array(7681),
        7682 => array(7683),
        7684 => array(7685),
        7686 => array(7687),
        7688 => array(7689),
        7690 => array(7691),
        7692 => array(7693),
        7694 => array(7695),
        7696 => array(7697),
        7698 => array(7699),
        7700 => array(7701),
        7702 => array(7703),
        7704 => array(7705),
        7706 => array(7707),
        7708 => array(7709),
        7710 => array(7711),
        7712 => array(7713),
        7714 => array(7715),
        7716 => array(7717),
        7718 => array(7719),
        7720 => array(7721),
        7722 => array(7723),
        7724 => array(7725),
        7726 => array(7727),
        7728 => array(7729),
        7730 => array(7731),
        7732 => array(7733),
        7734 => array(7735),
        7736 => array(7737),
        7738 => array(7739),
        7740 => array(7741),
        7742 => array(7743),
        7744 => array(7745),
        7746 => array(7747),
        7748 => array(7749),
        7750 => array(7751),
        7752 => array(7753),
        7754 => array(7755),
        7756 => array(7757),
        7758 => array(7759),
        7760 => array(7761),
        7762 => array(7763),
        7764 => array(7765),
        7766 => array(7767),
        7768 => array(7769),
        7770 => array(7771),
        7772 => array(7773),
        7774 => array(7775),
        7776 => array(7777),
        7778 => array(7779),
        7780 => array(7781),
        7782 => array(7783),
        7784 => array(7785),
        7786 => array(7787),
        7788 => array(7789),
        7790 => array(7791),
        7792 => array(7793),
        7794 => array(7795),
        7796 => array(7797),
        7798 => array(7799),
        7800 => array(7801),
        7802 => array(7803),
        7804 => array(7805),
        7806 => array(7807),
        7808 => array(7809),
        7810 => array(7811),
        7812 => array(7813),
        7814 => array(7815),
        7816 => array(7817),
        7818 => array(7819),
        7820 => array(7821),
        7822 => array(7823),
        7824 => array(7825),
        7826 => array(7827),
        7828 => array(7829),
        7834 => array(97, 702),
        7835 => array(7777),
        7838 => array(115, 115),
        7840 => array(7841),
        7842 => array(7843),
        7844 => array(7845),
        7846 => array(7847),
        7848 => array(7849),
        7850 => array(7851),
        7852 => array(7853),
        7854 => array(7855),
        7856 => array(7857),
        7858 => array(7859),
        7860 => array(7861),
        7862 => array(7863),
        7864 => array(7865),
        7866 => array(7867),
        7868 => array(7869),
        7870 => array(7871),
        7872 => array(7873),
        7874 => array(7875),
        7876 => array(7877),
        7878 => array(7879),
        7880 => array(7881),
        7882 => array(7883),
        7884 => array(7885),
        7886 => array(7887),
        7888 => array(7889),
        7890 => array(7891),
        7892 => array(7893),
        7894 => array(7895),
        7896 => array(7897),
        7898 => array(7899),
        7900 => array(7901),
        7902 => array(7903),
        7904 => array(7905),
        7906 => array(7907),
        7908 => array(7909),
        7910 => array(7911),
        7912 => array(7913),
        7914 => array(7915),
        7916 => array(7917),
        7918 => array(7919),
        7920 => array(7921),
        7922 => array(7923),
        7924 => array(7925),
        7926 => array(7927),
        7928 => array(7929),
        7930 => array(7931),
        7932 => array(7933),
        7934 => array(7935),
        7944 => array(7936),
        7945 => array(7937),
        7946 => array(7938),
        7947 => array(7939),
        7948 => array(7940),
        7949 => array(7941),
        7950 => array(7942),
        7951 => array(7943),
        7960 => array(7952),
        7961 => array(7953),
        7962 => array(7954),
        7963 => array(7955),
        7964 => array(7956),
        7965 => array(7957),
        7976 => array(7968),
        7977 => array(7969),
        7978 => array(7970),
        7979 => array(7971),
        7980 => array(7972),
        7981 => array(7973),
        7982 => array(7974),
        7983 => array(7975),
        7992 => array(7984),
        7993 => array(7985),
        7994 => array(7986),
        7995 => array(7987),
        7996 => array(7988),
        7997 => array(7989),
        7998 => array(7990),
        7999 => array(7991),
        8008 => array(8000),
        8009 => array(8001),
        8010 => array(8002),
        8011 => array(8003),
        8012 => array(8004),
        8013 => array(8005),
        8025 => array(8017),
        8027 => array(8019),
        8029 => array(8021),
        8031 => array(8023),
        8040 => array(8032),
        8041 => array(8033),
        8042 => array(8034),
        8043 => array(8035),
        8044 => array(8036),
        8045 => array(8037),
        8046 => array(8038),
        8047 => array(8039),
        8049 => array(940),
        8051 => array(941),
        8053 => array(942),
        8055 => array(943),
        8057 => array(972),
        8059 => array(973),
        8061 => array(974),
        8064 => array(7936, 953),
        8065 => array(7937, 953),
        8066 => array(7938, 953),
        8067 => array(7939, 953),
        8068 => array(7940, 953),
        8069 => array(7941, 953),
        8070 => array(7942, 953),
        8071 => array(7943, 953),
        8072 => array(7936, 953),
        8073 => array(7937, 953),
        8074 => array(7938, 953),
        8075 => array(7939, 953),
        8076 => array(7940, 953),
        8077 => array(7941, 953),
        8078 => array(7942, 953),
        8079 => array(7943, 953),
        8080 => array(7968, 953),
        8081 => array(7969, 953),
        8082 => array(7970, 953),
        8083 => array(7971, 953),
        8084 => array(7972, 953),
        8085 => array(7973, 953),
        8086 => array(7974, 953),
        8087 => array(7975, 953),
        8088 => array(7968, 953),
        8089 => array(7969, 953),
        8090 => array(7970, 953),
        8091 => array(7971, 953),
        8092 => array(7972, 953),
        8093 => array(7973, 953),
        8094 => array(7974, 953),
        8095 => array(7975, 953),
        8096 => array(8032, 953),
        8097 => array(8033, 953),
        8098 => array(8034, 953),
        8099 => array(8035, 953),
        8100 => array(8036, 953),
        8101 => array(8037, 953),
        8102 => array(8038, 953),
        8103 => array(8039, 953),
        8104 => array(8032, 953),
        8105 => array(8033, 953),
        8106 => array(8034, 953),
        8107 => array(8035, 953),
        8108 => array(8036, 953),
        8109 => array(8037, 953),
        8110 => array(8038, 953),
        8111 => array(8039, 953),
        8114 => array(8048, 953),
        8115 => array(945, 953),
        8116 => array(940, 953),
        8119 => array(8118, 953),
        8120 => array(8112),
        8121 => array(8113),
        8122 => array(8048),
        8123 => array(940),
        8124 => array(945, 953),
        8126 => array(953),
        8130 => array(8052, 953),
        8131 => array(951, 953),
        8132 => array(942, 953),
        8135 => array(8134, 953),
        8136 => array(8050),
        8137 => array(941),
        8138 => array(8052),
        8139 => array(942),
        8140 => array(951, 953),
        8147 => array(912),
        8152 => array(8144),
        8153 => array(8145),
        8154 => array(8054),
        8155 => array(943),
        8163 => array(944),
        8168 => array(8160),
        8169 => array(8161),
        8170 => array(8058),
        8171 => array(973),
        8172 => array(8165),
        8178 => array(8060, 953),
        8179 => array(969, 953),
        8180 => array(974, 953),
        8183 => array(8182, 953),
        8184 => array(8056),
        8185 => array(972),
        8186 => array(8060),
        8187 => array(974),
        8188 => array(969, 953),
        8209 => array(8208),
        8243 => array(8242, 8242),
        8244 => array(8242, 8242, 8242),
        8246 => array(8245, 8245),
        8247 => array(8245, 8245, 8245),
        8279 => array(8242, 8242, 8242, 8242),
        8304 => array(48),
        8305 => array(105),
        8308 => array(52),
        8309 => array(53),
        8310 => array(54),
        8311 => array(55),
        8312 => array(56),
        8313 => array(57),
        8315 => array(8722),
        8319 => array(110),
        8320 => array(48),
        8321 => array(49),
        8322 => array(50),
        8323 => array(51),
        8324 => array(52),
        8325 => array(53),
        8326 => array(54),
        8327 => array(55),
        8328 => array(56),
        8329 => array(57),
        8331 => array(8722),
        8336 => array(97),
        8337 => array(101),
        8338 => array(111),
        8339 => array(120),
        8340 => array(601),
        8341 => array(104),
        8342 => array(107),
        8343 => array(108),
        8344 => array(109),
        8345 => array(110),
        8346 => array(112),
        8347 => array(115),
        8348 => array(116),
        8360 => array(114, 115),
        8450 => array(99),
        8451 => array(176, 99),
        8455 => array(603),
        8457 => array(176, 102),
        8458 => array(103),
        8459 => array(104),
        8460 => array(104),
        8461 => array(104),
        8462 => array(104),
        8463 => array(295),
        8464 => array(105),
        8465 => array(105),
        8466 => array(108),
        8467 => array(108),
        8469 => array(110),
        8470 => array(110, 111),
        8473 => array(112),
        8474 => array(113),
        8475 => array(114),
        8476 => array(114),
        8477 => array(114),
        8480 => array(115, 109),
        8481 => array(116, 101, 108),
        8482 => array(116, 109),
        8484 => array(122),
        8486 => array(969),
        8488 => array(122),
        8490 => array(107),
        8491 => array(229),
        8492 => array(98),
        8493 => array(99),
        8495 => array(101),
        8496 => array(101),
        8497 => array(102),
        8499 => array(109),
        8500 => array(111),
        8501 => array(1488),
        8502 => array(1489),
        8503 => array(1490),
        8504 => array(1491),
        8505 => array(105),
        8507 => array(102, 97, 120),
        8508 => array(960),
        8509 => array(947),
        8510 => array(947),
        8511 => array(960),
        8512 => array(8721),
        8517 => array(100),
        8518 => array(100),
        8519 => array(101),
        8520 => array(105),
        8521 => array(106),
        8528 => array(49, 8260, 55),
        8529 => array(49, 8260, 57),
        8530 => array(49, 8260, 49, 48),
        8531 => array(49, 8260, 51),
        8532 => array(50, 8260, 51),
        8533 => array(49, 8260, 53),
        8534 => array(50, 8260, 53),
        8535 => array(51, 8260, 53),
        8536 => array(52, 8260, 53),
        8537 => array(49, 8260, 54),
        8538 => array(53, 8260, 54),
        8539 => array(49, 8260, 56),
        8540 => array(51, 8260, 56),
        8541 => array(53, 8260, 56),
        8542 => array(55, 8260, 56),
        8543 => array(49, 8260),
        8544 => array(105),
        8545 => array(105, 105),
        8546 => array(105, 105, 105),
        8547 => array(105, 118),
        8548 => array(118),
        8549 => array(118, 105),
        8550 => array(118, 105, 105),
        8551 => array(118, 105, 105, 105),
        8552 => array(105, 120),
        8553 => array(120),
        8554 => array(120, 105),
        8555 => array(120, 105, 105),
        8556 => array(108),
        8557 => array(99),
        8558 => array(100),
        8559 => array(109),
        8560 => array(105),
        8561 => array(105, 105),
        8562 => array(105, 105, 105),
        8563 => array(105, 118),
        8564 => array(118),
        8565 => array(118, 105),
        8566 => array(118, 105, 105),
        8567 => array(118, 105, 105, 105),
        8568 => array(105, 120),
        8569 => array(120),
        8570 => array(120, 105),
        8571 => array(120, 105, 105),
        8572 => array(108),
        8573 => array(99),
        8574 => array(100),
        8575 => array(109),
        8585 => array(48, 8260, 51),
        8748 => array(8747, 8747),
        8749 => array(8747, 8747, 8747),
        8751 => array(8750, 8750),
        8752 => array(8750, 8750, 8750),
        9001 => array(12296),
        9002 => array(12297),
        9312 => array(49),
        9313 => array(50),
        9314 => array(51),
        9315 => array(52),
        9316 => array(53),
        9317 => array(54),
        9318 => array(55),
        9319 => array(56),
        9320 => array(57),
        9321 => array(49, 48),
        9322 => array(49, 49),
        9323 => array(49, 50),
        9324 => array(49, 51),
        9325 => array(49, 52),
        9326 => array(49, 53),
        9327 => array(49, 54),
        9328 => array(49, 55),
        9329 => array(49, 56),
        9330 => array(49, 57),
        9331 => array(50, 48),
        9398 => array(97),
        9399 => array(98),
        9400 => array(99),
        9401 => array(100),
        9402 => array(101),
        9403 => array(102),
        9404 => array(103),
        9405 => array(104),
        9406 => array(105),
        9407 => array(106),
        9408 => array(107),
        9409 => array(108),
        9410 => array(109),
        9411 => array(110),
        9412 => array(111),
        9413 => array(112),
        9414 => array(113),
        9415 => array(114),
        9416 => array(115),
        9417 => array(116),
        9418 => array(117),
        9419 => array(118),
        9420 => array(119),
        9421 => array(120),
        9422 => array(121),
        9423 => array(122),
        9424 => array(97),
        9425 => array(98),
        9426 => array(99),
        9427 => array(100),
        9428 => array(101),
        9429 => array(102),
        9430 => array(103),
        9431 => array(104),
        9432 => array(105),
        9433 => array(106),
        9434 => array(107),
        9435 => array(108),
        9436 => array(109),
        9437 => array(110),
        9438 => array(111),
        9439 => array(112),
        9440 => array(113),
        9441 => array(114),
        9442 => array(115),
        9443 => array(116),
        9444 => array(117),
        9445 => array(118),
        9446 => array(119),
        9447 => array(120),
        9448 => array(121),
        9449 => array(122),
        9450 => array(48),
        10764 => array(8747, 8747, 8747, 8747),
        10972 => array(10973, 824),
        11264 => array(11312),
        11265 => array(11313),
        11266 => array(11314),
        11267 => array(11315),
        11268 => array(11316),
        11269 => array(11317),
        11270 => array(11318),
        11271 => array(11319),
        11272 => array(11320),
        11273 => array(11321),
        11274 => array(11322),
        11275 => array(11323),
        11276 => array(11324),
        11277 => array(11325),
        11278 => array(11326),
        11279 => array(11327),
        11280 => array(11328),
        11281 => array(11329),
        11282 => array(11330),
        11283 => array(11331),
        11284 => array(11332),
        11285 => array(11333),
        11286 => array(11334),
        11287 => array(11335),
        11288 => array(11336),
        11289 => array(11337),
        11290 => array(11338),
        11291 => array(11339),
        11292 => array(11340),
        11293 => array(11341),
        11294 => array(11342),
        11295 => array(11343),
        11296 => array(11344),
        11297 => array(11345),
        11298 => array(11346),
        11299 => array(11347),
        11300 => array(11348),
        11301 => array(11349),
        11302 => array(11350),
        11303 => array(11351),
        11304 => array(11352),
        11305 => array(11353),
        11306 => array(11354),
        11307 => array(11355),
        11308 => array(11356),
        11309 => array(11357),
        11310 => array(11358),
        11311 => array(11359),
        11360 => array(11361),
        11362 => array(619),
        11363 => array(7549),
        11364 => array(637),
        11367 => array(11368),
        11369 => array(11370),
        11371 => array(11372),
        11373 => array(593),
        11374 => array(625),
        11375 => array(592),
        11376 => array(594),
        11378 => array(11379),
        11381 => array(11382),
        11388 => array(106),
        11389 => array(118),
        11390 => array(575),
        11391 => array(576),
        11392 => array(11393),
        11394 => array(11395),
        11396 => array(11397),
        11398 => array(11399),
        11400 => array(11401),
        11402 => array(11403),
        11404 => array(11405),
        11406 => array(11407),
        11408 => array(11409),
        11410 => array(11411),
        11412 => array(11413),
        11414 => array(11415),
        11416 => array(11417),
        11418 => array(11419),
        11420 => array(11421),
        11422 => array(11423),
        11424 => array(11425),
        11426 => array(11427),
        11428 => array(11429),
        11430 => array(11431),
        11432 => array(11433),
        11434 => array(11435),
        11436 => array(11437),
        11438 => array(11439),
        11440 => array(11441),
        11442 => array(11443),
        11444 => array(11445),
        11446 => array(11447),
        11448 => array(11449),
        11450 => array(11451),
        11452 => array(11453),
        11454 => array(11455),
        11456 => array(11457),
        11458 => array(11459),
        11460 => array(11461),
        11462 => array(11463),
        11464 => array(11465),
        11466 => array(11467),
        11468 => array(11469),
        11470 => array(11471),
        11472 => array(11473),
        11474 => array(11475),
        11476 => array(11477),
        11478 => array(11479),
        11480 => array(11481),
        11482 => array(11483),
        11484 => array(11485),
        11486 => array(11487),
        11488 => array(11489),
        11490 => array(11491),
        11499 => array(11500),
        11501 => array(11502),
        11506 => array(11507),
        11631 => array(11617),
        11935 => array(27597),
        12019 => array(40863),
        12032 => array(19968),
        12033 => array(20008),
        12034 => array(20022),
        12035 => array(20031),
        12036 => array(20057),
        12037 => array(20101),
        12038 => array(20108),
        12039 => array(20128),
        12040 => array(20154),
        12041 => array(20799),
        12042 => array(20837),
        12043 => array(20843),
        12044 => array(20866),
        12045 => array(20886),
        12046 => array(20907),
        12047 => array(20960),
        12048 => array(20981),
        12049 => array(20992),
        12050 => array(21147),
        12051 => array(21241),
        12052 => array(21269),
        12053 => array(21274),
        12054 => array(21304),
        12055 => array(21313),
        12056 => array(21340),
        12057 => array(21353),
        12058 => array(21378),
        12059 => array(21430),
        12060 => array(21448),
        12061 => array(21475),
        12062 => array(22231),
        12063 => array(22303),
        12064 => array(22763),
        12065 => array(22786),
        12066 => array(22794),
        12067 => array(22805),
        12068 => array(22823),
        12069 => array(22899),
        12070 => array(23376),
        12071 => array(23424),
        12072 => array(23544),
        12073 => array(23567),
        12074 => array(23586),
        12075 => array(23608),
        12076 => array(23662),
        12077 => array(23665),
        12078 => array(24027),
        12079 => array(24037),
        12080 => array(24049),
        12081 => array(24062),
        12082 => array(24178),
        12083 => array(24186),
        12084 => array(24191),
        12085 => array(24308),
        12086 => array(24318),
        12087 => array(24331),
        12088 => array(24339),
        12089 => array(24400),
        12090 => array(24417),
        12091 => array(24435),
        12092 => array(24515),
        12093 => array(25096),
        12094 => array(25142),
        12095 => array(25163),
        12096 => array(25903),
        12097 => array(25908),
        12098 => array(25991),
        12099 => array(26007),
        12100 => array(26020),
        12101 => array(26041),
        12102 => array(26080),
        12103 => array(26085),
        12104 => array(26352),
        12105 => array(26376),
        12106 => array(26408),
        12107 => array(27424),
        12108 => array(27490),
        12109 => array(27513),
        12110 => array(27571),
        12111 => array(27595),
        12112 => array(27604),
        12113 => array(27611),
        12114 => array(27663),
        12115 => array(27668),
        12116 => array(27700),
        12117 => array(28779),
        12118 => array(29226),
        12119 => array(29238),
        12120 => array(29243),
        12121 => array(29247),
        12122 => array(29255),
        12123 => array(29273),
        12124 => array(29275),
        12125 => array(29356),
        12126 => array(29572),
        12127 => array(29577),
        12128 => array(29916),
        12129 => array(29926),
        12130 => array(29976),
        12131 => array(29983),
        12132 => array(29992),
        12133 => array(30000),
        12134 => array(30091),
        12135 => array(30098),
        12136 => array(30326),
        12137 => array(30333),
        12138 => array(30382),
        12139 => array(30399),
        12140 => array(30446),
        12141 => array(30683),
        12142 => array(30690),
        12143 => array(30707),
        12144 => array(31034),
        12145 => array(31160),
        12146 => array(31166),
        12147 => array(31348),
        12148 => array(31435),
        12149 => array(31481),
        12150 => array(31859),
        12151 => array(31992),
        12152 => array(32566),
        12153 => array(32593),
        12154 => array(32650),
        12155 => array(32701),
        12156 => array(32769),
        12157 => array(32780),
        12158 => array(32786),
        12159 => array(32819),
        12160 => array(32895),
        12161 => array(32905),
        12162 => array(33251),
        12163 => array(33258),
        12164 => array(33267),
        12165 => array(33276),
        12166 => array(33292),
        12167 => array(33307),
        12168 => array(33311),
        12169 => array(33390),
        12170 => array(33394),
        12171 => array(33400),
        12172 => array(34381),
        12173 => array(34411),
        12174 => array(34880),
        12175 => array(34892),
        12176 => array(34915),
        12177 => array(35198),
        12178 => array(35211),
        12179 => array(35282),
        12180 => array(35328),
        12181 => array(35895),
        12182 => array(35910),
        12183 => array(35925),
        12184 => array(35960),
        12185 => array(35997),
        12186 => array(36196),
        12187 => array(36208),
        12188 => array(36275),
        12189 => array(36523),
        12190 => array(36554),
        12191 => array(36763),
        12192 => array(36784),
        12193 => array(36789),
        12194 => array(37009),
        12195 => array(37193),
        12196 => array(37318),
        12197 => array(37324),
        12198 => array(37329),
        12199 => array(38263),
        12200 => array(38272),
        12201 => array(38428),
        12202 => array(38582),
        12203 => array(38585),
        12204 => array(38632),
        12205 => array(38737),
        12206 => array(38750),
        12207 => array(38754),
        12208 => array(38761),
        12209 => array(38859),
        12210 => array(38893),
        12211 => array(38899),
        12212 => array(38913),
        12213 => array(39080),
        12214 => array(39131),
        12215 => array(39135),
        12216 => array(39318),
        12217 => array(39321),
        12218 => array(39340),
        12219 => array(39592),
        12220 => array(39640),
        12221 => array(39647),
        12222 => array(39717),
        12223 => array(39727),
        12224 => array(39730),
        12225 => array(39740),
        12226 => array(39770),
        12227 => array(40165),
        12228 => array(40565),
        12229 => array(40575),
        12230 => array(40613),
        12231 => array(40635),
        12232 => array(40643),
        12233 => array(40653),
        12234 => array(40657),
        12235 => array(40697),
        12236 => array(40701),
        12237 => array(40718),
        12238 => array(40723),
        12239 => array(40736),
        12240 => array(40763),
        12241 => array(40778),
        12242 => array(40786),
        12243 => array(40845),
        12244 => array(40860),
        12245 => array(40864),
        12290 => array(46),
        12342 => array(12306),
        12344 => array(21313),
        12345 => array(21316),
        12346 => array(21317),
        12447 => array(12424, 12426),
        12543 => array(12467, 12488),
        12593 => array(4352),
        12594 => array(4353),
        12595 => array(4522),
        12596 => array(4354),
        12597 => array(4524),
        12598 => array(4525),
        12599 => array(4355),
        12600 => array(4356),
        12601 => array(4357),
        12602 => array(4528),
        12603 => array(4529),
        12604 => array(4530),
        12605 => array(4531),
        12606 => array(4532),
        12607 => array(4533),
        12608 => array(4378),
        12609 => array(4358),
        12610 => array(4359),
        12611 => array(4360),
        12612 => array(4385),
        12613 => array(4361),
        12614 => array(4362),
        12615 => array(4363),
        12616 => array(4364),
        12617 => array(4365),
        12618 => array(4366),
        12619 => array(4367),
        12620 => array(4368),
        12621 => array(4369),
        12622 => array(4370),
        12623 => array(4449),
        12624 => array(4450),
        12625 => array(4451),
        12626 => array(4452),
        12627 => array(4453),
        12628 => array(4454),
        12629 => array(4455),
        12630 => array(4456),
        12631 => array(4457),
        12632 => array(4458),
        12633 => array(4459),
        12634 => array(4460),
        12635 => array(4461),
        12636 => array(4462),
        12637 => array(4463),
        12638 => array(4464),
        12639 => array(4465),
        12640 => array(4466),
        12641 => array(4467),
        12642 => array(4468),
        12643 => array(4469),
        12645 => array(4372),
        12646 => array(4373),
        12647 => array(4551),
        12648 => array(4552),
        12649 => array(4556),
        12650 => array(4558),
        12651 => array(4563),
        12652 => array(4567),
        12653 => array(4569),
        12654 => array(4380),
        12655 => array(4573),
        12656 => array(4575),
        12657 => array(4381),
        12658 => array(4382),
        12659 => array(4384),
        12660 => array(4386),
        12661 => array(4387),
        12662 => array(4391),
        12663 => array(4393),
        12664 => array(4395),
        12665 => array(4396),
        12666 => array(4397),
        12667 => array(4398),
        12668 => array(4399),
        12669 => array(4402),
        12670 => array(4406),
        12671 => array(4416),
        12672 => array(4423),
        12673 => array(4428),
        12674 => array(4593),
        12675 => array(4594),
        12676 => array(4439),
        12677 => array(4440),
        12678 => array(4441),
        12679 => array(4484),
        12680 => array(4485),
        12681 => array(4488),
        12682 => array(4497),
        12683 => array(4498),
        12684 => array(4500),
        12685 => array(4510),
        12686 => array(4513),
        12690 => array(19968),
        12691 => array(20108),
        12692 => array(19977),
        12693 => array(22235),
        12694 => array(19978),
        12695 => array(20013),
        12696 => array(19979),
        12697 => array(30002),
        12698 => array(20057),
        12699 => array(19993),
        12700 => array(19969),
        12701 => array(22825),
        12702 => array(22320),
        12703 => array(20154),
        12868 => array(21839),
        12869 => array(24188),
        12870 => array(25991),
        12871 => array(31631),
        12880 => array(112, 116, 101),
        12881 => array(50, 49),
        12882 => array(50, 50),
        12883 => array(50, 51),
        12884 => array(50, 52),
        12885 => array(50, 53),
        12886 => array(50, 54),
        12887 => array(50, 55),
        12888 => array(50, 56),
        12889 => array(50, 57),
        12890 => array(51, 48),
        12891 => array(51, 49),
        12892 => array(51, 50),
        12893 => array(51, 51),
        12894 => array(51, 52),
        12895 => array(51, 53),
        12896 => array(4352),
        12897 => array(4354),
        12898 => array(4355),
        12899 => array(4357),
        12900 => array(4358),
        12901 => array(4359),
        12902 => array(4361),
        12903 => array(4363),
        12904 => array(4364),
        12905 => array(4366),
        12906 => array(4367),
        12907 => array(4368),
        12908 => array(4369),
        12909 => array(4370),
        12910 => array(44032),
        12911 => array(45208),
        12912 => array(45796),
        12913 => array(46972),
        12914 => array(47560),
        12915 => array(48148),
        12916 => array(49324),
        12917 => array(50500),
        12918 => array(51088),
        12919 => array(52264),
        12920 => array(52852),
        12921 => array(53440),
        12922 => array(54028),
        12923 => array(54616),
        12924 => array(52280, 44256),
        12925 => array(51452, 51032),
        12926 => array(50864),
        12928 => array(19968),
        12929 => array(20108),
        12930 => array(19977),
        12931 => array(22235),
        12932 => array(20116),
        12933 => array(20845),
        12934 => array(19971),
        12935 => array(20843),
        12936 => array(20061),
        12937 => array(21313),
        12938 => array(26376),
        12939 => array(28779),
        12940 => array(27700),
        12941 => array(26408),
        12942 => array(37329),
        12943 => array(22303),
        12944 => array(26085),
        12945 => array(26666),
        12946 => array(26377),
        12947 => array(31038),
        12948 => array(21517),
        12949 => array(29305),
        12950 => array(36001),
        12951 => array(31069),
        12952 => array(21172),
        12953 => array(31192),
        12954 => array(30007),
        12955 => array(22899),
        12956 => array(36969),
        12957 => array(20778),
        12958 => array(21360),
        12959 => array(27880),
        12960 => array(38917),
        12961 => array(20241),
        12962 => array(20889),
        12963 => array(27491),
        12964 => array(19978),
        12965 => array(20013),
        12966 => array(19979),
        12967 => array(24038),
        12968 => array(21491),
        12969 => array(21307),
        12970 => array(23447),
        12971 => array(23398),
        12972 => array(30435),
        12973 => array(20225),
        12974 => array(36039),
        12975 => array(21332),
        12976 => array(22812),
        12977 => array(51, 54),
        12978 => array(51, 55),
        12979 => array(51, 56),
        12980 => array(51, 57),
        12981 => array(52, 48),
        12982 => array(52, 49),
        12983 => array(52, 50),
        12984 => array(52, 51),
        12985 => array(52, 52),
        12986 => array(52, 53),
        12987 => array(52, 54),
        12988 => array(52, 55),
        12989 => array(52, 56),
        12990 => array(52, 57),
        12991 => array(53, 48),
        12992 => array(49, 26376),
        12993 => array(50, 26376),
        12994 => array(51, 26376),
        12995 => array(52, 26376),
        12996 => array(53, 26376),
        12997 => array(54, 26376),
        12998 => array(55, 26376),
        12999 => array(56, 26376),
        13000 => array(57, 26376),
        13001 => array(49, 48, 26376),
        13002 => array(49, 49, 26376),
        13003 => array(49, 50, 26376),
        13004 => array(104, 103),
        13005 => array(101, 114, 103),
        13006 => array(101, 118),
        13007 => array(108, 116, 100),
        13008 => array(12450),
        13009 => array(12452),
        13010 => array(12454),
        13011 => array(12456),
        13012 => array(12458),
        13013 => array(12459),
        13014 => array(12461),
        13015 => array(12463),
        13016 => array(12465),
        13017 => array(12467),
        13018 => array(12469),
        13019 => array(12471),
        13020 => array(12473),
        13021 => array(12475),
        13022 => array(12477),
        13023 => array(12479),
        13024 => array(12481),
        13025 => array(12484),
        13026 => array(12486),
        13027 => array(12488),
        13028 => array(12490),
        13029 => array(12491),
        13030 => array(12492),
        13031 => array(12493),
        13032 => array(12494),
        13033 => array(12495),
        13034 => array(12498),
        13035 => array(12501),
        13036 => array(12504),
        13037 => array(12507),
        13038 => array(12510),
        13039 => array(12511),
        13040 => array(12512),
        13041 => array(12513),
        13042 => array(12514),
        13043 => array(12516),
        13044 => array(12518),
        13045 => array(12520),
        13046 => array(12521),
        13047 => array(12522),
        13048 => array(12523),
        13049 => array(12524),
        13050 => array(12525),
        13051 => array(12527),
        13052 => array(12528),
        13053 => array(12529),
        13054 => array(12530),
        13055 => array(20196, 21644),
        13056 => array(12450, 12497, 12540, 12488),
        13057 => array(12450, 12523, 12501, 12449),
        13058 => array(12450, 12531, 12506, 12450),
        13059 => array(12450, 12540, 12523),
        13060 => array(12452, 12491, 12531, 12464),
        13061 => array(12452, 12531, 12481),
        13062 => array(12454, 12457, 12531),
        13063 => array(12456, 12473, 12463, 12540, 12489),
        13064 => array(12456, 12540, 12459, 12540),
        13065 => array(12458, 12531, 12473),
        13066 => array(12458, 12540, 12512),
        13067 => array(12459, 12452, 12522),
        13068 => array(12459, 12521, 12483, 12488),
        13069 => array(12459, 12525, 12522, 12540),
        13070 => array(12460, 12525, 12531),
        13071 => array(12460, 12531, 12510),
        13072 => array(12462, 12460),
        13073 => array(12462, 12491, 12540),
        13074 => array(12461, 12517, 12522, 12540),
        13075 => array(12462, 12523, 12480, 12540),
        13076 => array(12461, 12525),
        13077 => array(12461, 12525, 12464, 12521, 12512),
        13078 => array(12461, 12525, 12513, 12540, 12488, 12523),
        13079 => array(12461, 12525, 12527, 12483, 12488),
        13080 => array(12464, 12521, 12512),
        13081 => array(12464, 12521, 12512, 12488, 12531),
        13082 => array(12463, 12523, 12476, 12452, 12525),
        13083 => array(12463, 12525, 12540, 12493),
        13084 => array(12465, 12540, 12473),
        13085 => array(12467, 12523, 12490),
        13086 => array(12467, 12540, 12509),
        13087 => array(12469, 12452, 12463, 12523),
        13088 => array(12469, 12531, 12481, 12540, 12512),
        13089 => array(12471, 12522, 12531, 12464),
        13090 => array(12475, 12531, 12481),
        13091 => array(12475, 12531, 12488),
        13092 => array(12480, 12540, 12473),
        13093 => array(12487, 12471),
        13094 => array(12489, 12523),
        13095 => array(12488, 12531),
        13096 => array(12490, 12494),
        13097 => array(12494, 12483, 12488),
        13098 => array(12495, 12452, 12484),
        13099 => array(12497, 12540, 12475, 12531, 12488),
        13100 => array(12497, 12540, 12484),
        13101 => array(12496, 12540, 12524, 12523),
        13102 => array(12500, 12450, 12473, 12488, 12523),
        13103 => array(12500, 12463, 12523),
        13104 => array(12500, 12467),
        13105 => array(12499, 12523),
        13106 => array(12501, 12449, 12521, 12483, 12489),
        13107 => array(12501, 12451, 12540, 12488),
        13108 => array(12502, 12483, 12471, 12455, 12523),
        13109 => array(12501, 12521, 12531),
        13110 => array(12504, 12463, 12479, 12540, 12523),
        13111 => array(12506, 12477),
        13112 => array(12506, 12491, 12498),
        13113 => array(12504, 12523, 12484),
        13114 => array(12506, 12531, 12473),
        13115 => array(12506, 12540, 12472),
        13116 => array(12505, 12540, 12479),
        13117 => array(12509, 12452, 12531, 12488),
        13118 => array(12508, 12523, 12488),
        13119 => array(12507, 12531),
        13120 => array(12509, 12531, 12489),
        13121 => array(12507, 12540, 12523),
        13122 => array(12507, 12540, 12531),
        13123 => array(12510, 12452, 12463, 12525),
        13124 => array(12510, 12452, 12523),
        13125 => array(12510, 12483, 12495),
        13126 => array(12510, 12523, 12463),
        13127 => array(12510, 12531, 12471, 12519, 12531),
        13128 => array(12511, 12463, 12525, 12531),
        13129 => array(12511, 12522),
        13130 => array(12511, 12522, 12496, 12540, 12523),
        13131 => array(12513, 12460),
        13132 => array(12513, 12460, 12488, 12531),
        13133 => array(12513, 12540, 12488, 12523),
        13134 => array(12516, 12540, 12489),
        13135 => array(12516, 12540, 12523),
        13136 => array(12518, 12450, 12531),
        13137 => array(12522, 12483, 12488, 12523),
        13138 => array(12522, 12521),
        13139 => array(12523, 12500, 12540),
        13140 => array(12523, 12540, 12502, 12523),
        13141 => array(12524, 12512),
        13142 => array(12524, 12531, 12488, 12466, 12531),
        13143 => array(12527, 12483, 12488),
        13144 => array(48, 28857),
        13145 => array(49, 28857),
        13146 => array(50, 28857),
        13147 => array(51, 28857),
        13148 => array(52, 28857),
        13149 => array(53, 28857),
        13150 => array(54, 28857),
        13151 => array(55, 28857),
        13152 => array(56, 28857),
        13153 => array(57, 28857),
        13154 => array(49, 48, 28857),
        13155 => array(49, 49, 28857),
        13156 => array(49, 50, 28857),
        13157 => array(49, 51, 28857),
        13158 => array(49, 52, 28857),
        13159 => array(49, 53, 28857),
        13160 => array(49, 54, 28857),
        13161 => array(49, 55, 28857),
        13162 => array(49, 56, 28857),
        13163 => array(49, 57, 28857),
        13164 => array(50, 48, 28857),
        13165 => array(50, 49, 28857),
        13166 => array(50, 50, 28857),
        13167 => array(50, 51, 28857),
        13168 => array(50, 52, 28857),
        13169 => array(104, 112, 97),
        13170 => array(100, 97),
        13171 => array(97, 117),
        13172 => array(98, 97, 114),
        13173 => array(111, 118),
        13174 => array(112, 99),
        13175 => array(100, 109),
        13176 => array(100, 109, 50),
        13177 => array(100, 109, 51),
        13178 => array(105, 117),
        13179 => array(24179, 25104),
        13180 => array(26157, 21644),
        13181 => array(22823, 27491),
        13182 => array(26126, 27835),
        13183 => array(26666, 24335, 20250, 31038),
        13184 => array(112, 97),
        13185 => array(110, 97),
        13186 => array(956, 97),
        13187 => array(109, 97),
        13188 => array(107, 97),
        13189 => array(107, 98),
        13190 => array(109, 98),
        13191 => array(103, 98),
        13192 => array(99, 97, 108),
        13193 => array(107, 99, 97, 108),
        13194 => array(112, 102),
        13195 => array(110, 102),
        13196 => array(956, 102),
        13197 => array(956, 103),
        13198 => array(109, 103),
        13199 => array(107, 103),
        13200 => array(104, 122),
        13201 => array(107, 104, 122),
        13202 => array(109, 104, 122),
        13203 => array(103, 104, 122),
        13204 => array(116, 104, 122),
        13205 => array(956, 108),
        13206 => array(109, 108),
        13207 => array(100, 108),
        13208 => array(107, 108),
        13209 => array(102, 109),
        13210 => array(110, 109),
        13211 => array(956, 109),
        13212 => array(109, 109),
        13213 => array(99, 109),
        13214 => array(107, 109),
        13215 => array(109, 109, 50),
        13216 => array(99, 109, 50),
        13217 => array(109, 50),
        13218 => array(107, 109, 50),
        13219 => array(109, 109, 51),
        13220 => array(99, 109, 51),
        13221 => array(109, 51),
        13222 => array(107, 109, 51),
        13223 => array(109, 8725, 115),
        13224 => array(109, 8725, 115, 50),
        13225 => array(112, 97),
        13226 => array(107, 112, 97),
        13227 => array(109, 112, 97),
        13228 => array(103, 112, 97),
        13229 => array(114, 97, 100),
        13230 => array(114, 97, 100, 8725, 115),
        13231 => array(114, 97, 100, 8725, 115, 50),
        13232 => array(112, 115),
        13233 => array(110, 115),
        13234 => array(956, 115),
        13235 => array(109, 115),
        13236 => array(112, 118),
        13237 => array(110, 118),
        13238 => array(956, 118),
        13239 => array(109, 118),
        13240 => array(107, 118),
        13241 => array(109, 118),
        13242 => array(112, 119),
        13243 => array(110, 119),
        13244 => array(956, 119),
        13245 => array(109, 119),
        13246 => array(107, 119),
        13247 => array(109, 119),
        13248 => array(107, 969),
        13249 => array(109, 969),
        13251 => array(98, 113),
        13252 => array(99, 99),
        13253 => array(99, 100),
        13254 => array(99, 8725, 107, 103),
        13256 => array(100, 98),
        13257 => array(103, 121),
        13258 => array(104, 97),
        13259 => array(104, 112),
        13260 => array(105, 110),
        13261 => array(107, 107),
        13262 => array(107, 109),
        13263 => array(107, 116),
        13264 => array(108, 109),
        13265 => array(108, 110),
        13266 => array(108, 111, 103),
        13267 => array(108, 120),
        13268 => array(109, 98),
        13269 => array(109, 105, 108),
        13270 => array(109, 111, 108),
        13271 => array(112, 104),
        13273 => array(112, 112, 109),
        13274 => array(112, 114),
        13275 => array(115, 114),
        13276 => array(115, 118),
        13277 => array(119, 98),
        13278 => array(118, 8725, 109),
        13279 => array(97, 8725, 109),
        13280 => array(49, 26085),
        13281 => array(50, 26085),
        13282 => array(51, 26085),
        13283 => array(52, 26085),
        13284 => array(53, 26085),
        13285 => array(54, 26085),
        13286 => array(55, 26085),
        13287 => array(56, 26085),
        13288 => array(57, 26085),
        13289 => array(49, 48, 26085),
        13290 => array(49, 49, 26085),
        13291 => array(49, 50, 26085),
        13292 => array(49, 51, 26085),
        13293 => array(49, 52, 26085),
        13294 => array(49, 53, 26085),
        13295 => array(49, 54, 26085),
        13296 => array(49, 55, 26085),
        13297 => array(49, 56, 26085),
        13298 => array(49, 57, 26085),
        13299 => array(50, 48, 26085),
        13300 => array(50, 49, 26085),
        13301 => array(50, 50, 26085),
        13302 => array(50, 51, 26085),
        13303 => array(50, 52, 26085),
        13304 => array(50, 53, 26085),
        13305 => array(50, 54, 26085),
        13306 => array(50, 55, 26085),
        13307 => array(50, 56, 26085),
        13308 => array(50, 57, 26085),
        13309 => array(51, 48, 26085),
        13310 => array(51, 49, 26085),
        13311 => array(103, 97, 108),
        42560 => array(42561),
        42562 => array(42563),
        42564 => array(42565),
        42566 => array(42567),
        42568 => array(42569),
        42570 => array(42571),
        42572 => array(42573),
        42574 => array(42575),
        42576 => array(42577),
        42578 => array(42579),
        42580 => array(42581),
        42582 => array(42583),
        42584 => array(42585),
        42586 => array(42587),
        42588 => array(42589),
        42590 => array(42591),
        42592 => array(42593),
        42594 => array(42595),
        42596 => array(42597),
        42598 => array(42599),
        42600 => array(42601),
        42602 => array(42603),
        42604 => array(42605),
        42624 => array(42625),
        42626 => array(42627),
        42628 => array(42629),
        42630 => array(42631),
        42632 => array(42633),
        42634 => array(42635),
        42636 => array(42637),
        42638 => array(42639),
        42640 => array(42641),
        42642 => array(42643),
        42644 => array(42645),
        42646 => array(42647),
        42648 => array(42649),
        42650 => array(42651),
        42652 => array(1098),
        42653 => array(1100),
        42786 => array(42787),
        42788 => array(42789),
        42790 => array(42791),
        42792 => array(42793),
        42794 => array(42795),
        42796 => array(42797),
        42798 => array(42799),
        42802 => array(42803),
        42804 => array(42805),
        42806 => array(42807),
        42808 => array(42809),
        42810 => array(42811),
        42812 => array(42813),
        42814 => array(42815),
        42816 => array(42817),
        42818 => array(42819),
        42820 => array(42821),
        42822 => array(42823),
        42824 => array(42825),
        42826 => array(42827),
        42828 => array(42829),
        42830 => array(42831),
        42832 => array(42833),
        42834 => array(42835),
        42836 => array(42837),
        42838 => array(42839),
        42840 => array(42841),
        42842 => array(42843),
        42844 => array(42845),
        42846 => array(42847),
        42848 => array(42849),
        42850 => array(42851),
        42852 => array(42853),
        42854 => array(42855),
        42856 => array(42857),
        42858 => array(42859),
        42860 => array(42861),
        42862 => array(42863),
        42864 => array(42863),
        42873 => array(42874),
        42875 => array(42876),
        42877 => array(7545),
        42878 => array(42879),
        42880 => array(42881),
        42882 => array(42883),
        42884 => array(42885),
        42886 => array(42887),
        42891 => array(42892),
        42893 => array(613),
        42896 => array(42897),
        42898 => array(42899),
        42902 => array(42903),
        42904 => array(42905),
        42906 => array(42907),
        42908 => array(42909),
        42910 => array(42911),
        42912 => array(42913),
        42914 => array(42915),
        42916 => array(42917),
        42918 => array(42919),
        42920 => array(42921),
        42922 => array(614),
        42923 => array(604),
        42924 => array(609),
        42925 => array(620),
        42926 => array(618),
        42928 => array(670),
        42929 => array(647),
        42930 => array(669),
        42931 => array(43859),
        42932 => array(42933),
        42934 => array(42935),
        42936 => array(42937),
        42938 => array(42939),
        42940 => array(42941),
        42942 => array(42943),
        42944 => array(42945),
        42946 => array(42947),
        42948 => array(42900),
        42949 => array(642),
        42950 => array(7566),
        42951 => array(42952),
        42953 => array(42954),
        42960 => array(42961),
        42966 => array(42967),
        42968 => array(42969),
        42994 => array(99),
        42995 => array(102),
        42996 => array(113),
        42997 => array(42998),
        43000 => array(295),
        43001 => array(339),
        43868 => array(42791),
        43869 => array(43831),
        43870 => array(619),
        43871 => array(43858),
        43881 => array(653),
        43888 => array(5024),
        43889 => array(5025),
        43890 => array(5026),
        43891 => array(5027),
        43892 => array(5028),
        43893 => array(5029),
        43894 => array(5030),
        43895 => array(5031),
        43896 => array(5032),
        43897 => array(5033),
        43898 => array(5034),
        43899 => array(5035),
        43900 => array(5036),
        43901 => array(5037),
        43902 => array(5038),
        43903 => array(5039),
        43904 => array(5040),
        43905 => array(5041),
        43906 => array(5042),
        43907 => array(5043),
        43908 => array(5044),
        43909 => array(5045),
        43910 => array(5046),
        43911 => array(5047),
        43912 => array(5048),
        43913 => array(5049),
        43914 => array(5050),
        43915 => array(5051),
        43916 => array(5052),
        43917 => array(5053),
        43918 => array(5054),
        43919 => array(5055),
        43920 => array(5056),
        43921 => array(5057),
        43922 => array(5058),
        43923 => array(5059),
        43924 => array(5060),
        43925 => array(5061),
        43926 => array(5062),
        43927 => array(5063),
        43928 => array(5064),
        43929 => array(5065),
        43930 => array(5066),
        43931 => array(5067),
        43932 => array(5068),
        43933 => array(5069),
        43934 => array(5070),
        43935 => array(5071),
        43936 => array(5072),
        43937 => array(5073),
        43938 => array(5074),
        43939 => array(5075),
        43940 => array(5076),
        43941 => array(5077),
        43942 => array(5078),
        43943 => array(5079),
        43944 => array(5080),
        43945 => array(5081),
        43946 => array(5082),
        43947 => array(5083),
        43948 => array(5084),
        43949 => array(5085),
        43950 => array(5086),
        43951 => array(5087),
        43952 => array(5088),
        43953 => array(5089),
        43954 => array(5090),
        43955 => array(5091),
        43956 => array(5092),
        43957 => array(5093),
        43958 => array(5094),
        43959 => array(5095),
        43960 => array(5096),
        43961 => array(5097),
        43962 => array(5098),
        43963 => array(5099),
        43964 => array(5100),
        43965 => array(5101),
        43966 => array(5102),
        43967 => array(5103),
        63744 => array(35912),
        63745 => array(26356),
        63746 => array(36554),
        63747 => array(36040),
        63748 => array(28369),
        63749 => array(20018),
        63750 => array(21477),
        63751 => array(40860),
        63752 => array(40860),
        63753 => array(22865),
        63754 => array(37329),
        63755 => array(21895),
        63756 => array(22856),
        63757 => array(25078),
        63758 => array(30313),
        63759 => array(32645),
        63760 => array(34367),
        63761 => array(34746),
        63762 => array(35064),
        63763 => array(37007),
        63764 => array(27138),
        63765 => array(27931),
        63766 => array(28889),
        63767 => array(29662),
        63768 => array(33853),
        63769 => array(37226),
        63770 => array(39409),
        63771 => array(20098),
        63772 => array(21365),
        63773 => array(27396),
        63774 => array(29211),
        63775 => array(34349),
        63776 => array(40478),
        63777 => array(23888),
        63778 => array(28651),
        63779 => array(34253),
        63780 => array(35172),
        63781 => array(25289),
        63782 => array(33240),
        63783 => array(34847),
        63784 => array(24266),
        63785 => array(26391),
        63786 => array(28010),
        63787 => array(29436),
        63788 => array(37070),
        63789 => array(20358),
        63790 => array(20919),
        63791 => array(21214),
        63792 => array(25796),
        63793 => array(27347),
        63794 => array(29200),
        63795 => array(30439),
        63796 => array(32769),
        63797 => array(34310),
        63798 => array(34396),
        63799 => array(36335),
        63800 => array(38706),
        63801 => array(39791),
        63802 => array(40442),
        63803 => array(30860),
        63804 => array(31103),
        63805 => array(32160),
        63806 => array(33737),
        63807 => array(37636),
        63808 => array(40575),
        63809 => array(35542),
        63810 => array(22751),
        63811 => array(24324),
        63812 => array(31840),
        63813 => array(32894),
        63814 => array(29282),
        63815 => array(30922),
        63816 => array(36034),
        63817 => array(38647),
        63818 => array(22744),
        63819 => array(23650),
        63820 => array(27155),
        63821 => array(28122),
        63822 => array(28431),
        63823 => array(32047),
        63824 => array(32311),
        63825 => array(38475),
        63826 => array(21202),
        63827 => array(32907),
        63828 => array(20956),
        63829 => array(20940),
        63830 => array(31260),
        63831 => array(32190),
        63832 => array(33777),
        63833 => array(38517),
        63834 => array(35712),
        63835 => array(25295),
        63836 => array(27138),
        63837 => array(35582),
        63838 => array(20025),
        63839 => array(23527),
        63840 => array(24594),
        63841 => array(29575),
        63842 => array(30064),
        63843 => array(21271),
        63844 => array(30971),
        63845 => array(20415),
        63846 => array(24489),
        63847 => array(19981),
        63848 => array(27852),
        63849 => array(25976),
        63850 => array(32034),
        63851 => array(21443),
        63852 => array(22622),
        63853 => array(30465),
        63854 => array(33865),
        63855 => array(35498),
        63856 => array(27578),
        63857 => array(36784),
        63858 => array(27784),
        63859 => array(25342),
        63860 => array(33509),
        63861 => array(25504),
        63862 => array(30053),
        63863 => array(20142),
        63864 => array(20841),
        63865 => array(20937),
        63866 => array(26753),
        63867 => array(31975),
        63868 => array(33391),
        63869 => array(35538),
        63870 => array(37327),
        63871 => array(21237),
        63872 => array(21570),
        63873 => array(22899),
        63874 => array(24300),
        63875 => array(26053),
        63876 => array(28670),
        63877 => array(31018),
        63878 => array(38317),
        63879 => array(39530),
        63880 => array(40599),
        63881 => array(40654),
        63882 => array(21147),
        63883 => array(26310),
        63884 => array(27511),
        63885 => array(36706),
        63886 => array(24180),
        63887 => array(24976),
        63888 => array(25088),
        63889 => array(25754),
        63890 => array(28451),
        63891 => array(29001),
        63892 => array(29833),
        63893 => array(31178),
        63894 => array(32244),
        63895 => array(32879),
        63896 => array(36646),
        63897 => array(34030),
        63898 => array(36899),
        63899 => array(37706),
        63900 => array(21015),
        63901 => array(21155),
        63902 => array(21693),
        63903 => array(28872),
        63904 => array(35010),
        63905 => array(35498),
        63906 => array(24265),
        63907 => array(24565),
        63908 => array(25467),
        63909 => array(27566),
        63910 => array(31806),
        63911 => array(29557),
        63912 => array(20196),
        63913 => array(22265),
        63914 => array(23527),
        63915 => array(23994),
        63916 => array(24604),
        63917 => array(29618),
        63918 => array(29801),
        63919 => array(32666),
        63920 => array(32838),
        63921 => array(37428),
        63922 => array(38646),
        63923 => array(38728),
        63924 => array(38936),
        63925 => array(20363),
        63926 => array(31150),
        63927 => array(37300),
        63928 => array(38584),
        63929 => array(24801),
        63930 => array(20102),
        63931 => array(20698),
        63932 => array(23534),
        63933 => array(23615),
        63934 => array(26009),
        63935 => array(27138),
        63936 => array(29134),
        63937 => array(30274),
        63938 => array(34044),
        63939 => array(36988),
        63940 => array(40845),
        63941 => array(26248),
        63942 => array(38446),
        63943 => array(21129),
        63944 => array(26491),
        63945 => array(26611),
        63946 => array(27969),
        63947 => array(28316),
        63948 => array(29705),
        63949 => array(30041),
        63950 => array(30827),
        63951 => array(32016),
        63952 => array(39006),
        63953 => array(20845),
        63954 => array(25134),
        63955 => array(38520),
        63956 => array(20523),
        63957 => array(23833),
        63958 => array(28138),
        63959 => array(36650),
        63960 => array(24459),
        63961 => array(24900),
        63962 => array(26647),
        63963 => array(29575),
        63964 => array(38534),
        63965 => array(21033),
        63966 => array(21519),
        63967 => array(23653),
        63968 => array(26131),
        63969 => array(26446),
        63970 => array(26792),
        63971 => array(27877),
        63972 => array(29702),
        63973 => array(30178),
        63974 => array(32633),
        63975 => array(35023),
        63976 => array(35041),
        63977 => array(37324),
        63978 => array(38626),
        63979 => array(21311),
        63980 => array(28346),
        63981 => array(21533),
        63982 => array(29136),
        63983 => array(29848),
        63984 => array(34298),
        63985 => array(38563),
        63986 => array(40023),
        63987 => array(40607),
        63988 => array(26519),
        63989 => array(28107),
        63990 => array(33256),
        63991 => array(31435),
        63992 => array(31520),
        63993 => array(31890),
        63994 => array(29376),
        63995 => array(28825),
        63996 => array(35672),
        63997 => array(20160),
        63998 => array(33590),
        63999 => array(21050),
        64000 => array(20999),
        64001 => array(24230),
        64002 => array(25299),
        64003 => array(31958),
        64004 => array(23429),
        64005 => array(27934),
        64006 => array(26292),
        64007 => array(36667),
        64008 => array(34892),
        64009 => array(38477),
        64010 => array(35211),
        64011 => array(24275),
        64012 => array(20800),
        64013 => array(21952),
        64016 => array(22618),
        64018 => array(26228),
        64021 => array(20958),
        64022 => array(29482),
        64023 => array(30410),
        64024 => array(31036),
        64025 => array(31070),
        64026 => array(31077),
        64027 => array(31119),
        64028 => array(38742),
        64029 => array(31934),
        64030 => array(32701),
        64032 => array(34322),
        64034 => array(35576),
        64037 => array(36920),
        64038 => array(37117),
        64042 => array(39151),
        64043 => array(39164),
        64044 => array(39208),
        64045 => array(40372),
        64046 => array(37086),
        64047 => array(38583),
        64048 => array(20398),
        64049 => array(20711),
        64050 => array(20813),
        64051 => array(21193),
        64052 => array(21220),
        64053 => array(21329),
        64054 => array(21917),
        64055 => array(22022),
        64056 => array(22120),
        64057 => array(22592),
        64058 => array(22696),
        64059 => array(23652),
        64060 => array(23662),
        64061 => array(24724),
        64062 => array(24936),
        64063 => array(24974),
        64064 => array(25074),
        64065 => array(25935),
        64066 => array(26082),
        64067 => array(26257),
        64068 => array(26757),
        64069 => array(28023),
        64070 => array(28186),
        64071 => array(28450),
        64072 => array(29038),
        64073 => array(29227),
        64074 => array(29730),
        64075 => array(30865),
        64076 => array(31038),
        64077 => array(31049),
        64078 => array(31048),
        64079 => array(31056),
        64080 => array(31062),
        64081 => array(31069),
        64082 => array(31117),
        64083 => array(31118),
        64084 => array(31296),
        64085 => array(31361),
        64086 => array(31680),
        64087 => array(32244),
        64088 => array(32265),
        64089 => array(32321),
        64090 => array(32626),
        64091 => array(32773),
        64092 => array(33261),
        64093 => array(33401),
        64094 => array(33401),
        64095 => array(33879),
        64096 => array(35088),
        64097 => array(35222),
        64098 => array(35585),
        64099 => array(35641),
        64100 => array(36051),
        64101 => array(36104),
        64102 => array(36790),
        64103 => array(36920),
        64104 => array(38627),
        64105 => array(38911),
        64106 => array(38971),
        64107 => array(24693),
        64108 => array(148206),
        64109 => array(33304),
        64112 => array(20006),
        64113 => array(20917),
        64114 => array(20840),
        64115 => array(20352),
        64116 => array(20805),
        64117 => array(20864),
        64118 => array(21191),
        64119 => array(21242),
        64120 => array(21917),
        64121 => array(21845),
        64122 => array(21913),
        64123 => array(21986),
        64124 => array(22618),
        64125 => array(22707),
        64126 => array(22852),
        64127 => array(22868),
        64128 => array(23138),
        64129 => array(23336),
        64130 => array(24274),
        64131 => array(24281),
        64132 => array(24425),
        64133 => array(24493),
        64134 => array(24792),
        64135 => array(24910),
        64136 => array(24840),
        64137 => array(24974),
        64138 => array(24928),
        64139 => array(25074),
        64140 => array(25140),
        64141 => array(25540),
        64142 => array(25628),
        64143 => array(25682),
        64144 => array(25942),
        64145 => array(26228),
        64146 => array(26391),
        64147 => array(26395),
        64148 => array(26454),
        64149 => array(27513),
        64150 => array(27578),
        64151 => array(27969),
        64152 => array(28379),
        64153 => array(28363),
        64154 => array(28450),
        64155 => array(28702),
        64156 => array(29038),
        64157 => array(30631),
        64158 => array(29237),
        64159 => array(29359),
        64160 => array(29482),
        64161 => array(29809),
        64162 => array(29958),
        64163 => array(30011),
        64164 => array(30237),
        64165 => array(30239),
        64166 => array(30410),
        64167 => array(30427),
        64168 => array(30452),
        64169 => array(30538),
        64170 => array(30528),
        64171 => array(30924),
        64172 => array(31409),
        64173 => array(31680),
        64174 => array(31867),
        64175 => array(32091),
        64176 => array(32244),
        64177 => array(32574),
        64178 => array(32773),
        64179 => array(33618),
        64180 => array(33775),
        64181 => array(34681),
        64182 => array(35137),
        64183 => array(35206),
        64184 => array(35222),
        64185 => array(35519),
        64186 => array(35576),
        64187 => array(35531),
        64188 => array(35585),
        64189 => array(35582),
        64190 => array(35565),
        64191 => array(35641),
        64192 => array(35722),
        64193 => array(36104),
        64194 => array(36664),
        64195 => array(36978),
        64196 => array(37273),
        64197 => array(37494),
        64198 => array(38524),
        64199 => array(38627),
        64200 => array(38742),
        64201 => array(38875),
        64202 => array(38911),
        64203 => array(38923),
        64204 => array(38971),
        64205 => array(39698),
        64206 => array(40860),
        64207 => array(141386),
        64208 => array(141380),
        64209 => array(144341),
        64210 => array(15261),
        64211 => array(16408),
        64212 => array(16441),
        64213 => array(152137),
        64214 => array(154832),
        64215 => array(163539),
        64216 => array(40771),
        64217 => array(40846),
        64256 => array(102, 102),
        64257 => array(102, 105),
        64258 => array(102, 108),
        64259 => array(102, 102, 105),
        64260 => array(102, 102, 108),
        64261 => array(115, 116),
        64262 => array(115, 116),
        64275 => array(1396, 1398),
        64276 => array(1396, 1381),
        64277 => array(1396, 1387),
        64278 => array(1406, 1398),
        64279 => array(1396, 1389),
        64285 => array(1497, 1460),
        64287 => array(1522, 1463),
        64288 => array(1506),
        64289 => array(1488),
        64290 => array(1491),
        64291 => array(1492),
        64292 => array(1499),
        64293 => array(1500),
        64294 => array(1501),
        64295 => array(1512),
        64296 => array(1514),
        64298 => array(1513, 1473),
        64299 => array(1513, 1474),
        64300 => array(1513, 1468, 1473),
        64301 => array(1513, 1468, 1474),
        64302 => array(1488, 1463),
        64303 => array(1488, 1464),
        64304 => array(1488, 1468),
        64305 => array(1489, 1468),
        64306 => array(1490, 1468),
        64307 => array(1491, 1468),
        64308 => array(1492, 1468),
        64309 => array(1493, 1468),
        64310 => array(1494, 1468),
        64312 => array(1496, 1468),
        64313 => array(1497, 1468),
        64314 => array(1498, 1468),
        64315 => array(1499, 1468),
        64316 => array(1500, 1468),
        64318 => array(1502, 1468),
        64320 => array(1504, 1468),
        64321 => array(1505, 1468),
        64323 => array(1507, 1468),
        64324 => array(1508, 1468),
        64326 => array(1510, 1468),
        64327 => array(1511, 1468),
        64328 => array(1512, 1468),
        64329 => array(1513, 1468),
        64330 => array(1514, 1468),
        64331 => array(1493, 1465),
        64332 => array(1489, 1471),
        64333 => array(1499, 1471),
        64334 => array(1508, 1471),
        64335 => array(1488, 1500),
        64336 => array(1649),
        64337 => array(1649),
        64338 => array(1659),
        64339 => array(1659),
        64340 => array(1659),
        64341 => array(1659),
        64342 => array(1662),
        64343 => array(1662),
        64344 => array(1662),
        64345 => array(1662),
        64346 => array(1664),
        64347 => array(1664),
        64348 => array(1664),
        64349 => array(1664),
        64350 => array(1658),
        64351 => array(1658),
        64352 => array(1658),
        64353 => array(1658),
        64354 => array(1663),
        64355 => array(1663),
        64356 => array(1663),
        64357 => array(1663),
        64358 => array(1657),
        64359 => array(1657),
        64360 => array(1657),
        64361 => array(1657),
        64362 => array(1700),
        64363 => array(1700),
        64364 => array(1700),
        64365 => array(1700),
        64366 => array(1702),
        64367 => array(1702),
        64368 => array(1702),
        64369 => array(1702),
        64370 => array(1668),
        64371 => array(1668),
        64372 => array(1668),
        64373 => array(1668),
        64374 => array(1667),
        64375 => array(1667),
        64376 => array(1667),
        64377 => array(1667),
        64378 => array(1670),
        64379 => array(1670),
        64380 => array(1670),
        64381 => array(1670),
        64382 => array(1671),
        64383 => array(1671),
        64384 => array(1671),
        64385 => array(1671),
        64386 => array(1677),
        64387 => array(1677),
        64388 => array(1676),
        64389 => array(1676),
        64390 => array(1678),
        64391 => array(1678),
        64392 => array(1672),
        64393 => array(1672),
        64394 => array(1688),
        64395 => array(1688),
        64396 => array(1681),
        64397 => array(1681),
        64398 => array(1705),
        64399 => array(1705),
        64400 => array(1705),
        64401 => array(1705),
        64402 => array(1711),
        64403 => array(1711),
        64404 => array(1711),
        64405 => array(1711),
        64406 => array(1715),
        64407 => array(1715),
        64408 => array(1715),
        64409 => array(1715),
        64410 => array(1713),
        64411 => array(1713),
        64412 => array(1713),
        64413 => array(1713),
        64414 => array(1722),
        64415 => array(1722),
        64416 => array(1723),
        64417 => array(1723),
        64418 => array(1723),
        64419 => array(1723),
        64420 => array(1728),
        64421 => array(1728),
        64422 => array(1729),
        64423 => array(1729),
        64424 => array(1729),
        64425 => array(1729),
        64426 => array(1726),
        64427 => array(1726),
        64428 => array(1726),
        64429 => array(1726),
        64430 => array(1746),
        64431 => array(1746),
        64432 => array(1747),
        64433 => array(1747),
        64467 => array(1709),
        64468 => array(1709),
        64469 => array(1709),
        64470 => array(1709),
        64471 => array(1735),
        64472 => array(1735),
        64473 => array(1734),
        64474 => array(1734),
        64475 => array(1736),
        64476 => array(1736),
        64477 => array(1735, 1652),
        64478 => array(1739),
        64479 => array(1739),
        64480 => array(1733),
        64481 => array(1733),
        64482 => array(1737),
        64483 => array(1737),
        64484 => array(1744),
        64485 => array(1744),
        64486 => array(1744),
        64487 => array(1744),
        64488 => array(1609),
        64489 => array(1609),
        64490 => array(1574, 1575),
        64491 => array(1574, 1575),
        64492 => array(1574, 1749),
        64493 => array(1574, 1749),
        64494 => array(1574, 1608),
        64495 => array(1574, 1608),
        64496 => array(1574, 1735),
        64497 => array(1574, 1735),
        64498 => array(1574, 1734),
        64499 => array(1574, 1734),
        64500 => array(1574, 1736),
        64501 => array(1574, 1736),
        64502 => array(1574, 1744),
        64503 => array(1574, 1744),
        64504 => array(1574, 1744),
        64505 => array(1574, 1609),
        64506 => array(1574, 1609),
        64507 => array(1574, 1609),
        64508 => array(1740),
        64509 => array(1740),
        64510 => array(1740),
        64511 => array(1740),
        64512 => array(1574, 1580),
        64513 => array(1574, 1581),
        64514 => array(1574, 1605),
        64515 => array(1574, 1609),
        64516 => array(1574, 1610),
        64517 => array(1576, 1580),
        64518 => array(1576, 1581),
        64519 => array(1576, 1582),
        64520 => array(1576, 1605),
        64521 => array(1576, 1609),
        64522 => array(1576, 1610),
        64523 => array(1578, 1580),
        64524 => array(1578, 1581),
        64525 => array(1578, 1582),
        64526 => array(1578, 1605),
        64527 => array(1578, 1609),
        64528 => array(1578, 1610),
        64529 => array(1579, 1580),
        64530 => array(1579, 1605),
        64531 => array(1579, 1609),
        64532 => array(1579, 1610),
        64533 => array(1580, 1581),
        64534 => array(1580, 1605),
        64535 => array(1581, 1580),
        64536 => array(1581, 1605),
        64537 => array(1582, 1580),
        64538 => array(1582, 1581),
        64539 => array(1582, 1605),
        64540 => array(1587, 1580),
        64541 => array(1587, 1581),
        64542 => array(1587, 1582),
        64543 => array(1587, 1605),
        64544 => array(1589, 1581),
        64545 => array(1589, 1605),
        64546 => array(1590, 1580),
        64547 => array(1590, 1581),
        64548 => array(1590, 1582),
        64549 => array(1590, 1605),
        64550 => array(1591, 1581),
        64551 => array(1591, 1605),
        64552 => array(1592, 1605),
        64553 => array(1593, 1580),
        64554 => array(1593, 1605),
        64555 => array(1594, 1580),
        64556 => array(1594, 1605),
        64557 => array(1601, 1580),
        64558 => array(1601, 1581),
        64559 => array(1601, 1582),
        64560 => array(1601, 1605),
        64561 => array(1601, 1609),
        64562 => array(1601, 1610),
        64563 => array(1602, 1581),
        64564 => array(1602, 1605),
        64565 => array(1602, 1609),
        64566 => array(1602, 1610),
        64567 => array(1603, 1575),
        64568 => array(1603, 1580),
        64569 => array(1603, 1581),
        64570 => array(1603, 1582),
        64571 => array(1603, 1604),
        64572 => array(1603, 1605),
        64573 => array(1603, 1609),
        64574 => array(1603, 1610),
        64575 => array(1604, 1580),
        64576 => array(1604, 1581),
        64577 => array(1604, 1582),
        64578 => array(1604, 1605),
        64579 => array(1604, 1609),
        64580 => array(1604, 1610),
        64581 => array(1605, 1580),
        64582 => array(1605, 1581),
        64583 => array(1605, 1582),
        64584 => array(1605, 1605),
        64585 => array(1605, 1609),
        64586 => array(1605, 1610),
        64587 => array(1606, 1580),
        64588 => array(1606, 1581),
        64589 => array(1606, 1582),
        64590 => array(1606, 1605),
        64591 => array(1606, 1609),
        64592 => array(1606, 1610),
        64593 => array(1607, 1580),
        64594 => array(1607, 1605),
        64595 => array(1607, 1609),
        64596 => array(1607, 1610),
        64597 => array(1610, 1580),
        64598 => array(1610, 1581),
        64599 => array(1610, 1582),
        64600 => array(1610, 1605),
        64601 => array(1610, 1609),
        64602 => array(1610, 1610),
        64603 => array(1584, 1648),
        64604 => array(1585, 1648),
        64605 => array(1609, 1648),
        64612 => array(1574, 1585),
        64613 => array(1574, 1586),
        64614 => array(1574, 1605),
        64615 => array(1574, 1606),
        64616 => array(1574, 1609),
        64617 => array(1574, 1610),
        64618 => array(1576, 1585),
        64619 => array(1576, 1586),
        64620 => array(1576, 1605),
        64621 => array(1576, 1606),
        64622 => array(1576, 1609),
        64623 => array(1576, 1610),
        64624 => array(1578, 1585),
        64625 => array(1578, 1586),
        64626 => array(1578, 1605),
        64627 => array(1578, 1606),
        64628 => array(1578, 1609),
        64629 => array(1578, 1610),
        64630 => array(1579, 1585),
        64631 => array(1579, 1586),
        64632 => array(1579, 1605),
        64633 => array(1579, 1606),
        64634 => array(1579, 1609),
        64635 => array(1579, 1610),
        64636 => array(1601, 1609),
        64637 => array(1601, 1610),
        64638 => array(1602, 1609),
        64639 => array(1602, 1610),
        64640 => array(1603, 1575),
        64641 => array(1603, 1604),
        64642 => array(1603, 1605),
        64643 => array(1603, 1609),
        64644 => array(1603, 1610),
        64645 => array(1604, 1605),
        64646 => array(1604, 1609),
        64647 => array(1604, 1610),
        64648 => array(1605, 1575),
        64649 => array(1605, 1605),
        64650 => array(1606, 1585),
        64651 => array(1606, 1586),
        64652 => array(1606, 1605),
        64653 => array(1606, 1606),
        64654 => array(1606, 1609),
        64655 => array(1606, 1610),
        64656 => array(1609, 1648),
        64657 => array(1610, 1585),
        64658 => array(1610, 1586),
        64659 => array(1610, 1605),
        64660 => array(1610, 1606),
        64661 => array(1610, 1609),
        64662 => array(1610, 1610),
        64663 => array(1574, 1580),
        64664 => array(1574, 1581),
        64665 => array(1574, 1582),
        64666 => array(1574, 1605),
        64667 => array(1574, 1607),
        64668 => array(1576, 1580),
        64669 => array(1576, 1581),
        64670 => array(1576, 1582),
        64671 => array(1576, 1605),
        64672 => array(1576, 1607),
        64673 => array(1578, 1580),
        64674 => array(1578, 1581),
        64675 => array(1578, 1582),
        64676 => array(1578, 1605),
        64677 => array(1578, 1607),
        64678 => array(1579, 1605),
        64679 => array(1580, 1581),
        64680 => array(1580, 1605),
        64681 => array(1581, 1580),
        64682 => array(1581, 1605),
        64683 => array(1582, 1580),
        64684 => array(1582, 1605),
        64685 => array(1587, 1580),
        64686 => array(1587, 1581),
        64687 => array(1587, 1582),
        64688 => array(1587, 1605),
        64689 => array(1589, 1581),
        64690 => array(1589, 1582),
        64691 => array(1589, 1605),
        64692 => array(1590, 1580),
        64693 => array(1590, 1581),
        64694 => array(1590, 1582),
        64695 => array(1590, 1605),
        64696 => array(1591, 1581),
        64697 => array(1592, 1605),
        64698 => array(1593, 1580),
        64699 => array(1593, 1605),
        64700 => array(1594, 1580),
        64701 => array(1594, 1605),
        64702 => array(1601, 1580),
        64703 => array(1601, 1581),
        64704 => array(1601, 1582),
        64705 => array(1601, 1605),
        64706 => array(1602, 1581),
        64707 => array(1602, 1605),
        64708 => array(1603, 1580),
        64709 => array(1603, 1581),
        64710 => array(1603, 1582),
        64711 => array(1603, 1604),
        64712 => array(1603, 1605),
        64713 => array(1604, 1580),
        64714 => array(1604, 1581),
        64715 => array(1604, 1582),
        64716 => array(1604, 1605),
        64717 => array(1604, 1607),
        64718 => array(1605, 1580),
        64719 => array(1605, 1581),
        64720 => array(1605, 1582),
        64721 => array(1605, 1605),
        64722 => array(1606, 1580),
        64723 => array(1606, 1581),
        64724 => array(1606, 1582),
        64725 => array(1606, 1605),
        64726 => array(1606, 1607),
        64727 => array(1607, 1580),
        64728 => array(1607, 1605),
        64729 => array(1607, 1648),
        64730 => array(1610, 1580),
        64731 => array(1610, 1581),
        64732 => array(1610, 1582),
        64733 => array(1610, 1605),
        64734 => array(1610, 1607),
        64735 => array(1574, 1605),
        64736 => array(1574, 1607),
        64737 => array(1576, 1605),
        64738 => array(1576, 1607),
        64739 => array(1578, 1605),
        64740 => array(1578, 1607),
        64741 => array(1579, 1605),
        64742 => array(1579, 1607),
        64743 => array(1587, 1605),
        64744 => array(1587, 1607),
        64745 => array(1588, 1605),
        64746 => array(1588, 1607),
        64747 => array(1603, 1604),
        64748 => array(1603, 1605),
        64749 => array(1604, 1605),
        64750 => array(1606, 1605),
        64751 => array(1606, 1607),
        64752 => array(1610, 1605),
        64753 => array(1610, 1607),
        64754 => array(1600, 1614, 1617),
        64755 => array(1600, 1615, 1617),
        64756 => array(1600, 1616, 1617),
        64757 => array(1591, 1609),
        64758 => array(1591, 1610),
        64759 => array(1593, 1609),
        64760 => array(1593, 1610),
        64761 => array(1594, 1609),
        64762 => array(1594, 1610),
        64763 => array(1587, 1609),
        64764 => array(1587, 1610),
        64765 => array(1588, 1609),
        64766 => array(1588, 1610),
        64767 => array(1581, 1609),
        64768 => array(1581, 1610),
        64769 => array(1580, 1609),
        64770 => array(1580, 1610),
        64771 => array(1582, 1609),
        64772 => array(1582, 1610),
        64773 => array(1589, 1609),
        64774 => array(1589, 1610),
        64775 => array(1590, 1609),
        64776 => array(1590, 1610),
        64777 => array(1588, 1580),
        64778 => array(1588, 1581),
        64779 => array(1588, 1582),
        64780 => array(1588, 1605),
        64781 => array(1588, 1585),
        64782 => array(1587, 1585),
        64783 => array(1589, 1585),
        64784 => array(1590, 1585),
        64785 => array(1591, 1609),
        64786 => array(1591, 1610),
        64787 => array(1593, 1609),
        64788 => array(1593, 1610),
        64789 => array(1594, 1609),
        64790 => array(1594, 1610),
        64791 => array(1587, 1609),
        64792 => array(1587, 1610),
        64793 => array(1588, 1609),
        64794 => array(1588, 1610),
        64795 => array(1581, 1609),
        64796 => array(1581, 1610),
        64797 => array(1580, 1609),
        64798 => array(1580, 1610),
        64799 => array(1582, 1609),
        64800 => array(1582, 1610),
        64801 => array(1589, 1609),
        64802 => array(1589, 1610),
        64803 => array(1590, 1609),
        64804 => array(1590, 1610),
        64805 => array(1588, 1580),
        64806 => array(1588, 1581),
        64807 => array(1588, 1582),
        64808 => array(1588, 1605),
        64809 => array(1588, 1585),
        64810 => array(1587, 1585),
        64811 => array(1589, 1585),
        64812 => array(1590, 1585),
        64813 => array(1588, 1580),
        64814 => array(1588, 1581),
        64815 => array(1588, 1582),
        64816 => array(1588, 1605),
        64817 => array(1587, 1607),
        64818 => array(1588, 1607),
        64819 => array(1591, 1605),
        64820 => array(1587, 1580),
        64821 => array(1587, 1581),
        64822 => array(1587, 1582),
        64823 => array(1588, 1580),
        64824 => array(1588, 1581),
        64825 => array(1588, 1582),
        64826 => array(1591, 1605),
        64827 => array(1592, 1605),
        64828 => array(1575, 1611),
        64829 => array(1575, 1611),
        64848 => array(1578, 1580, 1605),
        64849 => array(1578, 1581, 1580),
        64850 => array(1578, 1581, 1580),
        64851 => array(1578, 1581, 1605),
        64852 => array(1578, 1582, 1605),
        64853 => array(1578, 1605, 1580),
        64854 => array(1578, 1605, 1581),
        64855 => array(1578, 1605, 1582),
        64856 => array(1580, 1605, 1581),
        64857 => array(1580, 1605, 1581),
        64858 => array(1581, 1605, 1610),
        64859 => array(1581, 1605, 1609),
        64860 => array(1587, 1581, 1580),
        64861 => array(1587, 1580, 1581),
        64862 => array(1587, 1580, 1609),
        64863 => array(1587, 1605, 1581),
        64864 => array(1587, 1605, 1581),
        64865 => array(1587, 1605, 1580),
        64866 => array(1587, 1605, 1605),
        64867 => array(1587, 1605, 1605),
        64868 => array(1589, 1581, 1581),
        64869 => array(1589, 1581, 1581),
        64870 => array(1589, 1605, 1605),
        64871 => array(1588, 1581, 1605),
        64872 => array(1588, 1581, 1605),
        64873 => array(1588, 1580, 1610),
        64874 => array(1588, 1605, 1582),
        64875 => array(1588, 1605, 1582),
        64876 => array(1588, 1605, 1605),
        64877 => array(1588, 1605, 1605),
        64878 => array(1590, 1581, 1609),
        64879 => array(1590, 1582, 1605),
        64880 => array(1590, 1582, 1605),
        64881 => array(1591, 1605, 1581),
        64882 => array(1591, 1605, 1581),
        64883 => array(1591, 1605, 1605),
        64884 => array(1591, 1605, 1610),
        64885 => array(1593, 1580, 1605),
        64886 => array(1593, 1605, 1605),
        64887 => array(1593, 1605, 1605),
        64888 => array(1593, 1605, 1609),
        64889 => array(1594, 1605, 1605),
        64890 => array(1594, 1605, 1610),
        64891 => array(1594, 1605, 1609),
        64892 => array(1601, 1582, 1605),
        64893 => array(1601, 1582, 1605),
        64894 => array(1602, 1605, 1581),
        64895 => array(1602, 1605, 1605),
        64896 => array(1604, 1581, 1605),
        64897 => array(1604, 1581, 1610),
        64898 => array(1604, 1581, 1609),
        64899 => array(1604, 1580, 1580),
        64900 => array(1604, 1580, 1580),
        64901 => array(1604, 1582, 1605),
        64902 => array(1604, 1582, 1605),
        64903 => array(1604, 1605, 1581),
        64904 => array(1604, 1605, 1581),
        64905 => array(1605, 1581, 1580),
        64906 => array(1605, 1581, 1605),
        64907 => array(1605, 1581, 1610),
        64908 => array(1605, 1580, 1581),
        64909 => array(1605, 1580, 1605),
        64910 => array(1605, 1582, 1580),
        64911 => array(1605, 1582, 1605),
        64914 => array(1605, 1580, 1582),
        64915 => array(1607, 1605, 1580),
        64916 => array(1607, 1605, 1605),
        64917 => array(1606, 1581, 1605),
        64918 => array(1606, 1581, 1609),
        64919 => array(1606, 1580, 1605),
        64920 => array(1606, 1580, 1605),
        64921 => array(1606, 1580, 1609),
        64922 => array(1606, 1605, 1610),
        64923 => array(1606, 1605, 1609),
        64924 => array(1610, 1605, 1605),
        64925 => array(1610, 1605, 1605),
        64926 => array(1576, 1582, 1610),
        64927 => array(1578, 1580, 1610),
        64928 => array(1578, 1580, 1609),
        64929 => array(1578, 1582, 1610),
        64930 => array(1578, 1582, 1609),
        64931 => array(1578, 1605, 1610),
        64932 => array(1578, 1605, 1609),
        64933 => array(1580, 1605, 1610),
        64934 => array(1580, 1581, 1609),
        64935 => array(1580, 1605, 1609),
        64936 => array(1587, 1582, 1609),
        64937 => array(1589, 1581, 1610),
        64938 => array(1588, 1581, 1610),
        64939 => array(1590, 1581, 1610),
        64940 => array(1604, 1580, 1610),
        64941 => array(1604, 1605, 1610),
        64942 => array(1610, 1581, 1610),
        64943 => array(1610, 1580, 1610),
        64944 => array(1610, 1605, 1610),
        64945 => array(1605, 1605, 1610),
        64946 => array(1602, 1605, 1610),
        64947 => array(1606, 1581, 1610),
        64948 => array(1602, 1605, 1581),
        64949 => array(1604, 1581, 1605),
        64950 => array(1593, 1605, 1610),
        64951 => array(1603, 1605, 1610),
        64952 => array(1606, 1580, 1581),
        64953 => array(1605, 1582, 1610),
        64954 => array(1604, 1580, 1605),
        64955 => array(1603, 1605, 1605),
        64956 => array(1604, 1580, 1605),
        64957 => array(1606, 1580, 1581),
        64958 => array(1580, 1581, 1610),
        64959 => array(1581, 1580, 1610),
        64960 => array(1605, 1580, 1610),
        64961 => array(1601, 1605, 1610),
        64962 => array(1576, 1581, 1610),
        64963 => array(1603, 1605, 1605),
        64964 => array(1593, 1580, 1605),
        64965 => array(1589, 1605, 1605),
        64966 => array(1587, 1582, 1610),
        64967 => array(1606, 1580, 1610),
        65008 => array(1589, 1604, 1746),
        65009 => array(1602, 1604, 1746),
        65010 => array(1575, 1604, 1604, 1607),
        65011 => array(1575, 1603, 1576, 1585),
        65012 => array(1605, 1581, 1605, 1583),
        65013 => array(1589, 1604, 1593, 1605),
        65014 => array(1585, 1587, 1608, 1604),
        65015 => array(1593, 1604, 1610, 1607),
        65016 => array(1608, 1587, 1604, 1605),
        65017 => array(1589, 1604, 1609),
        65020 => array(1585, 1740, 1575, 1604),
        65041 => array(12289),
        65047 => array(12310),
        65048 => array(12311),
        65073 => array(8212),
        65074 => array(8211),
        65081 => array(12308),
        65082 => array(12309),
        65083 => array(12304),
        65084 => array(12305),
        65085 => array(12298),
        65086 => array(12299),
        65087 => array(12296),
        65088 => array(12297),
        65089 => array(12300),
        65090 => array(12301),
        65091 => array(12302),
        65092 => array(12303),
        65105 => array(12289),
        65112 => array(8212),
        65117 => array(12308),
        65118 => array(12309),
        65123 => array(45),
        65137 => array(1600, 1611),
        65143 => array(1600, 1614),
        65145 => array(1600, 1615),
        65147 => array(1600, 1616),
        65149 => array(1600, 1617),
        65151 => array(1600, 1618),
        65152 => array(1569),
        65153 => array(1570),
        65154 => array(1570),
        65155 => array(1571),
        65156 => array(1571),
        65157 => array(1572),
        65158 => array(1572),
        65159 => array(1573),
        65160 => array(1573),
        65161 => array(1574),
        65162 => array(1574),
        65163 => array(1574),
        65164 => array(1574),
        65165 => array(1575),
        65166 => array(1575),
        65167 => array(1576),
        65168 => array(1576),
        65169 => array(1576),
        65170 => array(1576),
        65171 => array(1577),
        65172 => array(1577),
        65173 => array(1578),
        65174 => array(1578),
        65175 => array(1578),
        65176 => array(1578),
        65177 => array(1579),
        65178 => array(1579),
        65179 => array(1579),
        65180 => array(1579),
        65181 => array(1580),
        65182 => array(1580),
        65183 => array(1580),
        65184 => array(1580),
        65185 => array(1581),
        65186 => array(1581),
        65187 => array(1581),
        65188 => array(1581),
        65189 => array(1582),
        65190 => array(1582),
        65191 => array(1582),
        65192 => array(1582),
        65193 => array(1583),
        65194 => array(1583),
        65195 => array(1584),
        65196 => array(1584),
        65197 => array(1585),
        65198 => array(1585),
        65199 => array(1586),
        65200 => array(1586),
        65201 => array(1587),
        65202 => array(1587),
        65203 => array(1587),
        65204 => array(1587),
        65205 => array(1588),
        65206 => array(1588),
        65207 => array(1588),
        65208 => array(1588),
        65209 => array(1589),
        65210 => array(1589),
        65211 => array(1589),
        65212 => array(1589),
        65213 => array(1590),
        65214 => array(1590),
        65215 => array(1590),
        65216 => array(1590),
        65217 => array(1591),
        65218 => array(1591),
        65219 => array(1591),
        65220 => array(1591),
        65221 => array(1592),
        65222 => array(1592),
        65223 => array(1592),
        65224 => array(1592),
        65225 => array(1593),
        65226 => array(1593),
        65227 => array(1593),
        65228 => array(1593),
        65229 => array(1594),
        65230 => array(1594),
        65231 => array(1594),
        65232 => array(1594),
        65233 => array(1601),
        65234 => array(1601),
        65235 => array(1601),
        65236 => array(1601),
        65237 => array(1602),
        65238 => array(1602),
        65239 => array(1602),
        65240 => array(1602),
        65241 => array(1603),
        65242 => array(1603),
        65243 => array(1603),
        65244 => array(1603),
        65245 => array(1604),
        65246 => array(1604),
        65247 => array(1604),
        65248 => array(1604),
        65249 => array(1605),
        65250 => array(1605),
        65251 => array(1605),
        65252 => array(1605),
        65253 => array(1606),
        65254 => array(1606),
        65255 => array(1606),
        65256 => array(1606),
        65257 => array(1607),
        65258 => array(1607),
        65259 => array(1607),
        65260 => array(1607),
        65261 => array(1608),
        65262 => array(1608),
        65263 => array(1609),
        65264 => array(1609),
        65265 => array(1610),
        65266 => array(1610),
        65267 => array(1610),
        65268 => array(1610),
        65269 => array(1604, 1570),
        65270 => array(1604, 1570),
        65271 => array(1604, 1571),
        65272 => array(1604, 1571),
        65273 => array(1604, 1573),
        65274 => array(1604, 1573),
        65275 => array(1604, 1575),
        65276 => array(1604, 1575),
        65293 => array(45),
        65294 => array(46),
        65296 => array(48),
        65297 => array(49),
        65298 => array(50),
        65299 => array(51),
        65300 => array(52),
        65301 => array(53),
        65302 => array(54),
        65303 => array(55),
        65304 => array(56),
        65305 => array(57),
        65313 => array(97),
        65314 => array(98),
        65315 => array(99),
        65316 => array(100),
        65317 => array(101),
        65318 => array(102),
        65319 => array(103),
        65320 => array(104),
        65321 => array(105),
        65322 => array(106),
        65323 => array(107),
        65324 => array(108),
        65325 => array(109),
        65326 => array(110),
        65327 => array(111),
        65328 => array(112),
        65329 => array(113),
        65330 => array(114),
        65331 => array(115),
        65332 => array(116),
        65333 => array(117),
        65334 => array(118),
        65335 => array(119),
        65336 => array(120),
        65337 => array(121),
        65338 => array(122),
        65345 => array(97),
        65346 => array(98),
        65347 => array(99),
        65348 => array(100),
        65349 => array(101),
        65350 => array(102),
        65351 => array(103),
        65352 => array(104),
        65353 => array(105),
        65354 => array(106),
        65355 => array(107),
        65356 => array(108),
        65357 => array(109),
        65358 => array(110),
        65359 => array(111),
        65360 => array(112),
        65361 => array(113),
        65362 => array(114),
        65363 => array(115),
        65364 => array(116),
        65365 => array(117),
        65366 => array(118),
        65367 => array(119),
        65368 => array(120),
        65369 => array(121),
        65370 => array(122),
        65375 => array(10629),
        65376 => array(10630),
        65377 => array(46),
        65378 => array(12300),
        65379 => array(12301),
        65380 => array(12289),
        65381 => array(12539),
        65382 => array(12530),
        65383 => array(12449),
        65384 => array(12451),
        65385 => array(12453),
        65386 => array(12455),
        65387 => array(12457),
        65388 => array(12515),
        65389 => array(12517),
        65390 => array(12519),
        65391 => array(12483),
        65392 => array(12540),
        65393 => array(12450),
        65394 => array(12452),
        65395 => array(12454),
        65396 => array(12456),
        65397 => array(12458),
        65398 => array(12459),
        65399 => array(12461),
        65400 => array(12463),
        65401 => array(12465),
        65402 => array(12467),
        65403 => array(12469),
        65404 => array(12471),
        65405 => array(12473),
        65406 => array(12475),
        65407 => array(12477),
        65408 => array(12479),
        65409 => array(12481),
        65410 => array(12484),
        65411 => array(12486),
        65412 => array(12488),
        65413 => array(12490),
        65414 => array(12491),
        65415 => array(12492),
        65416 => array(12493),
        65417 => array(12494),
        65418 => array(12495),
        65419 => array(12498),
        65420 => array(12501),
        65421 => array(12504),
        65422 => array(12507),
        65423 => array(12510),
        65424 => array(12511),
        65425 => array(12512),
        65426 => array(12513),
        65427 => array(12514),
        65428 => array(12516),
        65429 => array(12518),
        65430 => array(12520),
        65431 => array(12521),
        65432 => array(12522),
        65433 => array(12523),
        65434 => array(12524),
        65435 => array(12525),
        65436 => array(12527),
        65437 => array(12531),
        65438 => array(12441),
        65439 => array(12442),
        65441 => array(4352),
        65442 => array(4353),
        65443 => array(4522),
        65444 => array(4354),
        65445 => array(4524),
        65446 => array(4525),
        65447 => array(4355),
        65448 => array(4356),
        65449 => array(4357),
        65450 => array(4528),
        65451 => array(4529),
        65452 => array(4530),
        65453 => array(4531),
        65454 => array(4532),
        65455 => array(4533),
        65456 => array(4378),
        65457 => array(4358),
        65458 => array(4359),
        65459 => array(4360),
        65460 => array(4385),
        65461 => array(4361),
        65462 => array(4362),
        65463 => array(4363),
        65464 => array(4364),
        65465 => array(4365),
        65466 => array(4366),
        65467 => array(4367),
        65468 => array(4368),
        65469 => array(4369),
        65470 => array(4370),
        65474 => array(4449),
        65475 => array(4450),
        65476 => array(4451),
        65477 => array(4452),
        65478 => array(4453),
        65479 => array(4454),
        65482 => array(4455),
        65483 => array(4456),
        65484 => array(4457),
        65485 => array(4458),
        65486 => array(4459),
        65487 => array(4460),
        65490 => array(4461),
        65491 => array(4462),
        65492 => array(4463),
        65493 => array(4464),
        65494 => array(4465),
        65495 => array(4466),
        65498 => array(4467),
        65499 => array(4468),
        65500 => array(4469),
        65504 => array(162),
        65505 => array(163),
        65506 => array(172),
        65508 => array(166),
        65509 => array(165),
        65510 => array(8361),
        65512 => array(9474),
        65513 => array(8592),
        65514 => array(8593),
        65515 => array(8594),
        65516 => array(8595),
        65517 => array(9632),
        65518 => array(9675),
        66560 => array(66600),
        66561 => array(66601),
        66562 => array(66602),
        66563 => array(66603),
        66564 => array(66604),
        66565 => array(66605),
        66566 => array(66606),
        66567 => array(66607),
        66568 => array(66608),
        66569 => array(66609),
        66570 => array(66610),
        66571 => array(66611),
        66572 => array(66612),
        66573 => array(66613),
        66574 => array(66614),
        66575 => array(66615),
        66576 => array(66616),
        66577 => array(66617),
        66578 => array(66618),
        66579 => array(66619),
        66580 => array(66620),
        66581 => array(66621),
        66582 => array(66622),
        66583 => array(66623),
        66584 => array(66624),
        66585 => array(66625),
        66586 => array(66626),
        66587 => array(66627),
        66588 => array(66628),
        66589 => array(66629),
        66590 => array(66630),
        66591 => array(66631),
        66592 => array(66632),
        66593 => array(66633),
        66594 => array(66634),
        66595 => array(66635),
        66596 => array(66636),
        66597 => array(66637),
        66598 => array(66638),
        66599 => array(66639),
        66736 => array(66776),
        66737 => array(66777),
        66738 => array(66778),
        66739 => array(66779),
        66740 => array(66780),
        66741 => array(66781),
        66742 => array(66782),
        66743 => array(66783),
        66744 => array(66784),
        66745 => array(66785),
        66746 => array(66786),
        66747 => array(66787),
        66748 => array(66788),
        66749 => array(66789),
        66750 => array(66790),
        66751 => array(66791),
        66752 => array(66792),
        66753 => array(66793),
        66754 => array(66794),
        66755 => array(66795),
        66756 => array(66796),
        66757 => array(66797),
        66758 => array(66798),
        66759 => array(66799),
        66760 => array(66800),
        66761 => array(66801),
        66762 => array(66802),
        66763 => array(66803),
        66764 => array(66804),
        66765 => array(66805),
        66766 => array(66806),
        66767 => array(66807),
        66768 => array(66808),
        66769 => array(66809),
        66770 => array(66810),
        66771 => array(66811),
        66928 => array(66967),
        66929 => array(66968),
        66930 => array(66969),
        66931 => array(66970),
        66932 => array(66971),
        66933 => array(66972),
        66934 => array(66973),
        66935 => array(66974),
        66936 => array(66975),
        66937 => array(66976),
        66938 => array(66977),
        66940 => array(66979),
        66941 => array(66980),
        66942 => array(66981),
        66943 => array(66982),
        66944 => array(66983),
        66945 => array(66984),
        66946 => array(66985),
        66947 => array(66986),
        66948 => array(66987),
        66949 => array(66988),
        66950 => array(66989),
        66951 => array(66990),
        66952 => array(66991),
        66953 => array(66992),
        66954 => array(66993),
        66956 => array(66995),
        66957 => array(66996),
        66958 => array(66997),
        66959 => array(66998),
        66960 => array(66999),
        66961 => array(67000),
        66962 => array(67001),
        66964 => array(67003),
        66965 => array(67004),
        67457 => array(720),
        67458 => array(721),
        67459 => array(230),
        67460 => array(665),
        67461 => array(595),
        67463 => array(675),
        67464 => array(43878),
        67465 => array(677),
        67466 => array(676),
        67467 => array(598),
        67468 => array(599),
        67469 => array(7569),
        67470 => array(600),
        67471 => array(606),
        67472 => array(681),
        67473 => array(612),
        67474 => array(610),
        67475 => array(608),
        67476 => array(667),
        67477 => array(295),
        67478 => array(668),
        67479 => array(615),
        67480 => array(644),
        67481 => array(682),
        67482 => array(683),
        67483 => array(620),
        67484 => array(122628),
        67485 => array(42894),
        67486 => array(622),
        67487 => array(122629),
        67488 => array(654),
        67489 => array(122630),
        67490 => array(248),
        67491 => array(630),
        67492 => array(631),
        67493 => array(113),
        67494 => array(634),
        67495 => array(122632),
        67496 => array(637),
        67497 => array(638),
        67498 => array(640),
        67499 => array(680),
        67500 => array(678),
        67501 => array(43879),
        67502 => array(679),
        67503 => array(648),
        67504 => array(11377),
        67506 => array(655),
        67507 => array(673),
        67508 => array(674),
        67509 => array(664),
        67510 => array(448),
        67511 => array(449),
        67512 => array(450),
        67513 => array(122634),
        67514 => array(122654),
        68736 => array(68800),
        68737 => array(68801),
        68738 => array(68802),
        68739 => array(68803),
        68740 => array(68804),
        68741 => array(68805),
        68742 => array(68806),
        68743 => array(68807),
        68744 => array(68808),
        68745 => array(68809),
        68746 => array(68810),
        68747 => array(68811),
        68748 => array(68812),
        68749 => array(68813),
        68750 => array(68814),
        68751 => array(68815),
        68752 => array(68816),
        68753 => array(68817),
        68754 => array(68818),
        68755 => array(68819),
        68756 => array(68820),
        68757 => array(68821),
        68758 => array(68822),
        68759 => array(68823),
        68760 => array(68824),
        68761 => array(68825),
        68762 => array(68826),
        68763 => array(68827),
        68764 => array(68828),
        68765 => array(68829),
        68766 => array(68830),
        68767 => array(68831),
        68768 => array(68832),
        68769 => array(68833),
        68770 => array(68834),
        68771 => array(68835),
        68772 => array(68836),
        68773 => array(68837),
        68774 => array(68838),
        68775 => array(68839),
        68776 => array(68840),
        68777 => array(68841),
        68778 => array(68842),
        68779 => array(68843),
        68780 => array(68844),
        68781 => array(68845),
        68782 => array(68846),
        68783 => array(68847),
        68784 => array(68848),
        68785 => array(68849),
        68786 => array(68850),
        71840 => array(71872),
        71841 => array(71873),
        71842 => array(71874),
        71843 => array(71875),
        71844 => array(71876),
        71845 => array(71877),
        71846 => array(71878),
        71847 => array(71879),
        71848 => array(71880),
        71849 => array(71881),
        71850 => array(71882),
        71851 => array(71883),
        71852 => array(71884),
        71853 => array(71885),
        71854 => array(71886),
        71855 => array(71887),
        71856 => array(71888),
        71857 => array(71889),
        71858 => array(71890),
        71859 => array(71891),
        71860 => array(71892),
        71861 => array(71893),
        71862 => array(71894),
        71863 => array(71895),
        71864 => array(71896),
        71865 => array(71897),
        71866 => array(71898),
        71867 => array(71899),
        71868 => array(71900),
        71869 => array(71901),
        71870 => array(71902),
        71871 => array(71903),
        93760 => array(93792),
        93761 => array(93793),
        93762 => array(93794),
        93763 => array(93795),
        93764 => array(93796),
        93765 => array(93797),
        93766 => array(93798),
        93767 => array(93799),
        93768 => array(93800),
        93769 => array(93801),
        93770 => array(93802),
        93771 => array(93803),
        93772 => array(93804),
        93773 => array(93805),
        93774 => array(93806),
        93775 => array(93807),
        93776 => array(93808),
        93777 => array(93809),
        93778 => array(93810),
        93779 => array(93811),
        93780 => array(93812),
        93781 => array(93813),
        93782 => array(93814),
        93783 => array(93815),
        93784 => array(93816),
        93785 => array(93817),
        93786 => array(93818),
        93787 => array(93819),
        93788 => array(93820),
        93789 => array(93821),
        93790 => array(93822),
        93791 => array(93823),
        119134 => array(119127, 119141),
        119135 => array(119128, 119141),
        119136 => array(119128, 119141, 119150),
        119137 => array(119128, 119141, 119151),
        119138 => array(119128, 119141, 119152),
        119139 => array(119128, 119141, 119153),
        119140 => array(119128, 119141, 119154),
        119227 => array(119225, 119141),
        119228 => array(119226, 119141),
        119229 => array(119225, 119141, 119150),
        119230 => array(119226, 119141, 119150),
        119231 => array(119225, 119141, 119151),
        119232 => array(119226, 119141, 119151),
        119808 => array(97),
        119809 => array(98),
        119810 => array(99),
        119811 => array(100),
        119812 => array(101),
        119813 => array(102),
        119814 => array(103),
        119815 => array(104),
        119816 => array(105),
        119817 => array(106),
        119818 => array(107),
        119819 => array(108),
        119820 => array(109),
        119821 => array(110),
        119822 => array(111),
        119823 => array(112),
        119824 => array(113),
        119825 => array(114),
        119826 => array(115),
        119827 => array(116),
        119828 => array(117),
        119829 => array(118),
        119830 => array(119),
        119831 => array(120),
        119832 => array(121),
        119833 => array(122),
        119834 => array(97),
        119835 => array(98),
        119836 => array(99),
        119837 => array(100),
        119838 => array(101),
        119839 => array(102),
        119840 => array(103),
        119841 => array(104),
        119842 => array(105),
        119843 => array(106),
        119844 => array(107),
        119845 => array(108),
        119846 => array(109),
        119847 => array(110),
        119848 => array(111),
        119849 => array(112),
        119850 => array(113),
        119851 => array(114),
        119852 => array(115),
        119853 => array(116),
        119854 => array(117),
        119855 => array(118),
        119856 => array(119),
        119857 => array(120),
        119858 => array(121),
        119859 => array(122),
        119860 => array(97),
        119861 => array(98),
        119862 => array(99),
        119863 => array(100),
        119864 => array(101),
        119865 => array(102),
        119866 => array(103),
        119867 => array(104),
        119868 => array(105),
        119869 => array(106),
        119870 => array(107),
        119871 => array(108),
        119872 => array(109),
        119873 => array(110),
        119874 => array(111),
        119875 => array(112),
        119876 => array(113),
        119877 => array(114),
        119878 => array(115),
        119879 => array(116),
        119880 => array(117),
        119881 => array(118),
        119882 => array(119),
        119883 => array(120),
        119884 => array(121),
        119885 => array(122),
        119886 => array(97),
        119887 => array(98),
        119888 => array(99),
        119889 => array(100),
        119890 => array(101),
        119891 => array(102),
        119892 => array(103),
        119894 => array(105),
        119895 => array(106),
        119896 => array(107),
        119897 => array(108),
        119898 => array(109),
        119899 => array(110),
        119900 => array(111),
        119901 => array(112),
        119902 => array(113),
        119903 => array(114),
        119904 => array(115),
        119905 => array(116),
        119906 => array(117),
        119907 => array(118),
        119908 => array(119),
        119909 => array(120),
        119910 => array(121),
        119911 => array(122),
        119912 => array(97),
        119913 => array(98),
        119914 => array(99),
        119915 => array(100),
        119916 => array(101),
        119917 => array(102),
        119918 => array(103),
        119919 => array(104),
        119920 => array(105),
        119921 => array(106),
        119922 => array(107),
        119923 => array(108),
        119924 => array(109),
        119925 => array(110),
        119926 => array(111),
        119927 => array(112),
        119928 => array(113),
        119929 => array(114),
        119930 => array(115),
        119931 => array(116),
        119932 => array(117),
        119933 => array(118),
        119934 => array(119),
        119935 => array(120),
        119936 => array(121),
        119937 => array(122),
        119938 => array(97),
        119939 => array(98),
        119940 => array(99),
        119941 => array(100),
        119942 => array(101),
        119943 => array(102),
        119944 => array(103),
        119945 => array(104),
        119946 => array(105),
        119947 => array(106),
        119948 => array(107),
        119949 => array(108),
        119950 => array(109),
        119951 => array(110),
        119952 => array(111),
        119953 => array(112),
        119954 => array(113),
        119955 => array(114),
        119956 => array(115),
        119957 => array(116),
        119958 => array(117),
        119959 => array(118),
        119960 => array(119),
        119961 => array(120),
        119962 => array(121),
        119963 => array(122),
        119964 => array(97),
        119966 => array(99),
        119967 => array(100),
        119970 => array(103),
        119973 => array(106),
        119974 => array(107),
        119977 => array(110),
        119978 => array(111),
        119979 => array(112),
        119980 => array(113),
        119982 => array(115),
        119983 => array(116),
        119984 => array(117),
        119985 => array(118),
        119986 => array(119),
        119987 => array(120),
        119988 => array(121),
        119989 => array(122),
        119990 => array(97),
        119991 => array(98),
        119992 => array(99),
        119993 => array(100),
        119995 => array(102),
        119997 => array(104),
        119998 => array(105),
        119999 => array(106),
        120000 => array(107),
        120001 => array(108),
        120002 => array(109),
        120003 => array(110),
        120005 => array(112),
        120006 => array(113),
        120007 => array(114),
        120008 => array(115),
        120009 => array(116),
        120010 => array(117),
        120011 => array(118),
        120012 => array(119),
        120013 => array(120),
        120014 => array(121),
        120015 => array(122),
        120016 => array(97),
        120017 => array(98),
        120018 => array(99),
        120019 => array(100),
        120020 => array(101),
        120021 => array(102),
        120022 => array(103),
        120023 => array(104),
        120024 => array(105),
        120025 => array(106),
        120026 => array(107),
        120027 => array(108),
        120028 => array(109),
        120029 => array(110),
        120030 => array(111),
        120031 => array(112),
        120032 => array(113),
        120033 => array(114),
        120034 => array(115),
        120035 => array(116),
        120036 => array(117),
        120037 => array(118),
        120038 => array(119),
        120039 => array(120),
        120040 => array(121),
        120041 => array(122),
        120042 => array(97),
        120043 => array(98),
        120044 => array(99),
        120045 => array(100),
        120046 => array(101),
        120047 => array(102),
        120048 => array(103),
        120049 => array(104),
        120050 => array(105),
        120051 => array(106),
        120052 => array(107),
        120053 => array(108),
        120054 => array(109),
        120055 => array(110),
        120056 => array(111),
        120057 => array(112),
        120058 => array(113),
        120059 => array(114),
        120060 => array(115),
        120061 => array(116),
        120062 => array(117),
        120063 => array(118),
        120064 => array(119),
        120065 => array(120),
        120066 => array(121),
        120067 => array(122),
        120068 => array(97),
        120069 => array(98),
        120071 => array(100),
        120072 => array(101),
        120073 => array(102),
        120074 => array(103),
        120077 => array(106),
        120078 => array(107),
        120079 => array(108),
        120080 => array(109),
        120081 => array(110),
        120082 => array(111),
        120083 => array(112),
        120084 => array(113),
        120086 => array(115),
        120087 => array(116),
        120088 => array(117),
        120089 => array(118),
        120090 => array(119),
        120091 => array(120),
        120092 => array(121),
        120094 => array(97),
        120095 => array(98),
        120096 => array(99),
        120097 => array(100),
        120098 => array(101),
        120099 => array(102),
        120100 => array(103),
        120101 => array(104),
        120102 => array(105),
        120103 => array(106),
        120104 => array(107),
        120105 => array(108),
        120106 => array(109),
        120107 => array(110),
        120108 => array(111),
        120109 => array(112),
        120110 => array(113),
        120111 => array(114),
        120112 => array(115),
        120113 => array(116),
        120114 => array(117),
        120115 => array(118),
        120116 => array(119),
        120117 => array(120),
        120118 => array(121),
        120119 => array(122),
        120120 => array(97),
        120121 => array(98),
        120123 => array(100),
        120124 => array(101),
        120125 => array(102),
        120126 => array(103),
        120128 => array(105),
        120129 => array(106),
        120130 => array(107),
        120131 => array(108),
        120132 => array(109),
        120134 => array(111),
        120138 => array(115),
        120139 => array(116),
        120140 => array(117),
        120141 => array(118),
        120142 => array(119),
        120143 => array(120),
        120144 => array(121),
        120146 => array(97),
        120147 => array(98),
        120148 => array(99),
        120149 => array(100),
        120150 => array(101),
        120151 => array(102),
        120152 => array(103),
        120153 => array(104),
        120154 => array(105),
        120155 => array(106),
        120156 => array(107),
        120157 => array(108),
        120158 => array(109),
        120159 => array(110),
        120160 => array(111),
        120161 => array(112),
        120162 => array(113),
        120163 => array(114),
        120164 => array(115),
        120165 => array(116),
        120166 => array(117),
        120167 => array(118),
        120168 => array(119),
        120169 => array(120),
        120170 => array(121),
        120171 => array(122),
        120172 => array(97),
        120173 => array(98),
        120174 => array(99),
        120175 => array(100),
        120176 => array(101),
        120177 => array(102),
        120178 => array(103),
        120179 => array(104),
        120180 => array(105),
        120181 => array(106),
        120182 => array(107),
        120183 => array(108),
        120184 => array(109),
        120185 => array(110),
        120186 => array(111),
        120187 => array(112),
        120188 => array(113),
        120189 => array(114),
        120190 => array(115),
        120191 => array(116),
        120192 => array(117),
        120193 => array(118),
        120194 => array(119),
        120195 => array(120),
        120196 => array(121),
        120197 => array(122),
        120198 => array(97),
        120199 => array(98),
        120200 => array(99),
        120201 => array(100),
        120202 => array(101),
        120203 => array(102),
        120204 => array(103),
        120205 => array(104),
        120206 => array(105),
        120207 => array(106),
        120208 => array(107),
        120209 => array(108),
        120210 => array(109),
        120211 => array(110),
        120212 => array(111),
        120213 => array(112),
        120214 => array(113),
        120215 => array(114),
        120216 => array(115),
        120217 => array(116),
        120218 => array(117),
        120219 => array(118),
        120220 => array(119),
        120221 => array(120),
        120222 => array(121),
        120223 => array(122),
        120224 => array(97),
        120225 => array(98),
        120226 => array(99),
        120227 => array(100),
        120228 => array(101),
        120229 => array(102),
        120230 => array(103),
        120231 => array(104),
        120232 => array(105),
        120233 => array(106),
        120234 => array(107),
        120235 => array(108),
        120236 => array(109),
        120237 => array(110),
        120238 => array(111),
        120239 => array(112),
        120240 => array(113),
        120241 => array(114),
        120242 => array(115),
        120243 => array(116),
        120244 => array(117),
        120245 => array(118),
        120246 => array(119),
        120247 => array(120),
        120248 => array(121),
        120249 => array(122),
        120250 => array(97),
        120251 => array(98),
        120252 => array(99),
        120253 => array(100),
        120254 => array(101),
        120255 => array(102),
        120256 => array(103),
        120257 => array(104),
        120258 => array(105),
        120259 => array(106),
        120260 => array(107),
        120261 => array(108),
        120262 => array(109),
        120263 => array(110),
        120264 => array(111),
        120265 => array(112),
        120266 => array(113),
        120267 => array(114),
        120268 => array(115),
        120269 => array(116),
        120270 => array(117),
        120271 => array(118),
        120272 => array(119),
        120273 => array(120),
        120274 => array(121),
        120275 => array(122),
        120276 => array(97),
        120277 => array(98),
        120278 => array(99),
        120279 => array(100),
        120280 => array(101),
        120281 => array(102),
        120282 => array(103),
        120283 => array(104),
        120284 => array(105),
        120285 => array(106),
        120286 => array(107),
        120287 => array(108),
        120288 => array(109),
        120289 => array(110),
        120290 => array(111),
        120291 => array(112),
        120292 => array(113),
        120293 => array(114),
        120294 => array(115),
        120295 => array(116),
        120296 => array(117),
        120297 => array(118),
        120298 => array(119),
        120299 => array(120),
        120300 => array(121),
        120301 => array(122),
        120302 => array(97),
        120303 => array(98),
        120304 => array(99),
        120305 => array(100),
        120306 => array(101),
        120307 => array(102),
        120308 => array(103),
        120309 => array(104),
        120310 => array(105),
        120311 => array(106),
        120312 => array(107),
        120313 => array(108),
        120314 => array(109),
        120315 => array(110),
        120316 => array(111),
        120317 => array(112),
        120318 => array(113),
        120319 => array(114),
        120320 => array(115),
        120321 => array(116),
        120322 => array(117),
        120323 => array(118),
        120324 => array(119),
        120325 => array(120),
        120326 => array(121),
        120327 => array(122),
        120328 => array(97),
        120329 => array(98),
        120330 => array(99),
        120331 => array(100),
        120332 => array(101),
        120333 => array(102),
        120334 => array(103),
        120335 => array(104),
        120336 => array(105),
        120337 => array(106),
        120338 => array(107),
        120339 => array(108),
        120340 => array(109),
        120341 => array(110),
        120342 => array(111),
        120343 => array(112),
        120344 => array(113),
        120345 => array(114),
        120346 => array(115),
        120347 => array(116),
        120348 => array(117),
        120349 => array(118),
        120350 => array(119),
        120351 => array(120),
        120352 => array(121),
        120353 => array(122),
        120354 => array(97),
        120355 => array(98),
        120356 => array(99),
        120357 => array(100),
        120358 => array(101),
        120359 => array(102),
        120360 => array(103),
        120361 => array(104),
        120362 => array(105),
        120363 => array(106),
        120364 => array(107),
        120365 => array(108),
        120366 => array(109),
        120367 => array(110),
        120368 => array(111),
        120369 => array(112),
        120370 => array(113),
        120371 => array(114),
        120372 => array(115),
        120373 => array(116),
        120374 => array(117),
        120375 => array(118),
        120376 => array(119),
        120377 => array(120),
        120378 => array(121),
        120379 => array(122),
        120380 => array(97),
        120381 => array(98),
        120382 => array(99),
        120383 => array(100),
        120384 => array(101),
        120385 => array(102),
        120386 => array(103),
        120387 => array(104),
        120388 => array(105),
        120389 => array(106),
        120390 => array(107),
        120391 => array(108),
        120392 => array(109),
        120393 => array(110),
        120394 => array(111),
        120395 => array(112),
        120396 => array(113),
        120397 => array(114),
        120398 => array(115),
        120399 => array(116),
        120400 => array(117),
        120401 => array(118),
        120402 => array(119),
        120403 => array(120),
        120404 => array(121),
        120405 => array(122),
        120406 => array(97),
        120407 => array(98),
        120408 => array(99),
        120409 => array(100),
        120410 => array(101),
        120411 => array(102),
        120412 => array(103),
        120413 => array(104),
        120414 => array(105),
        120415 => array(106),
        120416 => array(107),
        120417 => array(108),
        120418 => array(109),
        120419 => array(110),
        120420 => array(111),
        120421 => array(112),
        120422 => array(113),
        120423 => array(114),
        120424 => array(115),
        120425 => array(116),
        120426 => array(117),
        120427 => array(118),
        120428 => array(119),
        120429 => array(120),
        120430 => array(121),
        120431 => array(122),
        120432 => array(97),
        120433 => array(98),
        120434 => array(99),
        120435 => array(100),
        120436 => array(101),
        120437 => array(102),
        120438 => array(103),
        120439 => array(104),
        120440 => array(105),
        120441 => array(106),
        120442 => array(107),
        120443 => array(108),
        120444 => array(109),
        120445 => array(110),
        120446 => array(111),
        120447 => array(112),
        120448 => array(113),
        120449 => array(114),
        120450 => array(115),
        120451 => array(116),
        120452 => array(117),
        120453 => array(118),
        120454 => array(119),
        120455 => array(120),
        120456 => array(121),
        120457 => array(122),
        120458 => array(97),
        120459 => array(98),
        120460 => array(99),
        120461 => array(100),
        120462 => array(101),
        120463 => array(102),
        120464 => array(103),
        120465 => array(104),
        120466 => array(105),
        120467 => array(106),
        120468 => array(107),
        120469 => array(108),
        120470 => array(109),
        120471 => array(110),
        120472 => array(111),
        120473 => array(112),
        120474 => array(113),
        120475 => array(114),
        120476 => array(115),
        120477 => array(116),
        120478 => array(117),
        120479 => array(118),
        120480 => array(119),
        120481 => array(120),
        120482 => array(121),
        120483 => array(122),
        120484 => array(305),
        120485 => array(567),
        120488 => array(945),
        120489 => array(946),
        120490 => array(947),
        120491 => array(948),
        120492 => array(949),
        120493 => array(950),
        120494 => array(951),
        120495 => array(952),
        120496 => array(953),
        120497 => array(954),
        120498 => array(955),
        120499 => array(956),
        120500 => array(957),
        120501 => array(958),
        120502 => array(959),
        120503 => array(960),
        120504 => array(961),
        120505 => array(952),
        120506 => array(963),
        120507 => array(964),
        120508 => array(965),
        120509 => array(966),
        120510 => array(967),
        120511 => array(968),
        120512 => array(969),
        120513 => array(8711),
        120514 => array(945),
        120515 => array(946),
        120516 => array(947),
        120517 => array(948),
        120518 => array(949),
        120519 => array(950),
        120520 => array(951),
        120521 => array(952),
        120522 => array(953),
        120523 => array(954),
        120524 => array(955),
        120525 => array(956),
        120526 => array(957),
        120527 => array(958),
        120528 => array(959),
        120529 => array(960),
        120530 => array(961),
        120531 => array(963),
        120532 => array(963),
        120533 => array(964),
        120534 => array(965),
        120535 => array(966),
        120536 => array(967),
        120537 => array(968),
        120538 => array(969),
        120539 => array(8706),
        120540 => array(949),
        120541 => array(952),
        120542 => array(954),
        120543 => array(966),
        120544 => array(961),
        120545 => array(960),
        120546 => array(945),
        120547 => array(946),
        120548 => array(947),
        120549 => array(948),
        120550 => array(949),
        120551 => array(950),
        120552 => array(951),
        120553 => array(952),
        120554 => array(953),
        120555 => array(954),
        120556 => array(955),
        120557 => array(956),
        120558 => array(957),
        120559 => array(958),
        120560 => array(959),
        120561 => array(960),
        120562 => array(961),
        120563 => array(952),
        120564 => array(963),
        120565 => array(964),
        120566 => array(965),
        120567 => array(966),
        120568 => array(967),
        120569 => array(968),
        120570 => array(969),
        120571 => array(8711),
        120572 => array(945),
        120573 => array(946),
        120574 => array(947),
        120575 => array(948),
        120576 => array(949),
        120577 => array(950),
        120578 => array(951),
        120579 => array(952),
        120580 => array(953),
        120581 => array(954),
        120582 => array(955),
        120583 => array(956),
        120584 => array(957),
        120585 => array(958),
        120586 => array(959),
        120587 => array(960),
        120588 => array(961),
        120589 => array(963),
        120590 => array(963),
        120591 => array(964),
        120592 => array(965),
        120593 => array(966),
        120594 => array(967),
        120595 => array(968),
        120596 => array(969),
        120597 => array(8706),
        120598 => array(949),
        120599 => array(952),
        120600 => array(954),
        120601 => array(966),
        120602 => array(961),
        120603 => array(960),
        120604 => array(945),
        120605 => array(946),
        120606 => array(947),
        120607 => array(948),
        120608 => array(949),
        120609 => array(950),
        120610 => array(951),
        120611 => array(952),
        120612 => array(953),
        120613 => array(954),
        120614 => array(955),
        120615 => array(956),
        120616 => array(957),
        120617 => array(958),
        120618 => array(959),
        120619 => array(960),
        120620 => array(961),
        120621 => array(952),
        120622 => array(963),
        120623 => array(964),
        120624 => array(965),
        120625 => array(966),
        120626 => array(967),
        120627 => array(968),
        120628 => array(969),
        120629 => array(8711),
        120630 => array(945),
        120631 => array(946),
        120632 => array(947),
        120633 => array(948),
        120634 => array(949),
        120635 => array(950),
        120636 => array(951),
        120637 => array(952),
        120638 => array(953),
        120639 => array(954),
        120640 => array(955),
        120641 => array(956),
        120642 => array(957),
        120643 => array(958),
        120644 => array(959),
        120645 => array(960),
        120646 => array(961),
        120647 => array(963),
        120648 => array(963),
        120649 => array(964),
        120650 => array(965),
        120651 => array(966),
        120652 => array(967),
        120653 => array(968),
        120654 => array(969),
        120655 => array(8706),
        120656 => array(949),
        120657 => array(952),
        120658 => array(954),
        120659 => array(966),
        120660 => array(961),
        120661 => array(960),
        120662 => array(945),
        120663 => array(946),
        120664 => array(947),
        120665 => array(948),
        120666 => array(949),
        120667 => array(950),
        120668 => array(951),
        120669 => array(952),
        120670 => array(953),
        120671 => array(954),
        120672 => array(955),
        120673 => array(956),
        120674 => array(957),
        120675 => array(958),
        120676 => array(959),
        120677 => array(960),
        120678 => array(961),
        120679 => array(952),
        120680 => array(963),
        120681 => array(964),
        120682 => array(965),
        120683 => array(966),
        120684 => array(967),
        120685 => array(968),
        120686 => array(969),
        120687 => array(8711),
        120688 => array(945),
        120689 => array(946),
        120690 => array(947),
        120691 => array(948),
        120692 => array(949),
        120693 => array(950),
        120694 => array(951),
        120695 => array(952),
        120696 => array(953),
        120697 => array(954),
        120698 => array(955),
        120699 => array(956),
        120700 => array(957),
        120701 => array(958),
        120702 => array(959),
        120703 => array(960),
        120704 => array(961),
        120705 => array(963),
        120706 => array(963),
        120707 => array(964),
        120708 => array(965),
        120709 => array(966),
        120710 => array(967),
        120711 => array(968),
        120712 => array(969),
        120713 => array(8706),
        120714 => array(949),
        120715 => array(952),
        120716 => array(954),
        120717 => array(966),
        120718 => array(961),
        120719 => array(960),
        120720 => array(945),
        120721 => array(946),
        120722 => array(947),
        120723 => array(948),
        120724 => array(949),
        120725 => array(950),
        120726 => array(951),
        120727 => array(952),
        120728 => array(953),
        120729 => array(954),
        120730 => array(955),
        120731 => array(956),
        120732 => array(957),
        120733 => array(958),
        120734 => array(959),
        120735 => array(960),
        120736 => array(961),
        120737 => array(952),
        120738 => array(963),
        120739 => array(964),
        120740 => array(965),
        120741 => array(966),
        120742 => array(967),
        120743 => array(968),
        120744 => array(969),
        120745 => array(8711),
        120746 => array(945),
        120747 => array(946),
        120748 => array(947),
        120749 => array(948),
        120750 => array(949),
        120751 => array(950),
        120752 => array(951),
        120753 => array(952),
        120754 => array(953),
        120755 => array(954),
        120756 => array(955),
        120757 => array(956),
        120758 => array(957),
        120759 => array(958),
        120760 => array(959),
        120761 => array(960),
        120762 => array(961),
        120763 => array(963),
        120764 => array(963),
        120765 => array(964),
        120766 => array(965),
        120767 => array(966),
        120768 => array(967),
        120769 => array(968),
        120770 => array(969),
        120771 => array(8706),
        120772 => array(949),
        120773 => array(952),
        120774 => array(954),
        120775 => array(966),
        120776 => array(961),
        120777 => array(960),
        120778 => array(989),
        120779 => array(989),
        120782 => array(48),
        120783 => array(49),
        120784 => array(50),
        120785 => array(51),
        120786 => array(52),
        120787 => array(53),
        120788 => array(54),
        120789 => array(55),
        120790 => array(56),
        120791 => array(57),
        120792 => array(48),
        120793 => array(49),
        120794 => array(50),
        120795 => array(51),
        120796 => array(52),
        120797 => array(53),
        120798 => array(54),
        120799 => array(55),
        120800 => array(56),
        120801 => array(57),
        120802 => array(48),
        120803 => array(49),
        120804 => array(50),
        120805 => array(51),
        120806 => array(52),
        120807 => array(53),
        120808 => array(54),
        120809 => array(55),
        120810 => array(56),
        120811 => array(57),
        120812 => array(48),
        120813 => array(49),
        120814 => array(50),
        120815 => array(51),
        120816 => array(52),
        120817 => array(53),
        120818 => array(54),
        120819 => array(55),
        120820 => array(56),
        120821 => array(57),
        120822 => array(48),
        120823 => array(49),
        120824 => array(50),
        120825 => array(51),
        120826 => array(52),
        120827 => array(53),
        120828 => array(54),
        120829 => array(55),
        120830 => array(56),
        120831 => array(57),
        122928 => array(1072),
        122929 => array(1073),
        122930 => array(1074),
        122931 => array(1075),
        122932 => array(1076),
        122933 => array(1077),
        122934 => array(1078),
        122935 => array(1079),
        122936 => array(1080),
        122937 => array(1082),
        122938 => array(1083),
        122939 => array(1084),
        122940 => array(1086),
        122941 => array(1087),
        122942 => array(1088),
        122943 => array(1089),
        122944 => array(1090),
        122945 => array(1091),
        122946 => array(1092),
        122947 => array(1093),
        122948 => array(1094),
        122949 => array(1095),
        122950 => array(1096),
        122951 => array(1099),
        122952 => array(1101),
        122953 => array(1102),
        122954 => array(42633),
        122955 => array(1241),
        122956 => array(1110),
        122957 => array(1112),
        122958 => array(1257),
        122959 => array(1199),
        122960 => array(1231),
        122961 => array(1072),
        122962 => array(1073),
        122963 => array(1074),
        122964 => array(1075),
        122965 => array(1076),
        122966 => array(1077),
        122967 => array(1078),
        122968 => array(1079),
        122969 => array(1080),
        122970 => array(1082),
        122971 => array(1083),
        122972 => array(1086),
        122973 => array(1087),
        122974 => array(1089),
        122975 => array(1091),
        122976 => array(1092),
        122977 => array(1093),
        122978 => array(1094),
        122979 => array(1095),
        122980 => array(1096),
        122981 => array(1098),
        122982 => array(1099),
        122983 => array(1169),
        122984 => array(1110),
        122985 => array(1109),
        122986 => array(1119),
        122987 => array(1195),
        122988 => array(42577),
        122989 => array(1201),
        125184 => array(125218),
        125185 => array(125219),
        125186 => array(125220),
        125187 => array(125221),
        125188 => array(125222),
        125189 => array(125223),
        125190 => array(125224),
        125191 => array(125225),
        125192 => array(125226),
        125193 => array(125227),
        125194 => array(125228),
        125195 => array(125229),
        125196 => array(125230),
        125197 => array(125231),
        125198 => array(125232),
        125199 => array(125233),
        125200 => array(125234),
        125201 => array(125235),
        125202 => array(125236),
        125203 => array(125237),
        125204 => array(125238),
        125205 => array(125239),
        125206 => array(125240),
        125207 => array(125241),
        125208 => array(125242),
        125209 => array(125243),
        125210 => array(125244),
        125211 => array(125245),
        125212 => array(125246),
        125213 => array(125247),
        125214 => array(125248),
        125215 => array(125249),
        125216 => array(125250),
        125217 => array(125251),
        126464 => array(1575),
        126465 => array(1576),
        126466 => array(1580),
        126467 => array(1583),
        126469 => array(1608),
        126470 => array(1586),
        126471 => array(1581),
        126472 => array(1591),
        126473 => array(1610),
        126474 => array(1603),
        126475 => array(1604),
        126476 => array(1605),
        126477 => array(1606),
        126478 => array(1587),
        126479 => array(1593),
        126480 => array(1601),
        126481 => array(1589),
        126482 => array(1602),
        126483 => array(1585),
        126484 => array(1588),
        126485 => array(1578),
        126486 => array(1579),
        126487 => array(1582),
        126488 => array(1584),
        126489 => array(1590),
        126490 => array(1592),
        126491 => array(1594),
        126492 => array(1646),
        126493 => array(1722),
        126494 => array(1697),
        126495 => array(1647),
        126497 => array(1576),
        126498 => array(1580),
        126500 => array(1607),
        126503 => array(1581),
        126505 => array(1610),
        126506 => array(1603),
        126507 => array(1604),
        126508 => array(1605),
        126509 => array(1606),
        126510 => array(1587),
        126511 => array(1593),
        126512 => array(1601),
        126513 => array(1589),
        126514 => array(1602),
        126516 => array(1588),
        126517 => array(1578),
        126518 => array(1579),
        126519 => array(1582),
        126521 => array(1590),
        126523 => array(1594),
        126530 => array(1580),
        126535 => array(1581),
        126537 => array(1610),
        126539 => array(1604),
        126541 => array(1606),
        126542 => array(1587),
        126543 => array(1593),
        126545 => array(1589),
        126546 => array(1602),
        126548 => array(1588),
        126551 => array(1582),
        126553 => array(1590),
        126555 => array(1594),
        126557 => array(1722),
        126559 => array(1647),
        126561 => array(1576),
        126562 => array(1580),
        126564 => array(1607),
        126567 => array(1581),
        126568 => array(1591),
        126569 => array(1610),
        126570 => array(1603),
        126572 => array(1605),
        126573 => array(1606),
        126574 => array(1587),
        126575 => array(1593),
        126576 => array(1601),
        126577 => array(1589),
        126578 => array(1602),
        126580 => array(1588),
        126581 => array(1578),
        126582 => array(1579),
        126583 => array(1582),
        126585 => array(1590),
        126586 => array(1592),
        126587 => array(1594),
        126588 => array(1646),
        126590 => array(1697),
        126592 => array(1575),
        126593 => array(1576),
        126594 => array(1580),
        126595 => array(1583),
        126596 => array(1607),
        126597 => array(1608),
        126598 => array(1586),
        126599 => array(1581),
        126600 => array(1591),
        126601 => array(1610),
        126603 => array(1604),
        126604 => array(1605),
        126605 => array(1606),
        126606 => array(1587),
        126607 => array(1593),
        126608 => array(1601),
        126609 => array(1589),
        126610 => array(1602),
        126611 => array(1585),
        126612 => array(1588),
        126613 => array(1578),
        126614 => array(1579),
        126615 => array(1582),
        126616 => array(1584),
        126617 => array(1590),
        126618 => array(1592),
        126619 => array(1594),
        126625 => array(1576),
        126626 => array(1580),
        126627 => array(1583),
        126629 => array(1608),
        126630 => array(1586),
        126631 => array(1581),
        126632 => array(1591),
        126633 => array(1610),
        126635 => array(1604),
        126636 => array(1605),
        126637 => array(1606),
        126638 => array(1587),
        126639 => array(1593),
        126640 => array(1601),
        126641 => array(1589),
        126642 => array(1602),
        126643 => array(1585),
        126644 => array(1588),
        126645 => array(1578),
        126646 => array(1579),
        126647 => array(1582),
        126648 => array(1584),
        126649 => array(1590),
        126650 => array(1592),
        126651 => array(1594),
        127274 => array(12308, 115, 12309),
        127275 => array(99),
        127276 => array(114),
        127277 => array(99, 100),
        127278 => array(119, 122),
        127280 => array(97),
        127281 => array(98),
        127282 => array(99),
        127283 => array(100),
        127284 => array(101),
        127285 => array(102),
        127286 => array(103),
        127287 => array(104),
        127288 => array(105),
        127289 => array(106),
        127290 => array(107),
        127291 => array(108),
        127292 => array(109),
        127293 => array(110),
        127294 => array(111),
        127295 => array(112),
        127296 => array(113),
        127297 => array(114),
        127298 => array(115),
        127299 => array(116),
        127300 => array(117),
        127301 => array(118),
        127302 => array(119),
        127303 => array(120),
        127304 => array(121),
        127305 => array(122),
        127306 => array(104, 118),
        127307 => array(109, 118),
        127308 => array(115, 100),
        127309 => array(115, 115),
        127310 => array(112, 112, 118),
        127311 => array(119, 99),
        127338 => array(109, 99),
        127339 => array(109, 100),
        127340 => array(109, 114),
        127376 => array(100, 106),
        127488 => array(12411, 12363),
        127489 => array(12467, 12467),
        127490 => array(12469),
        127504 => array(25163),
        127505 => array(23383),
        127506 => array(21452),
        127507 => array(12487),
        127508 => array(20108),
        127509 => array(22810),
        127510 => array(35299),
        127511 => array(22825),
        127512 => array(20132),
        127513 => array(26144),
        127514 => array(28961),
        127515 => array(26009),
        127516 => array(21069),
        127517 => array(24460),
        127518 => array(20877),
        127519 => array(26032),
        127520 => array(21021),
        127521 => array(32066),
        127522 => array(29983),
        127523 => array(36009),
        127524 => array(22768),
        127525 => array(21561),
        127526 => array(28436),
        127527 => array(25237),
        127528 => array(25429),
        127529 => array(19968),
        127530 => array(19977),
        127531 => array(36938),
        127532 => array(24038),
        127533 => array(20013),
        127534 => array(21491),
        127535 => array(25351),
        127536 => array(36208),
        127537 => array(25171),
        127538 => array(31105),
        127539 => array(31354),
        127540 => array(21512),
        127541 => array(28288),
        127542 => array(26377),
        127543 => array(26376),
        127544 => array(30003),
        127545 => array(21106),
        127546 => array(21942),
        127547 => array(37197),
        127552 => array(12308, 26412, 12309),
        127553 => array(12308, 19977, 12309),
        127554 => array(12308, 20108, 12309),
        127555 => array(12308, 23433, 12309),
        127556 => array(12308, 28857, 12309),
        127557 => array(12308, 25171, 12309),
        127558 => array(12308, 30423, 12309),
        127559 => array(12308, 21213, 12309),
        127560 => array(12308, 25943, 12309),
        127568 => array(24471),
        127569 => array(21487),
        130032 => array(48),
        130033 => array(49),
        130034 => array(50),
        130035 => array(51),
        130036 => array(52),
        130037 => array(53),
        130038 => array(54),
        130039 => array(55),
        130040 => array(56),
        130041 => array(57),
        194560 => array(20029),
        194561 => array(20024),
        194562 => array(20033),
        194563 => array(131362),
        194564 => array(20320),
        194565 => array(20398),
        194566 => array(20411),
        194567 => array(20482),
        194568 => array(20602),
        194569 => array(20633),
        194570 => array(20711),
        194571 => array(20687),
        194572 => array(13470),
        194573 => array(132666),
        194574 => array(20813),
        194575 => array(20820),
        194576 => array(20836),
        194577 => array(20855),
        194578 => array(132380),
        194579 => array(13497),
        194580 => array(20839),
        194581 => array(20877),
        194582 => array(132427),
        194583 => array(20887),
        194584 => array(20900),
        194585 => array(20172),
        194586 => array(20908),
        194587 => array(20917),
        194588 => array(168415),
        194589 => array(20981),
        194590 => array(20995),
        194591 => array(13535),
        194592 => array(21051),
        194593 => array(21062),
        194594 => array(21106),
        194595 => array(21111),
        194596 => array(13589),
        194597 => array(21191),
        194598 => array(21193),
        194599 => array(21220),
        194600 => array(21242),
        194601 => array(21253),
        194602 => array(21254),
        194603 => array(21271),
        194604 => array(21321),
        194605 => array(21329),
        194606 => array(21338),
        194607 => array(21363),
        194608 => array(21373),
        194609 => array(21375),
        194610 => array(21375),
        194611 => array(21375),
        194612 => array(133676),
        194613 => array(28784),
        194614 => array(21450),
        194615 => array(21471),
        194616 => array(133987),
        194617 => array(21483),
        194618 => array(21489),
        194619 => array(21510),
        194620 => array(21662),
        194621 => array(21560),
        194622 => array(21576),
        194623 => array(21608),
        194624 => array(21666),
        194625 => array(21750),
        194626 => array(21776),
        194627 => array(21843),
        194628 => array(21859),
        194629 => array(21892),
        194630 => array(21892),
        194631 => array(21913),
        194632 => array(21931),
        194633 => array(21939),
        194634 => array(21954),
        194635 => array(22294),
        194636 => array(22022),
        194637 => array(22295),
        194638 => array(22097),
        194639 => array(22132),
        194640 => array(20999),
        194641 => array(22766),
        194642 => array(22478),
        194643 => array(22516),
        194644 => array(22541),
        194645 => array(22411),
        194646 => array(22578),
        194647 => array(22577),
        194648 => array(22700),
        194649 => array(136420),
        194650 => array(22770),
        194651 => array(22775),
        194652 => array(22790),
        194653 => array(22810),
        194654 => array(22818),
        194655 => array(22882),
        194656 => array(136872),
        194657 => array(136938),
        194658 => array(23020),
        194659 => array(23067),
        194660 => array(23079),
        194661 => array(23000),
        194662 => array(23142),
        194663 => array(14062),
        194665 => array(23304),
        194666 => array(23358),
        194667 => array(23358),
        194668 => array(137672),
        194669 => array(23491),
        194670 => array(23512),
        194671 => array(23527),
        194672 => array(23539),
        194673 => array(138008),
        194674 => array(23551),
        194675 => array(23558),
        194677 => array(23586),
        194678 => array(14209),
        194679 => array(23648),
        194680 => array(23662),
        194681 => array(23744),
        194682 => array(23693),
        194683 => array(138724),
        194684 => array(23875),
        194685 => array(138726),
        194686 => array(23918),
        194687 => array(23915),
        194688 => array(23932),
        194689 => array(24033),
        194690 => array(24034),
        194691 => array(14383),
        194692 => array(24061),
        194693 => array(24104),
        194694 => array(24125),
        194695 => array(24169),
        194696 => array(14434),
        194697 => array(139651),
        194698 => array(14460),
        194699 => array(24240),
        194700 => array(24243),
        194701 => array(24246),
        194702 => array(24266),
        194703 => array(172946),
        194704 => array(24318),
        194705 => array(140081),
        194706 => array(140081),
        194707 => array(33281),
        194708 => array(24354),
        194709 => array(24354),
        194710 => array(14535),
        194711 => array(144056),
        194712 => array(156122),
        194713 => array(24418),
        194714 => array(24427),
        194715 => array(14563),
        194716 => array(24474),
        194717 => array(24525),
        194718 => array(24535),
        194719 => array(24569),
        194720 => array(24705),
        194721 => array(14650),
        194722 => array(14620),
        194723 => array(24724),
        194724 => array(141012),
        194725 => array(24775),
        194726 => array(24904),
        194727 => array(24908),
        194728 => array(24910),
        194729 => array(24908),
        194730 => array(24954),
        194731 => array(24974),
        194732 => array(25010),
        194733 => array(24996),
        194734 => array(25007),
        194735 => array(25054),
        194736 => array(25074),
        194737 => array(25078),
        194738 => array(25104),
        194739 => array(25115),
        194740 => array(25181),
        194741 => array(25265),
        194742 => array(25300),
        194743 => array(25424),
        194744 => array(142092),
        194745 => array(25405),
        194746 => array(25340),
        194747 => array(25448),
        194748 => array(25475),
        194749 => array(25572),
        194750 => array(142321),
        194751 => array(25634),
        194752 => array(25541),
        194753 => array(25513),
        194754 => array(14894),
        194755 => array(25705),
        194756 => array(25726),
        194757 => array(25757),
        194758 => array(25719),
        194759 => array(14956),
        194760 => array(25935),
        194761 => array(25964),
        194762 => array(143370),
        194763 => array(26083),
        194764 => array(26360),
        194765 => array(26185),
        194766 => array(15129),
        194767 => array(26257),
        194768 => array(15112),
        194769 => array(15076),
        194770 => array(20882),
        194771 => array(20885),
        194772 => array(26368),
        194773 => array(26268),
        194774 => array(32941),
        194775 => array(17369),
        194776 => array(26391),
        194777 => array(26395),
        194778 => array(26401),
        194779 => array(26462),
        194780 => array(26451),
        194781 => array(144323),
        194782 => array(15177),
        194783 => array(26618),
        194784 => array(26501),
        194785 => array(26706),
        194786 => array(26757),
        194787 => array(144493),
        194788 => array(26766),
        194789 => array(26655),
        194790 => array(26900),
        194791 => array(15261),
        194792 => array(26946),
        194793 => array(27043),
        194794 => array(27114),
        194795 => array(27304),
        194796 => array(145059),
        194797 => array(27355),
        194798 => array(15384),
        194799 => array(27425),
        194800 => array(145575),
        194801 => array(27476),
        194802 => array(15438),
        194803 => array(27506),
        194804 => array(27551),
        194805 => array(27578),
        194806 => array(27579),
        194807 => array(146061),
        194808 => array(138507),
        194809 => array(146170),
        194810 => array(27726),
        194811 => array(146620),
        194812 => array(27839),
        194813 => array(27853),
        194814 => array(27751),
        194815 => array(27926),
        194816 => array(27966),
        194817 => array(28023),
        194818 => array(27969),
        194819 => array(28009),
        194820 => array(28024),
        194821 => array(28037),
        194822 => array(146718),
        194823 => array(27956),
        194824 => array(28207),
        194825 => array(28270),
        194826 => array(15667),
        194827 => array(28363),
        194828 => array(28359),
        194829 => array(147153),
        194830 => array(28153),
        194831 => array(28526),
        194832 => array(147294),
        194833 => array(147342),
        194834 => array(28614),
        194835 => array(28729),
        194836 => array(28702),
        194837 => array(28699),
        194838 => array(15766),
        194839 => array(28746),
        194840 => array(28797),
        194841 => array(28791),
        194842 => array(28845),
        194843 => array(132389),
        194844 => array(28997),
        194845 => array(148067),
        194846 => array(29084),
        194848 => array(29224),
        194849 => array(29237),
        194850 => array(29264),
        194851 => array(149000),
        194852 => array(29312),
        194853 => array(29333),
        194854 => array(149301),
        194855 => array(149524),
        194856 => array(29562),
        194857 => array(29579),
        194858 => array(16044),
        194859 => array(29605),
        194860 => array(16056),
        194861 => array(16056),
        194862 => array(29767),
        194863 => array(29788),
        194864 => array(29809),
        194865 => array(29829),
        194866 => array(29898),
        194867 => array(16155),
        194868 => array(29988),
        194869 => array(150582),
        194870 => array(30014),
        194871 => array(150674),
        194872 => array(30064),
        194873 => array(139679),
        194874 => array(30224),
        194875 => array(151457),
        194876 => array(151480),
        194877 => array(151620),
        194878 => array(16380),
        194879 => array(16392),
        194880 => array(30452),
        194881 => array(151795),
        194882 => array(151794),
        194883 => array(151833),
        194884 => array(151859),
        194885 => array(30494),
        194886 => array(30495),
        194887 => array(30495),
        194888 => array(30538),
        194889 => array(16441),
        194890 => array(30603),
        194891 => array(16454),
        194892 => array(16534),
        194893 => array(152605),
        194894 => array(30798),
        194895 => array(30860),
        194896 => array(30924),
        194897 => array(16611),
        194898 => array(153126),
        194899 => array(31062),
        194900 => array(153242),
        194901 => array(153285),
        194902 => array(31119),
        194903 => array(31211),
        194904 => array(16687),
        194905 => array(31296),
        194906 => array(31306),
        194907 => array(31311),
        194908 => array(153980),
        194909 => array(154279),
        194910 => array(154279),
        194912 => array(16898),
        194913 => array(154539),
        194914 => array(31686),
        194915 => array(31689),
        194916 => array(16935),
        194917 => array(154752),
        194918 => array(31954),
        194919 => array(17056),
        194920 => array(31976),
        194921 => array(31971),
        194922 => array(32000),
        194923 => array(155526),
        194924 => array(32099),
        194925 => array(17153),
        194926 => array(32199),
        194927 => array(32258),
        194928 => array(32325),
        194929 => array(17204),
        194930 => array(156200),
        194931 => array(156231),
        194932 => array(17241),
        194933 => array(156377),
        194934 => array(32634),
        194935 => array(156478),
        194936 => array(32661),
        194937 => array(32762),
        194938 => array(32773),
        194939 => array(156890),
        194940 => array(156963),
        194941 => array(32864),
        194942 => array(157096),
        194943 => array(32880),
        194944 => array(144223),
        194945 => array(17365),
        194946 => array(32946),
        194947 => array(33027),
        194948 => array(17419),
        194949 => array(33086),
        194950 => array(23221),
        194951 => array(157607),
        194952 => array(157621),
        194953 => array(144275),
        194954 => array(144284),
        194955 => array(33281),
        194956 => array(33284),
        194957 => array(36766),
        194958 => array(17515),
        194959 => array(33425),
        194960 => array(33419),
        194961 => array(33437),
        194962 => array(21171),
        194963 => array(33457),
        194964 => array(33459),
        194965 => array(33469),
        194966 => array(33510),
        194967 => array(158524),
        194968 => array(33509),
        194969 => array(33565),
        194970 => array(33635),
        194971 => array(33709),
        194972 => array(33571),
        194973 => array(33725),
        194974 => array(33767),
        194975 => array(33879),
        194976 => array(33619),
        194977 => array(33738),
        194978 => array(33740),
        194979 => array(33756),
        194980 => array(158774),
        194981 => array(159083),
        194982 => array(158933),
        194983 => array(17707),
        194984 => array(34033),
        194985 => array(34035),
        194986 => array(34070),
        194987 => array(160714),
        194988 => array(34148),
        194989 => array(159532),
        194990 => array(17757),
        194991 => array(17761),
        194992 => array(159665),
        194993 => array(159954),
        194994 => array(17771),
        194995 => array(34384),
        194996 => array(34396),
        194997 => array(34407),
        194998 => array(34409),
        194999 => array(34473),
        195000 => array(34440),
        195001 => array(34574),
        195002 => array(34530),
        195003 => array(34681),
        195004 => array(34600),
        195005 => array(34667),
        195006 => array(34694),
        195008 => array(34785),
        195009 => array(34817),
        195010 => array(17913),
        195011 => array(34912),
        195012 => array(34915),
        195013 => array(161383),
        195014 => array(35031),
        195015 => array(35038),
        195016 => array(17973),
        195017 => array(35066),
        195018 => array(13499),
        195019 => array(161966),
        195020 => array(162150),
        195021 => array(18110),
        195022 => array(18119),
        195023 => array(35488),
        195024 => array(35565),
        195025 => array(35722),
        195026 => array(35925),
        195027 => array(162984),
        195028 => array(36011),
        195029 => array(36033),
        195030 => array(36123),
        195031 => array(36215),
        195032 => array(163631),
        195033 => array(133124),
        195034 => array(36299),
        195035 => array(36284),
        195036 => array(36336),
        195037 => array(133342),
        195038 => array(36564),
        195039 => array(36664),
        195040 => array(165330),
        195041 => array(165357),
        195042 => array(37012),
        195043 => array(37105),
        195044 => array(37137),
        195045 => array(165678),
        195046 => array(37147),
        195047 => array(37432),
        195048 => array(37591),
        195049 => array(37592),
        195050 => array(37500),
        195051 => array(37881),
        195052 => array(37909),
        195053 => array(166906),
        195054 => array(38283),
        195055 => array(18837),
        195056 => array(38327),
        195057 => array(167287),
        195058 => array(18918),
        195059 => array(38595),
        195060 => array(23986),
        195061 => array(38691),
        195062 => array(168261),
        195063 => array(168474),
        195064 => array(19054),
        195065 => array(19062),
        195066 => array(38880),
        195067 => array(168970),
        195068 => array(19122),
        195069 => array(169110),
        195070 => array(38923),
        195071 => array(38923),
        195072 => array(38953),
        195073 => array(169398),
        195074 => array(39138),
        195075 => array(19251),
        195076 => array(39209),
        195077 => array(39335),
        195078 => array(39362),
        195079 => array(39422),
        195080 => array(19406),
        195081 => array(170800),
        195082 => array(39698),
        195083 => array(40000),
        195084 => array(40189),
        195085 => array(19662),
        195086 => array(19693),
        195087 => array(40295),
        195088 => array(172238),
        195089 => array(19704),
        195090 => array(172293),
        195091 => array(172558),
        195092 => array(172689),
        195093 => array(40635),
        195094 => array(19798),
        195095 => array(40697),
        195096 => array(40702),
        195097 => array(40709),
        195098 => array(40719),
        195099 => array(40726),
        195100 => array(40763),
        195101 => array(173568),
    );

    protected static $mappedDisallowedSTD3 = array(
        160 => array(32),
        168 => array(32, 776),
        175 => array(32, 772),
        180 => array(32, 769),
        184 => array(32, 807),
        728 => array(32, 774),
        729 => array(32, 775),
        730 => array(32, 778),
        731 => array(32, 808),
        732 => array(32, 771),
        733 => array(32, 779),
        890 => array(32, 953),
        894 => array(59),
        900 => array(32, 769),
        901 => array(32, 776, 769),
        8125 => array(32, 787),
        8127 => array(32, 787),
        8128 => array(32, 834),
        8129 => array(32, 776, 834),
        8141 => array(32, 787, 768),
        8142 => array(32, 787, 769),
        8143 => array(32, 787, 834),
        8157 => array(32, 788, 768),
        8158 => array(32, 788, 769),
        8159 => array(32, 788, 834),
        8173 => array(32, 776, 768),
        8174 => array(32, 776, 769),
        8175 => array(96),
        8189 => array(32, 769),
        8190 => array(32, 788),
        8192 => array(32),
        8193 => array(32),
        8194 => array(32),
        8195 => array(32),
        8196 => array(32),
        8197 => array(32),
        8198 => array(32),
        8199 => array(32),
        8200 => array(32),
        8201 => array(32),
        8202 => array(32),
        8215 => array(32, 819),
        8239 => array(32),
        8252 => array(33, 33),
        8254 => array(32, 773),
        8263 => array(63, 63),
        8264 => array(63, 33),
        8265 => array(33, 63),
        8287 => array(32),
        8314 => array(43),
        8316 => array(61),
        8317 => array(40),
        8318 => array(41),
        8330 => array(43),
        8332 => array(61),
        8333 => array(40),
        8334 => array(41),
        8448 => array(97, 47, 99),
        8449 => array(97, 47, 115),
        8453 => array(99, 47, 111),
        8454 => array(99, 47, 117),
        9332 => array(40, 49, 41),
        9333 => array(40, 50, 41),
        9334 => array(40, 51, 41),
        9335 => array(40, 52, 41),
        9336 => array(40, 53, 41),
        9337 => array(40, 54, 41),
        9338 => array(40, 55, 41),
        9339 => array(40, 56, 41),
        9340 => array(40, 57, 41),
        9341 => array(40, 49, 48, 41),
        9342 => array(40, 49, 49, 41),
        9343 => array(40, 49, 50, 41),
        9344 => array(40, 49, 51, 41),
        9345 => array(40, 49, 52, 41),
        9346 => array(40, 49, 53, 41),
        9347 => array(40, 49, 54, 41),
        9348 => array(40, 49, 55, 41),
        9349 => array(40, 49, 56, 41),
        9350 => array(40, 49, 57, 41),
        9351 => array(40, 50, 48, 41),
        9372 => array(40, 97, 41),
        9373 => array(40, 98, 41),
        9374 => array(40, 99, 41),
        9375 => array(40, 100, 41),
        9376 => array(40, 101, 41),
        9377 => array(40, 102, 41),
        9378 => array(40, 103, 41),
        9379 => array(40, 104, 41),
        9380 => array(40, 105, 41),
        9381 => array(40, 106, 41),
        9382 => array(40, 107, 41),
        9383 => array(40, 108, 41),
        9384 => array(40, 109, 41),
        9385 => array(40, 110, 41),
        9386 => array(40, 111, 41),
        9387 => array(40, 112, 41),
        9388 => array(40, 113, 41),
        9389 => array(40, 114, 41),
        9390 => array(40, 115, 41),
        9391 => array(40, 116, 41),
        9392 => array(40, 117, 41),
        9393 => array(40, 118, 41),
        9394 => array(40, 119, 41),
        9395 => array(40, 120, 41),
        9396 => array(40, 121, 41),
        9397 => array(40, 122, 41),
        10868 => array(58, 58, 61),
        10869 => array(61, 61),
        10870 => array(61, 61, 61),
        12288 => array(32),
        12443 => array(32, 12441),
        12444 => array(32, 12442),
        12800 => array(40, 4352, 41),
        12801 => array(40, 4354, 41),
        12802 => array(40, 4355, 41),
        12803 => array(40, 4357, 41),
        12804 => array(40, 4358, 41),
        12805 => array(40, 4359, 41),
        12806 => array(40, 4361, 41),
        12807 => array(40, 4363, 41),
        12808 => array(40, 4364, 41),
        12809 => array(40, 4366, 41),
        12810 => array(40, 4367, 41),
        12811 => array(40, 4368, 41),
        12812 => array(40, 4369, 41),
        12813 => array(40, 4370, 41),
        12814 => array(40, 44032, 41),
        12815 => array(40, 45208, 41),
        12816 => array(40, 45796, 41),
        12817 => array(40, 46972, 41),
        12818 => array(40, 47560, 41),
        12819 => array(40, 48148, 41),
        12820 => array(40, 49324, 41),
        12821 => array(40, 50500, 41),
        12822 => array(40, 51088, 41),
        12823 => array(40, 52264, 41),
        12824 => array(40, 52852, 41),
        12825 => array(40, 53440, 41),
        12826 => array(40, 54028, 41),
        12827 => array(40, 54616, 41),
        12828 => array(40, 51452, 41),
        12829 => array(40, 50724, 51204, 41),
        12830 => array(40, 50724, 54980, 41),
        12832 => array(40, 19968, 41),
        12833 => array(40, 20108, 41),
        12834 => array(40, 19977, 41),
        12835 => array(40, 22235, 41),
        12836 => array(40, 20116, 41),
        12837 => array(40, 20845, 41),
        12838 => array(40, 19971, 41),
        12839 => array(40, 20843, 41),
        12840 => array(40, 20061, 41),
        12841 => array(40, 21313, 41),
        12842 => array(40, 26376, 41),
        12843 => array(40, 28779, 41),
        12844 => array(40, 27700, 41),
        12845 => array(40, 26408, 41),
        12846 => array(40, 37329, 41),
        12847 => array(40, 22303, 41),
        12848 => array(40, 26085, 41),
        12849 => array(40, 26666, 41),
        12850 => array(40, 26377, 41),
        12851 => array(40, 31038, 41),
        12852 => array(40, 21517, 41),
        12853 => array(40, 29305, 41),
        12854 => array(40, 36001, 41),
        12855 => array(40, 31069, 41),
        12856 => array(40, 21172, 41),
        12857 => array(40, 20195, 41),
        12858 => array(40, 21628, 41),
        12859 => array(40, 23398, 41),
        12860 => array(40, 30435, 41),
        12861 => array(40, 20225, 41),
        12862 => array(40, 36039, 41),
        12863 => array(40, 21332, 41),
        12864 => array(40, 31085, 41),
        12865 => array(40, 20241, 41),
        12866 => array(40, 33258, 41),
        12867 => array(40, 33267, 41),
        64297 => array(43),
        64606 => array(32, 1612, 1617),
        64607 => array(32, 1613, 1617),
        64608 => array(32, 1614, 1617),
        64609 => array(32, 1615, 1617),
        64610 => array(32, 1616, 1617),
        64611 => array(32, 1617, 1648),
        65018 => array(1589, 1604, 1609, 32, 1575, 1604, 1604, 1607, 32, 1593, 1604, 1610, 1607, 32, 1608, 1587, 1604, 1605),
        65019 => array(1580, 1604, 32, 1580, 1604, 1575, 1604, 1607),
        65040 => array(44),
        65043 => array(58),
        65044 => array(59),
        65045 => array(33),
        65046 => array(63),
        65075 => array(95),
        65076 => array(95),
        65077 => array(40),
        65078 => array(41),
        65079 => array(123),
        65080 => array(125),
        65095 => array(91),
        65096 => array(93),
        65097 => array(32, 773),
        65098 => array(32, 773),
        65099 => array(32, 773),
        65100 => array(32, 773),
        65101 => array(95),
        65102 => array(95),
        65103 => array(95),
        65104 => array(44),
        65108 => array(59),
        65109 => array(58),
        65110 => array(63),
        65111 => array(33),
        65113 => array(40),
        65114 => array(41),
        65115 => array(123),
        65116 => array(125),
        65119 => array(35),
        65120 => array(38),
        65121 => array(42),
        65122 => array(43),
        65124 => array(60),
        65125 => array(62),
        65126 => array(61),
        65128 => array(92),
        65129 => array(36),
        65130 => array(37),
        65131 => array(64),
        65136 => array(32, 1611),
        65138 => array(32, 1612),
        65140 => array(32, 1613),
        65142 => array(32, 1614),
        65144 => array(32, 1615),
        65146 => array(32, 1616),
        65148 => array(32, 1617),
        65150 => array(32, 1618),
        65281 => array(33),
        65282 => array(34),
        65283 => array(35),
        65284 => array(36),
        65285 => array(37),
        65286 => array(38),
        65287 => array(39),
        65288 => array(40),
        65289 => array(41),
        65290 => array(42),
        65291 => array(43),
        65292 => array(44),
        65295 => array(47),
        65306 => array(58),
        65307 => array(59),
        65308 => array(60),
        65309 => array(61),
        65310 => array(62),
        65311 => array(63),
        65312 => array(64),
        65339 => array(91),
        65340 => array(92),
        65341 => array(93),
        65342 => array(94),
        65343 => array(95),
        65344 => array(96),
        65371 => array(123),
        65372 => array(124),
        65373 => array(125),
        65374 => array(126),
        65507 => array(32, 772),
        127233 => array(48, 44),
        127234 => array(49, 44),
        127235 => array(50, 44),
        127236 => array(51, 44),
        127237 => array(52, 44),
        127238 => array(53, 44),
        127239 => array(54, 44),
        127240 => array(55, 44),
        127241 => array(56, 44),
        127242 => array(57, 44),
        127248 => array(40, 97, 41),
        127249 => array(40, 98, 41),
        127250 => array(40, 99, 41),
        127251 => array(40, 100, 41),
        127252 => array(40, 101, 41),
        127253 => array(40, 102, 41),
        127254 => array(40, 103, 41),
        127255 => array(40, 104, 41),
        127256 => array(40, 105, 41),
        127257 => array(40, 106, 41),
        127258 => array(40, 107, 41),
        127259 => array(40, 108, 41),
        127260 => array(40, 109, 41),
        127261 => array(40, 110, 41),
        127262 => array(40, 111, 41),
        127263 => array(40, 112, 41),
        127264 => array(40, 113, 41),
        127265 => array(40, 114, 41),
        127266 => array(40, 115, 41),
        127267 => array(40, 116, 41),
        127268 => array(40, 117, 41),
        127269 => array(40, 118, 41),
        127270 => array(40, 119, 41),
        127271 => array(40, 120, 41),
        127272 => array(40, 121, 41),
        127273 => array(40, 122, 41),
    );

    /**
     * Get the mapping for a specific code point.
     *
     * @param int $codepoint
     * @param bool $useSTD3ASCIIRules
     *
     * @return int[]|null
     */
    public static function getMapped($codepoint, $useSTD3ASCIIRules = true)
    {
        $result = null;
        if (isset(static::$mapped[$codepoint])) {
            $result = static::$mapped[$codepoint];
        } elseif (!$useSTD3ASCIIRules && isset(static::$mappedDisallowedSTD3[$codepoint])) {
            $result = static::$mappedDisallowedSTD3[$codepoint];
        }

        return $result;
    }

    protected static $validSingle = array(
        45 => 1,
        46 => 1,
        48 => 1,
        49 => 1,
        50 => 1,
        51 => 1,
        52 => 1,
        53 => 1,
        54 => 1,
        55 => 1,
        56 => 1,
        57 => 1,
        97 => 1,
        98 => 1,
        99 => 1,
        100 => 1,
        101 => 1,
        102 => 1,
        103 => 1,
        104 => 1,
        105 => 1,
        106 => 1,
        107 => 1,
        108 => 1,
        109 => 1,
        110 => 1,
        111 => 1,
        112 => 1,
        113 => 1,
        114 => 1,
        115 => 1,
        116 => 1,
        117 => 1,
        118 => 1,
        119 => 1,
        120 => 1,
        121 => 1,
        122 => 1,
        161 => 2,
        162 => 2,
        163 => 2,
        164 => 2,
        165 => 2,
        166 => 2,
        167 => 2,
        169 => 2,
        171 => 2,
        172 => 2,
        174 => 2,
        176 => 2,
        177 => 2,
        182 => 2,
        183 => 1,
        187 => 2,
        191 => 2,
        215 => 2,
        224 => 1,
        225 => 1,
        226 => 1,
        227 => 1,
        228 => 1,
        229 => 1,
        230 => 1,
        231 => 1,
        232 => 1,
        233 => 1,
        234 => 1,
        235 => 1,
        236 => 1,
        237 => 1,
        238 => 1,
        239 => 1,
        240 => 1,
        241 => 1,
        242 => 1,
        243 => 1,
        244 => 1,
        245 => 1,
        246 => 1,
        247 => 2,
        248 => 1,
        249 => 1,
        250 => 1,
        251 => 1,
        252 => 1,
        253 => 1,
        254 => 1,
        255 => 1,
        257 => 1,
        259 => 1,
        261 => 1,
        263 => 1,
        265 => 1,
        267 => 1,
        269 => 1,
        271 => 1,
        273 => 1,
        275 => 1,
        277 => 1,
        279 => 1,
        281 => 1,
        283 => 1,
        285 => 1,
        287 => 1,
        289 => 1,
        291 => 1,
        293 => 1,
        295 => 1,
        297 => 1,
        299 => 1,
        301 => 1,
        303 => 1,
        305 => 1,
        309 => 1,
        311 => 1,
        312 => 1,
        314 => 1,
        316 => 1,
        318 => 1,
        322 => 1,
        324 => 1,
        326 => 1,
        328 => 1,
        331 => 1,
        333 => 1,
        335 => 1,
        337 => 1,
        339 => 1,
        341 => 1,
        343 => 1,
        345 => 1,
        347 => 1,
        349 => 1,
        351 => 1,
        353 => 1,
        355 => 1,
        357 => 1,
        359 => 1,
        361 => 1,
        363 => 1,
        365 => 1,
        367 => 1,
        369 => 1,
        371 => 1,
        373 => 1,
        375 => 1,
        378 => 1,
        380 => 1,
        382 => 1,
        384 => 1,
        387 => 1,
        389 => 1,
        392 => 1,
        396 => 1,
        397 => 1,
        402 => 1,
        405 => 1,
        409 => 1,
        410 => 1,
        411 => 1,
        414 => 1,
        417 => 1,
        419 => 1,
        421 => 1,
        424 => 1,
        426 => 1,
        427 => 1,
        429 => 1,
        432 => 1,
        436 => 1,
        438 => 1,
        441 => 1,
        442 => 1,
        443 => 1,
        445 => 1,
        446 => 1,
        447 => 1,
        448 => 1,
        449 => 1,
        450 => 1,
        451 => 1,
        462 => 1,
        464 => 1,
        466 => 1,
        468 => 1,
        470 => 1,
        472 => 1,
        474 => 1,
        476 => 1,
        477 => 1,
        479 => 1,
        481 => 1,
        483 => 1,
        485 => 1,
        487 => 1,
        489 => 1,
        491 => 1,
        493 => 1,
        495 => 1,
        496 => 1,
        501 => 1,
        505 => 1,
        507 => 1,
        509 => 1,
        511 => 1,
        513 => 1,
        515 => 1,
        517 => 1,
        519 => 1,
        521 => 1,
        523 => 1,
        525 => 1,
        527 => 1,
        529 => 1,
        531 => 1,
        533 => 1,
        535 => 1,
        537 => 1,
        539 => 1,
        541 => 1,
        543 => 1,
        545 => 1,
        547 => 1,
        549 => 1,
        551 => 1,
        553 => 1,
        555 => 1,
        557 => 1,
        559 => 1,
        561 => 1,
        563 => 1,
        564 => 1,
        565 => 1,
        566 => 1,
        567 => 1,
        568 => 1,
        569 => 1,
        572 => 1,
        575 => 1,
        576 => 1,
        578 => 1,
        583 => 1,
        585 => 1,
        587 => 1,
        589 => 1,
        591 => 1,
        681 => 1,
        682 => 1,
        683 => 1,
        684 => 1,
        685 => 1,
        686 => 1,
        687 => 1,
        697 => 1,
        698 => 1,
        699 => 1,
        700 => 1,
        701 => 1,
        702 => 1,
        703 => 1,
        704 => 1,
        705 => 1,
        706 => 2,
        707 => 2,
        708 => 2,
        709 => 2,
        710 => 1,
        711 => 1,
        712 => 1,
        713 => 1,
        714 => 1,
        715 => 1,
        716 => 1,
        717 => 1,
        718 => 1,
        719 => 1,
        720 => 1,
        721 => 1,
        722 => 2,
        723 => 2,
        724 => 2,
        725 => 2,
        726 => 2,
        727 => 2,
        734 => 2,
        735 => 2,
        741 => 2,
        742 => 2,
        743 => 2,
        744 => 2,
        745 => 2,
        746 => 2,
        747 => 2,
        748 => 1,
        749 => 2,
        750 => 1,
        751 => 2,
        752 => 2,
        753 => 2,
        754 => 2,
        755 => 2,
        756 => 2,
        757 => 2,
        758 => 2,
        759 => 2,
        760 => 2,
        761 => 2,
        762 => 2,
        763 => 2,
        764 => 2,
        765 => 2,
        766 => 2,
        767 => 2,
        834 => 1,
        838 => 1,
        839 => 1,
        840 => 1,
        841 => 1,
        842 => 1,
        843 => 1,
        844 => 1,
        845 => 1,
        846 => 1,
        848 => 1,
        849 => 1,
        850 => 1,
        851 => 1,
        852 => 1,
        853 => 1,
        854 => 1,
        855 => 1,
        856 => 1,
        857 => 1,
        858 => 1,
        859 => 1,
        860 => 1,
        861 => 1,
        862 => 1,
        863 => 1,
        864 => 1,
        865 => 1,
        866 => 1,
        867 => 1,
        868 => 1,
        869 => 1,
        870 => 1,
        871 => 1,
        872 => 1,
        873 => 1,
        874 => 1,
        875 => 1,
        876 => 1,
        877 => 1,
        878 => 1,
        879 => 1,
        881 => 1,
        883 => 1,
        885 => 1,
        887 => 1,
        891 => 1,
        892 => 1,
        893 => 1,
        912 => 1,
        940 => 1,
        941 => 1,
        942 => 1,
        943 => 1,
        944 => 1,
        945 => 1,
        946 => 1,
        947 => 1,
        948 => 1,
        949 => 1,
        950 => 1,
        951 => 1,
        952 => 1,
        953 => 1,
        954 => 1,
        955 => 1,
        956 => 1,
        957 => 1,
        958 => 1,
        959 => 1,
        960 => 1,
        961 => 1,
        963 => 1,
        964 => 1,
        965 => 1,
        966 => 1,
        967 => 1,
        968 => 1,
        969 => 1,
        970 => 1,
        971 => 1,
        972 => 1,
        973 => 1,
        974 => 1,
        983 => 1,
        985 => 1,
        987 => 1,
        989 => 1,
        991 => 1,
        993 => 1,
        995 => 1,
        997 => 1,
        999 => 1,
        1001 => 1,
        1003 => 1,
        1005 => 1,
        1007 => 1,
        1011 => 1,
        1014 => 2,
        1016 => 1,
        1019 => 1,
        1020 => 1,
        1104 => 1,
        1105 => 1,
        1106 => 1,
        1107 => 1,
        1108 => 1,
        1109 => 1,
        1110 => 1,
        1111 => 1,
        1112 => 1,
        1113 => 1,
        1114 => 1,
        1115 => 1,
        1116 => 1,
        1117 => 1,
        1118 => 1,
        1119 => 1,
        1121 => 1,
        1123 => 1,
        1125 => 1,
        1127 => 1,
        1129 => 1,
        1131 => 1,
        1133 => 1,
        1135 => 1,
        1137 => 1,
        1139 => 1,
        1141 => 1,
        1143 => 1,
        1145 => 1,
        1147 => 1,
        1149 => 1,
        1151 => 1,
        1153 => 1,
        1154 => 2,
        1155 => 1,
        1156 => 1,
        1157 => 1,
        1158 => 1,
        1159 => 1,
        1160 => 2,
        1161 => 2,
        1163 => 1,
        1165 => 1,
        1167 => 1,
        1169 => 1,
        1171 => 1,
        1173 => 1,
        1175 => 1,
        1177 => 1,
        1179 => 1,
        1181 => 1,
        1183 => 1,
        1185 => 1,
        1187 => 1,
        1189 => 1,
        1191 => 1,
        1193 => 1,
        1195 => 1,
        1197 => 1,
        1199 => 1,
        1201 => 1,
        1203 => 1,
        1205 => 1,
        1207 => 1,
        1209 => 1,
        1211 => 1,
        1213 => 1,
        1215 => 1,
        1218 => 1,
        1220 => 1,
        1222 => 1,
        1224 => 1,
        1226 => 1,
        1228 => 1,
        1230 => 1,
        1231 => 1,
        1233 => 1,
        1235 => 1,
        1237 => 1,
        1239 => 1,
        1241 => 1,
        1243 => 1,
        1245 => 1,
        1247 => 1,
        1249 => 1,
        1251 => 1,
        1253 => 1,
        1255 => 1,
        1257 => 1,
        1259 => 1,
        1261 => 1,
        1263 => 1,
        1265 => 1,
        1267 => 1,
        1269 => 1,
        1271 => 1,
        1273 => 1,
        1275 => 1,
        1277 => 1,
        1279 => 1,
        1281 => 1,
        1283 => 1,
        1285 => 1,
        1287 => 1,
        1289 => 1,
        1291 => 1,
        1293 => 1,
        1295 => 1,
        1297 => 1,
        1299 => 1,
        1301 => 1,
        1303 => 1,
        1305 => 1,
        1307 => 1,
        1309 => 1,
        1311 => 1,
        1313 => 1,
        1315 => 1,
        1317 => 1,
        1319 => 1,
        1321 => 1,
        1323 => 1,
        1325 => 1,
        1327 => 1,
        1369 => 1,
        1370 => 2,
        1371 => 2,
        1372 => 2,
        1373 => 2,
        1374 => 2,
        1375 => 2,
        1376 => 1,
        1416 => 1,
        1417 => 2,
        1418 => 2,
        1421 => 2,
        1422 => 2,
        1423 => 2,
        1425 => 1,
        1426 => 1,
        1427 => 1,
        1428 => 1,
        1429 => 1,
        1430 => 1,
        1431 => 1,
        1432 => 1,
        1433 => 1,
        1434 => 1,
        1435 => 1,
        1436 => 1,
        1437 => 1,
        1438 => 1,
        1439 => 1,
        1440 => 1,
        1441 => 1,
        1442 => 1,
        1443 => 1,
        1444 => 1,
        1445 => 1,
        1446 => 1,
        1447 => 1,
        1448 => 1,
        1449 => 1,
        1450 => 1,
        1451 => 1,
        1452 => 1,
        1453 => 1,
        1454 => 1,
        1455 => 1,
        1456 => 1,
        1457 => 1,
        1458 => 1,
        1459 => 1,
        1460 => 1,
        1461 => 1,
        1462 => 1,
        1463 => 1,
        1464 => 1,
        1465 => 1,
        1466 => 1,
        1467 => 1,
        1468 => 1,
        1469 => 1,
        1470 => 2,
        1471 => 1,
        1472 => 2,
        1473 => 1,
        1474 => 1,
        1475 => 2,
        1476 => 1,
        1477 => 1,
        1478 => 2,
        1479 => 1,
        1488 => 1,
        1489 => 1,
        1490 => 1,
        1491 => 1,
        1492 => 1,
        1493 => 1,
        1494 => 1,
        1495 => 1,
        1496 => 1,
        1497 => 1,
        1498 => 1,
        1499 => 1,
        1500 => 1,
        1501 => 1,
        1502 => 1,
        1503 => 1,
        1504 => 1,
        1505 => 1,
        1506 => 1,
        1507 => 1,
        1508 => 1,
        1509 => 1,
        1510 => 1,
        1511 => 1,
        1512 => 1,
        1513 => 1,
        1514 => 1,
        1519 => 1,
        1520 => 1,
        1521 => 1,
        1522 => 1,
        1523 => 1,
        1524 => 1,
        1542 => 2,
        1543 => 2,
        1544 => 2,
        1545 => 2,
        1546 => 2,
        1547 => 2,
        1548 => 2,
        1549 => 2,
        1550 => 2,
        1551 => 2,
        1552 => 1,
        1553 => 1,
        1554 => 1,
        1555 => 1,
        1556 => 1,
        1557 => 1,
        1558 => 1,
        1559 => 1,
        1560 => 1,
        1561 => 1,
        1562 => 1,
        1563 => 2,
        1565 => 2,
        1566 => 2,
        1567 => 2,
        1568 => 1,
        1569 => 1,
        1570 => 1,
        1571 => 1,
        1572 => 1,
        1573 => 1,
        1574 => 1,
        1575 => 1,
        1576 => 1,
        1577 => 1,
        1578 => 1,
        1579 => 1,
        1580 => 1,
        1581 => 1,
        1582 => 1,
        1583 => 1,
        1584 => 1,
        1585 => 1,
        1586 => 1,
        1587 => 1,
        1588 => 1,
        1589 => 1,
        1590 => 1,
        1591 => 1,
        1592 => 1,
        1593 => 1,
        1594 => 1,
        1595 => 1,
        1596 => 1,
        1597 => 1,
        1598 => 1,
        1599 => 1,
        1600 => 2,
        1601 => 1,
        1602 => 1,
        1603 => 1,
        1604 => 1,
        1605 => 1,
        1606 => 1,
        1607 => 1,
        1608 => 1,
        1609 => 1,
        1610 => 1,
        1611 => 1,
        1612 => 1,
        1613 => 1,
        1614 => 1,
        1615 => 1,
        1616 => 1,
        1617 => 1,
        1618 => 1,
        1619 => 1,
        1620 => 1,
        1621 => 1,
        1622 => 1,
        1623 => 1,
        1624 => 1,
        1625 => 1,
        1626 => 1,
        1627 => 1,
        1628 => 1,
        1629 => 1,
        1630 => 1,
        1631 => 1,
        1632 => 1,
        1633 => 1,
        1634 => 1,
        1635 => 1,
        1636 => 1,
        1637 => 1,
        1638 => 1,
        1639 => 1,
        1640 => 1,
        1641 => 1,
        1642 => 2,
        1643 => 2,
        1644 => 2,
        1645 => 2,
        1646 => 1,
        1647 => 1,
        1648 => 1,
        1649 => 1,
        1650 => 1,
        1651 => 1,
        1652 => 1,
        1720 => 1,
        1721 => 1,
        1722 => 1,
        1723 => 1,
        1724 => 1,
        1725 => 1,
        1726 => 1,
        1727 => 1,
        1728 => 1,
        1729 => 1,
        1730 => 1,
        1731 => 1,
        1732 => 1,
        1733 => 1,
        1734 => 1,
        1735 => 1,
        1736 => 1,
        1737 => 1,
        1738 => 1,
        1739 => 1,
        1740 => 1,
        1741 => 1,
        1742 => 1,
        1743 => 1,
        1744 => 1,
        1745 => 1,
        1746 => 1,
        1747 => 1,
        1748 => 2,
        1749 => 1,
        1750 => 1,
        1751 => 1,
        1752 => 1,
        1753 => 1,
        1754 => 1,
        1755 => 1,
        1756 => 1,
        1758 => 2,
        1759 => 1,
        1760 => 1,
        1761 => 1,
        1762 => 1,
        1763 => 1,
        1764 => 1,
        1765 => 1,
        1766 => 1,
        1767 => 1,
        1768 => 1,
        1769 => 2,
        1770 => 1,
        1771 => 1,
        1772 => 1,
        1773 => 1,
        1774 => 1,
        1775 => 1,
        1776 => 1,
        1777 => 1,
        1778 => 1,
        1779 => 1,
        1780 => 1,
        1781 => 1,
        1782 => 1,
        1783 => 1,
        1784 => 1,
        1785 => 1,
        1786 => 1,
        1787 => 1,
        1788 => 1,
        1789 => 1,
        1790 => 1,
        1791 => 1,
        1792 => 2,
        1793 => 2,
        1794 => 2,
        1795 => 2,
        1796 => 2,
        1797 => 2,
        1798 => 2,
        1799 => 2,
        1800 => 2,
        1801 => 2,
        1802 => 2,
        1803 => 2,
        1804 => 2,
        1805 => 2,
        1837 => 1,
        1838 => 1,
        1839 => 1,
        1840 => 1,
        1841 => 1,
        1842 => 1,
        1843 => 1,
        1844 => 1,
        1845 => 1,
        1846 => 1,
        1847 => 1,
        1848 => 1,
        1849 => 1,
        1850 => 1,
        1851 => 1,
        1852 => 1,
        1853 => 1,
        1854 => 1,
        1855 => 1,
        1856 => 1,
        1857 => 1,
        1858 => 1,
        1859 => 1,
        1860 => 1,
        1861 => 1,
        1862 => 1,
        1863 => 1,
        1864 => 1,
        1865 => 1,
        1866 => 1,
        1869 => 1,
        1870 => 1,
        1871 => 1,
        1902 => 1,
        1903 => 1,
        1904 => 1,
        1905 => 1,
        1906 => 1,
        1907 => 1,
        1908 => 1,
        1909 => 1,
        1910 => 1,
        1911 => 1,
        1912 => 1,
        1913 => 1,
        1914 => 1,
        1915 => 1,
        1916 => 1,
        1917 => 1,
        1918 => 1,
        1919 => 1,
        1969 => 1,
        2038 => 2,
        2039 => 2,
        2040 => 2,
        2041 => 2,
        2042 => 2,
        2045 => 1,
        2046 => 2,
        2047 => 2,
        2096 => 2,
        2097 => 2,
        2098 => 2,
        2099 => 2,
        2100 => 2,
        2101 => 2,
        2102 => 2,
        2103 => 2,
        2104 => 2,
        2105 => 2,
        2106 => 2,
        2107 => 2,
        2108 => 2,
        2109 => 2,
        2110 => 2,
        2142 => 2,
        2144 => 1,
        2145 => 1,
        2146 => 1,
        2147 => 1,
        2148 => 1,
        2149 => 1,
        2150 => 1,
        2151 => 1,
        2152 => 1,
        2153 => 1,
        2154 => 1,
        2160 => 1,
        2161 => 1,
        2162 => 1,
        2163 => 1,
        2164 => 1,
        2165 => 1,
        2166 => 1,
        2167 => 1,
        2168 => 1,
        2169 => 1,
        2170 => 1,
        2171 => 1,
        2172 => 1,
        2173 => 1,
        2174 => 1,
        2175 => 1,
        2176 => 1,
        2177 => 1,
        2178 => 1,
        2179 => 1,
        2180 => 1,
        2181 => 1,
        2182 => 1,
        2183 => 1,
        2184 => 2,
        2185 => 1,
        2186 => 1,
        2187 => 1,
        2188 => 1,
        2189 => 1,
        2190 => 1,
        2200 => 1,
        2201 => 1,
        2202 => 1,
        2203 => 1,
        2204 => 1,
        2205 => 1,
        2206 => 1,
        2207 => 1,
        2208 => 1,
        2209 => 1,
        2210 => 1,
        2211 => 1,
        2212 => 1,
        2213 => 1,
        2214 => 1,
        2215 => 1,
        2216 => 1,
        2217 => 1,
        2218 => 1,
        2219 => 1,
        2220 => 1,
        2221 => 1,
        2222 => 1,
        2223 => 1,
        2224 => 1,
        2225 => 1,
        2226 => 1,
        2227 => 1,
        2228 => 1,
        2229 => 1,
        2230 => 1,
        2231 => 1,
        2232 => 1,
        2233 => 1,
        2234 => 1,
        2235 => 1,
        2236 => 1,
        2237 => 1,
        2238 => 1,
        2239 => 1,
        2240 => 1,
        2241 => 1,
        2242 => 1,
        2243 => 1,
        2244 => 1,
        2245 => 1,
        2246 => 1,
        2247 => 1,
        2248 => 1,
        2249 => 1,
        2250 => 1,
        2251 => 1,
        2252 => 1,
        2253 => 1,
        2254 => 1,
        2255 => 1,
        2256 => 1,
        2257 => 1,
        2258 => 1,
        2259 => 1,
        2260 => 1,
        2261 => 1,
        2262 => 1,
        2263 => 1,
        2264 => 1,
        2265 => 1,
        2266 => 1,
        2267 => 1,
        2268 => 1,
        2269 => 1,
        2270 => 1,
        2271 => 1,
        2272 => 1,
        2273 => 1,
        2275 => 1,
        2276 => 1,
        2277 => 1,
        2278 => 1,
        2279 => 1,
        2280 => 1,
        2281 => 1,
        2282 => 1,
        2283 => 1,
        2284 => 1,
        2285 => 1,
        2286 => 1,
        2287 => 1,
        2288 => 1,
        2289 => 1,
        2290 => 1,
        2291 => 1,
        2292 => 1,
        2293 => 1,
        2294 => 1,
        2295 => 1,
        2296 => 1,
        2297 => 1,
        2298 => 1,
        2299 => 1,
        2300 => 1,
        2301 => 1,
        2302 => 1,
        2303 => 1,
        2304 => 1,
        2305 => 1,
        2306 => 1,
        2307 => 1,
        2308 => 1,
        2362 => 1,
        2363 => 1,
        2364 => 1,
        2365 => 1,
        2366 => 1,
        2367 => 1,
        2368 => 1,
        2369 => 1,
        2370 => 1,
        2371 => 1,
        2372 => 1,
        2373 => 1,
        2374 => 1,
        2375 => 1,
        2376 => 1,
        2377 => 1,
        2378 => 1,
        2379 => 1,
        2380 => 1,
        2381 => 1,
        2382 => 1,
        2383 => 1,
        2384 => 1,
        2385 => 1,
        2386 => 1,
        2387 => 1,
        2388 => 1,
        2389 => 1,
        2390 => 1,
        2391 => 1,
        2400 => 1,
        2401 => 1,
        2402 => 1,
        2403 => 1,
        2404 => 2,
        2405 => 2,
        2406 => 1,
        2407 => 1,
        2408 => 1,
        2409 => 1,
        2410 => 1,
        2411 => 1,
        2412 => 1,
        2413 => 1,
        2414 => 1,
        2415 => 1,
        2416 => 2,
        2417 => 1,
        2418 => 1,
        2419 => 1,
        2420 => 1,
        2421 => 1,
        2422 => 1,
        2423 => 1,
        2424 => 1,
        2425 => 1,
        2426 => 1,
        2427 => 1,
        2428 => 1,
        2429 => 1,
        2430 => 1,
        2431 => 1,
        2432 => 1,
        2433 => 1,
        2434 => 1,
        2435 => 1,
        2437 => 1,
        2438 => 1,
        2439 => 1,
        2440 => 1,
        2441 => 1,
        2442 => 1,
        2443 => 1,
        2444 => 1,
        2447 => 1,
        2448 => 1,
        2451 => 1,
        2452 => 1,
        2453 => 1,
        2454 => 1,
        2455 => 1,
        2456 => 1,
        2457 => 1,
        2458 => 1,
        2459 => 1,
        2460 => 1,
        2461 => 1,
        2462 => 1,
        2463 => 1,
        2464 => 1,
        2465 => 1,
        2466 => 1,
        2467 => 1,
        2468 => 1,
        2469 => 1,
        2470 => 1,
        2471 => 1,
        2472 => 1,
        2474 => 1,
        2475 => 1,
        2476 => 1,
        2477 => 1,
        2478 => 1,
        2479 => 1,
        2480 => 1,
        2482 => 1,
        2486 => 1,
        2487 => 1,
        2488 => 1,
        2489 => 1,
        2492 => 1,
        2493 => 1,
        2494 => 1,
        2495 => 1,
        2496 => 1,
        2497 => 1,
        2498 => 1,
        2499 => 1,
        2500 => 1,
        2503 => 1,
        2504 => 1,
        2507 => 1,
        2508 => 1,
        2509 => 1,
        2510 => 1,
        2519 => 1,
        2528 => 1,
        2529 => 1,
        2530 => 1,
        2531 => 1,
        2534 => 1,
        2535 => 1,
        2536 => 1,
        2537 => 1,
        2538 => 1,
        2539 => 1,
        2540 => 1,
        2541 => 1,
        2542 => 1,
        2543 => 1,
        2544 => 1,
        2545 => 1,
        2546 => 2,
        2547 => 2,
        2548 => 2,
        2549 => 2,
        2550 => 2,
        2551 => 2,
        2552 => 2,
        2553 => 2,
        2554 => 2,
        2555 => 2,
        2556 => 1,
        2557 => 2,
        2558 => 1,
        2561 => 1,
        2562 => 1,
        2563 => 1,
        2565 => 1,
        2566 => 1,
        2567 => 1,
        2568 => 1,
        2569 => 1,
        2570 => 1,
        2575 => 1,
        2576 => 1,
        2579 => 1,
        2580 => 1,
        2581 => 1,
        2582 => 1,
        2583 => 1,
        2584 => 1,
        2585 => 1,
        2586 => 1,
        2587 => 1,
        2588 => 1,
        2589 => 1,
        2590 => 1,
        2591 => 1,
        2592 => 1,
        2593 => 1,
        2594 => 1,
        2595 => 1,
        2596 => 1,
        2597 => 1,
        2598 => 1,
        2599 => 1,
        2600 => 1,
        2602 => 1,
        2603 => 1,
        2604 => 1,
        2605 => 1,
        2606 => 1,
        2607 => 1,
        2608 => 1,
        2610 => 1,
        2613 => 1,
        2616 => 1,
        2617 => 1,
        2620 => 1,
        2622 => 1,
        2623 => 1,
        2624 => 1,
        2625 => 1,
        2626 => 1,
        2631 => 1,
        2632 => 1,
        2635 => 1,
        2636 => 1,
        2637 => 1,
        2641 => 1,
        2652 => 1,
        2662 => 1,
        2663 => 1,
        2664 => 1,
        2665 => 1,
        2666 => 1,
        2667 => 1,
        2668 => 1,
        2669 => 1,
        2670 => 1,
        2671 => 1,
        2672 => 1,
        2673 => 1,
        2674 => 1,
        2675 => 1,
        2676 => 1,
        2677 => 1,
        2678 => 2,
        2689 => 1,
        2690 => 1,
        2691 => 1,
        2693 => 1,
        2694 => 1,
        2695 => 1,
        2696 => 1,
        2697 => 1,
        2698 => 1,
        2699 => 1,
        2700 => 1,
        2701 => 1,
        2703 => 1,
        2704 => 1,
        2705 => 1,
        2707 => 1,
        2708 => 1,
        2709 => 1,
        2710 => 1,
        2711 => 1,
        2712 => 1,
        2713 => 1,
        2714 => 1,
        2715 => 1,
        2716 => 1,
        2717 => 1,
        2718 => 1,
        2719 => 1,
        2720 => 1,
        2721 => 1,
        2722 => 1,
        2723 => 1,
        2724 => 1,
        2725 => 1,
        2726 => 1,
        2727 => 1,
        2728 => 1,
        2730 => 1,
        2731 => 1,
        2732 => 1,
        2733 => 1,
        2734 => 1,
        2735 => 1,
        2736 => 1,
        2738 => 1,
        2739 => 1,
        2741 => 1,
        2742 => 1,
        2743 => 1,
        2744 => 1,
        2745 => 1,
        2748 => 1,
        2749 => 1,
        2750 => 1,
        2751 => 1,
        2752 => 1,
        2753 => 1,
        2754 => 1,
        2755 => 1,
        2756 => 1,
        2757 => 1,
        2759 => 1,
        2760 => 1,
        2761 => 1,
        2763 => 1,
        2764 => 1,
        2765 => 1,
        2768 => 1,
        2784 => 1,
        2785 => 1,
        2786 => 1,
        2787 => 1,
        2790 => 1,
        2791 => 1,
        2792 => 1,
        2793 => 1,
        2794 => 1,
        2795 => 1,
        2796 => 1,
        2797 => 1,
        2798 => 1,
        2799 => 1,
        2800 => 2,
        2801 => 2,
        2809 => 1,
        2810 => 1,
        2811 => 1,
        2812 => 1,
        2813 => 1,
        2814 => 1,
        2815 => 1,
        2817 => 1,
        2818 => 1,
        2819 => 1,
        2821 => 1,
        2822 => 1,
        2823 => 1,
        2824 => 1,
        2825 => 1,
        2826 => 1,
        2827 => 1,
        2828 => 1,
        2831 => 1,
        2832 => 1,
        2835 => 1,
        2836 => 1,
        2837 => 1,
        2838 => 1,
        2839 => 1,
        2840 => 1,
        2841 => 1,
        2842 => 1,
        2843 => 1,
        2844 => 1,
        2845 => 1,
        2846 => 1,
        2847 => 1,
        2848 => 1,
        2849 => 1,
        2850 => 1,
        2851 => 1,
        2852 => 1,
        2853 => 1,
        2854 => 1,
        2855 => 1,
        2856 => 1,
        2858 => 1,
        2859 => 1,
        2860 => 1,
        2861 => 1,
        2862 => 1,
        2863 => 1,
        2864 => 1,
        2866 => 1,
        2867 => 1,
        2869 => 1,
        2870 => 1,
        2871 => 1,
        2872 => 1,
        2873 => 1,
        2876 => 1,
        2877 => 1,
        2878 => 1,
        2879 => 1,
        2880 => 1,
        2881 => 1,
        2882 => 1,
        2883 => 1,
        2884 => 1,
        2887 => 1,
        2888 => 1,
        2891 => 1,
        2892 => 1,
        2893 => 1,
        2901 => 1,
        2902 => 1,
        2903 => 1,
        2911 => 1,
        2912 => 1,
        2913 => 1,
        2914 => 1,
        2915 => 1,
        2918 => 1,
        2919 => 1,
        2920 => 1,
        2921 => 1,
        2922 => 1,
        2923 => 1,
        2924 => 1,
        2925 => 1,
        2926 => 1,
        2927 => 1,
        2928 => 2,
        2929 => 1,
        2930 => 2,
        2931 => 2,
        2932 => 2,
        2933 => 2,
        2934 => 2,
        2935 => 2,
        2946 => 1,
        2947 => 1,
        2949 => 1,
        2950 => 1,
        2951 => 1,
        2952 => 1,
        2953 => 1,
        2954 => 1,
        2958 => 1,
        2959 => 1,
        2960 => 1,
        2962 => 1,
        2963 => 1,
        2964 => 1,
        2965 => 1,
        2969 => 1,
        2970 => 1,
        2972 => 1,
        2974 => 1,
        2975 => 1,
        2979 => 1,
        2980 => 1,
        2984 => 1,
        2985 => 1,
        2986 => 1,
        2990 => 1,
        2991 => 1,
        2992 => 1,
        2993 => 1,
        2994 => 1,
        2995 => 1,
        2996 => 1,
        2997 => 1,
        2998 => 1,
        2999 => 1,
        3000 => 1,
        3001 => 1,
        3006 => 1,
        3007 => 1,
        3008 => 1,
        3009 => 1,
        3010 => 1,
        3014 => 1,
        3015 => 1,
        3016 => 1,
        3018 => 1,
        3019 => 1,
        3020 => 1,
        3021 => 1,
        3024 => 1,
        3031 => 1,
        3046 => 1,
        3047 => 1,
        3048 => 1,
        3049 => 1,
        3050 => 1,
        3051 => 1,
        3052 => 1,
        3053 => 1,
        3054 => 1,
        3055 => 1,
        3056 => 2,
        3057 => 2,
        3058 => 2,
        3059 => 2,
        3060 => 2,
        3061 => 2,
        3062 => 2,
        3063 => 2,
        3064 => 2,
        3065 => 2,
        3066 => 2,
        3072 => 1,
        3073 => 1,
        3074 => 1,
        3075 => 1,
        3076 => 1,
        3077 => 1,
        3078 => 1,
        3079 => 1,
        3080 => 1,
        3081 => 1,
        3082 => 1,
        3083 => 1,
        3084 => 1,
        3086 => 1,
        3087 => 1,
        3088 => 1,
        3090 => 1,
        3091 => 1,
        3092 => 1,
        3093 => 1,
        3094 => 1,
        3095 => 1,
        3096 => 1,
        3097 => 1,
        3098 => 1,
        3099 => 1,
        3100 => 1,
        3101 => 1,
        3102 => 1,
        3103 => 1,
        3104 => 1,
        3105 => 1,
        3106 => 1,
        3107 => 1,
        3108 => 1,
        3109 => 1,
        3110 => 1,
        3111 => 1,
        3112 => 1,
        3114 => 1,
        3115 => 1,
        3116 => 1,
        3117 => 1,
        3118 => 1,
        3119 => 1,
        3120 => 1,
        3121 => 1,
        3122 => 1,
        3123 => 1,
        3124 => 1,
        3125 => 1,
        3126 => 1,
        3127 => 1,
        3128 => 1,
        3129 => 1,
        3132 => 1,
        3133 => 1,
        3134 => 1,
        3135 => 1,
        3136 => 1,
        3137 => 1,
        3138 => 1,
        3139 => 1,
        3140 => 1,
        3142 => 1,
        3143 => 1,
        3144 => 1,
        3146 => 1,
        3147 => 1,
        3148 => 1,
        3149 => 1,
        3157 => 1,
        3158 => 1,
        3160 => 1,
        3161 => 1,
        3162 => 1,
        3165 => 1,
        3168 => 1,
        3169 => 1,
        3170 => 1,
        3171 => 1,
        3174 => 1,
        3175 => 1,
        3176 => 1,
        3177 => 1,
        3178 => 1,
        3179 => 1,
        3180 => 1,
        3181 => 1,
        3182 => 1,
        3183 => 1,
        3191 => 2,
        3192 => 2,
        3193 => 2,
        3194 => 2,
        3195 => 2,
        3196 => 2,
        3197 => 2,
        3198 => 2,
        3199 => 2,
        3200 => 1,
        3201 => 1,
        3202 => 1,
        3203 => 1,
        3204 => 2,
        3205 => 1,
        3206 => 1,
        3207 => 1,
        3208 => 1,
        3209 => 1,
        3210 => 1,
        3211 => 1,
        3212 => 1,
        3214 => 1,
        3215 => 1,
        3216 => 1,
        3218 => 1,
        3219 => 1,
        3220 => 1,
        3221 => 1,
        3222 => 1,
        3223 => 1,
        3224 => 1,
        3225 => 1,
        3226 => 1,
        3227 => 1,
        3228 => 1,
        3229 => 1,
        3230 => 1,
        3231 => 1,
        3232 => 1,
        3233 => 1,
        3234 => 1,
        3235 => 1,
        3236 => 1,
        3237 => 1,
        3238 => 1,
        3239 => 1,
        3240 => 1,
        3242 => 1,
        3243 => 1,
        3244 => 1,
        3245 => 1,
        3246 => 1,
        3247 => 1,
        3248 => 1,
        3249 => 1,
        3250 => 1,
        3251 => 1,
        3253 => 1,
        3254 => 1,
        3255 => 1,
        3256 => 1,
        3257 => 1,
        3260 => 1,
        3261 => 1,
        3262 => 1,
        3263 => 1,
        3264 => 1,
        3265 => 1,
        3266 => 1,
        3267 => 1,
        3268 => 1,
        3270 => 1,
        3271 => 1,
        3272 => 1,
        3274 => 1,
        3275 => 1,
        3276 => 1,
        3277 => 1,
        3285 => 1,
        3286 => 1,
        3293 => 1,
        3294 => 1,
        3296 => 1,
        3297 => 1,
        3298 => 1,
        3299 => 1,
        3302 => 1,
        3303 => 1,
        3304 => 1,
        3305 => 1,
        3306 => 1,
        3307 => 1,
        3308 => 1,
        3309 => 1,
        3310 => 1,
        3311 => 1,
        3313 => 1,
        3314 => 1,
        3315 => 1,
        3328 => 1,
        3329 => 1,
        3330 => 1,
        3331 => 1,
        3332 => 1,
        3333 => 1,
        3334 => 1,
        3335 => 1,
        3336 => 1,
        3337 => 1,
        3338 => 1,
        3339 => 1,
        3340 => 1,
        3342 => 1,
        3343 => 1,
        3344 => 1,
        3346 => 1,
        3347 => 1,
        3348 => 1,
        3349 => 1,
        3350 => 1,
        3351 => 1,
        3352 => 1,
        3353 => 1,
        3354 => 1,
        3355 => 1,
        3356 => 1,
        3357 => 1,
        3358 => 1,
        3359 => 1,
        3360 => 1,
        3361 => 1,
        3362 => 1,
        3363 => 1,
        3364 => 1,
        3365 => 1,
        3366 => 1,
        3367 => 1,
        3368 => 1,
        3369 => 1,
        3370 => 1,
        3371 => 1,
        3372 => 1,
        3373 => 1,
        3374 => 1,
        3375 => 1,
        3376 => 1,
        3377 => 1,
        3378 => 1,
        3379 => 1,
        3380 => 1,
        3381 => 1,
        3382 => 1,
        3383 => 1,
        3384 => 1,
        3385 => 1,
        3386 => 1,
        3387 => 1,
        3388 => 1,
        3389 => 1,
        3390 => 1,
        3391 => 1,
        3392 => 1,
        3393 => 1,
        3394 => 1,
        3395 => 1,
        3396 => 1,
        3398 => 1,
        3399 => 1,
        3400 => 1,
        3402 => 1,
        3403 => 1,
        3404 => 1,
        3405 => 1,
        3406 => 1,
        3407 => 2,
        3412 => 1,
        3413 => 1,
        3414 => 1,
        3415 => 1,
        3416 => 2,
        3417 => 2,
        3418 => 2,
        3419 => 2,
        3420 => 2,
        3421 => 2,
        3422 => 2,
        3423 => 1,
        3424 => 1,
        3425 => 1,
        3426 => 1,
        3427 => 1,
        3430 => 1,
        3431 => 1,
        3432 => 1,
        3433 => 1,
        3434 => 1,
        3435 => 1,
        3436 => 1,
        3437 => 1,
        3438 => 1,
        3439 => 1,
        3440 => 2,
        3441 => 2,
        3442 => 2,
        3443 => 2,
        3444 => 2,
        3445 => 2,
        3446 => 2,
        3447 => 2,
        3448 => 2,
        3449 => 2,
        3450 => 1,
        3451 => 1,
        3452 => 1,
        3453 => 1,
        3454 => 1,
        3455 => 1,
        3457 => 1,
        3458 => 1,
        3459 => 1,
        3461 => 1,
        3462 => 1,
        3463 => 1,
        3464 => 1,
        3465 => 1,
        3466 => 1,
        3467 => 1,
        3468 => 1,
        3469 => 1,
        3470 => 1,
        3471 => 1,
        3472 => 1,
        3473 => 1,
        3474 => 1,
        3475 => 1,
        3476 => 1,
        3477 => 1,
        3478 => 1,
        3482 => 1,
        3483 => 1,
        3484 => 1,
        3485 => 1,
        3486 => 1,
        3487 => 1,
        3488 => 1,
        3489 => 1,
        3490 => 1,
        3491 => 1,
        3492 => 1,
        3493 => 1,
        3494 => 1,
        3495 => 1,
        3496 => 1,
        3497 => 1,
        3498 => 1,
        3499 => 1,
        3500 => 1,
        3501 => 1,
        3502 => 1,
        3503 => 1,
        3504 => 1,
        3505 => 1,
        3507 => 1,
        3508 => 1,
        3509 => 1,
        3510 => 1,
        3511 => 1,
        3512 => 1,
        3513 => 1,
        3514 => 1,
        3515 => 1,
        3517 => 1,
        3520 => 1,
        3521 => 1,
        3522 => 1,
        3523 => 1,
        3524 => 1,
        3525 => 1,
        3526 => 1,
        3530 => 1,
        3535 => 1,
        3536 => 1,
        3537 => 1,
        3538 => 1,
        3539 => 1,
        3540 => 1,
        3542 => 1,
        3544 => 1,
        3545 => 1,
        3546 => 1,
        3547 => 1,
        3548 => 1,
        3549 => 1,
        3550 => 1,
        3551 => 1,
        3558 => 1,
        3559 => 1,
        3560 => 1,
        3561 => 1,
        3562 => 1,
        3563 => 1,
        3564 => 1,
        3565 => 1,
        3566 => 1,
        3567 => 1,
        3570 => 1,
        3571 => 1,
        3572 => 2,
        3636 => 1,
        3637 => 1,
        3638 => 1,
        3639 => 1,
        3640 => 1,
        3641 => 1,
        3642 => 1,
        3647 => 2,
        3648 => 1,
        3649 => 1,
        3650 => 1,
        3651 => 1,
        3652 => 1,
        3653 => 1,
        3654 => 1,
        3655 => 1,
        3656 => 1,
        3657 => 1,
        3658 => 1,
        3659 => 1,
        3660 => 1,
        3661 => 1,
        3662 => 1,
        3663 => 2,
        3664 => 1,
        3665 => 1,
        3666 => 1,
        3667 => 1,
        3668 => 1,
        3669 => 1,
        3670 => 1,
        3671 => 1,
        3672 => 1,
        3673 => 1,
        3674 => 2,
        3675 => 2,
        3713 => 1,
        3714 => 1,
        3716 => 1,
        3718 => 1,
        3719 => 1,
        3720 => 1,
        3721 => 1,
        3722 => 1,
        3724 => 1,
        3725 => 1,
        3726 => 1,
        3727 => 1,
        3728 => 1,
        3729 => 1,
        3730 => 1,
        3731 => 1,
        3732 => 1,
        3733 => 1,
        3734 => 1,
        3735 => 1,
        3736 => 1,
        3737 => 1,
        3738 => 1,
        3739 => 1,
        3740 => 1,
        3741 => 1,
        3742 => 1,
        3743 => 1,
        3744 => 1,
        3745 => 1,
        3746 => 1,
        3747 => 1,
        3749 => 1,
        3751 => 1,
        3752 => 1,
        3753 => 1,
        3754 => 1,
        3755 => 1,
        3756 => 1,
        3757 => 1,
        3758 => 1,
        3759 => 1,
        3760 => 1,
        3761 => 1,
        3762 => 1,
        3764 => 1,
        3765 => 1,
        3766 => 1,
        3767 => 1,
        3768 => 1,
        3769 => 1,
        3770 => 1,
        3771 => 1,
        3772 => 1,
        3773 => 1,
        3776 => 1,
        3777 => 1,
        3778 => 1,
        3779 => 1,
        3780 => 1,
        3782 => 1,
        3784 => 1,
        3785 => 1,
        3786 => 1,
        3787 => 1,
        3788 => 1,
        3789 => 1,
        3790 => 1,
        3792 => 1,
        3793 => 1,
        3794 => 1,
        3795 => 1,
        3796 => 1,
        3797 => 1,
        3798 => 1,
        3799 => 1,
        3800 => 1,
        3801 => 1,
        3806 => 1,
        3807 => 1,
        3840 => 1,
        3841 => 2,
        3842 => 2,
        3843 => 2,
        3844 => 2,
        3845 => 2,
        3846 => 2,
        3847 => 2,
        3848 => 2,
        3849 => 2,
        3850 => 2,
        3851 => 1,
        3853 => 2,
        3854 => 2,
        3855 => 2,
        3856 => 2,
        3857 => 2,
        3858 => 2,
        3859 => 2,
        3860 => 2,
        3861 => 2,
        3862 => 2,
        3863 => 2,
        3864 => 1,
        3865 => 1,
        3866 => 2,
        3867 => 2,
        3868 => 2,
        3869 => 2,
        3870 => 2,
        3871 => 2,
        3872 => 1,
        3873 => 1,
        3874 => 1,
        3875 => 1,
        3876 => 1,
        3877 => 1,
        3878 => 1,
        3879 => 1,
        3880 => 1,
        3881 => 1,
        3882 => 2,
        3883 => 2,
        3884 => 2,
        3885 => 2,
        3886 => 2,
        3887 => 2,
        3888 => 2,
        3889 => 2,
        3890 => 2,
        3891 => 2,
        3892 => 2,
        3893 => 1,
        3894 => 2,
        3895 => 1,
        3896 => 2,
        3897 => 1,
        3898 => 2,
        3899 => 2,
        3900 => 2,
        3901 => 2,
        3902 => 1,
        3903 => 1,
        3904 => 1,
        3905 => 1,
        3906 => 1,
        3908 => 1,
        3909 => 1,
        3910 => 1,
        3911 => 1,
        3913 => 1,
        3914 => 1,
        3915 => 1,
        3916 => 1,
        3918 => 1,
        3919 => 1,
        3920 => 1,
        3921 => 1,
        3923 => 1,
        3924 => 1,
        3925 => 1,
        3926 => 1,
        3928 => 1,
        3929 => 1,
        3930 => 1,
        3931 => 1,
        3933 => 1,
        3934 => 1,
        3935 => 1,
        3936 => 1,
        3937 => 1,
        3938 => 1,
        3939 => 1,
        3940 => 1,
        3941 => 1,
        3942 => 1,
        3943 => 1,
        3944 => 1,
        3946 => 1,
        3947 => 1,
        3948 => 1,
        3953 => 1,
        3954 => 1,
        3956 => 1,
        3962 => 1,
        3963 => 1,
        3964 => 1,
        3965 => 1,
        3966 => 1,
        3967 => 1,
        3968 => 1,
        3970 => 1,
        3971 => 1,
        3972 => 1,
        3973 => 2,
        3974 => 1,
        3975 => 1,
        3976 => 1,
        3977 => 1,
        3978 => 1,
        3979 => 1,
        3980 => 1,
        3981 => 1,
        3982 => 1,
        3983 => 1,
        3984 => 1,
        3985 => 1,
        3986 => 1,
        3988 => 1,
        3989 => 1,
        3990 => 1,
        3991 => 1,
        3993 => 1,
        3994 => 1,
        3995 => 1,
        3996 => 1,
        3998 => 1,
        3999 => 1,
        4000 => 1,
        4001 => 1,
        4003 => 1,
        4004 => 1,
        4005 => 1,
        4006 => 1,
        4008 => 1,
        4009 => 1,
        4010 => 1,
        4011 => 1,
        4013 => 1,
        4014 => 1,
        4015 => 1,
        4016 => 1,
        4017 => 1,
        4018 => 1,
        4019 => 1,
        4020 => 1,
        4021 => 1,
        4022 => 1,
        4023 => 1,
        4024 => 1,
        4026 => 1,
        4027 => 1,
        4028 => 1,
        4030 => 2,
        4031 => 2,
        4032 => 2,
        4033 => 2,
        4034 => 2,
        4035 => 2,
        4036 => 2,
        4037 => 2,
        4038 => 1,
        4039 => 2,
        4040 => 2,
        4041 => 2,
        4042 => 2,
        4043 => 2,
        4044 => 2,
        4046 => 2,
        4047 => 2,
        4048 => 2,
        4049 => 2,
        4050 => 2,
        4051 => 2,
        4052 => 2,
        4053 => 2,
        4054 => 2,
        4055 => 2,
        4056 => 2,
        4057 => 2,
        4058 => 2,
        4130 => 1,
        4131 => 1,
        4132 => 1,
        4133 => 1,
        4134 => 1,
        4135 => 1,
        4136 => 1,
        4137 => 1,
        4138 => 1,
        4139 => 1,
        4140 => 1,
        4141 => 1,
        4142 => 1,
        4143 => 1,
        4144 => 1,
        4145 => 1,
        4146 => 1,
        4147 => 1,
        4148 => 1,
        4149 => 1,
        4150 => 1,
        4151 => 1,
        4152 => 1,
        4153 => 1,
        4154 => 1,
        4155 => 1,
        4156 => 1,
        4157 => 1,
        4158 => 1,
        4159 => 1,
        4160 => 1,
        4161 => 1,
        4162 => 1,
        4163 => 1,
        4164 => 1,
        4165 => 1,
        4166 => 1,
        4167 => 1,
        4168 => 1,
        4169 => 1,
        4170 => 2,
        4171 => 2,
        4172 => 2,
        4173 => 2,
        4174 => 2,
        4175 => 2,
        4176 => 1,
        4177 => 1,
        4178 => 1,
        4179 => 1,
        4180 => 1,
        4181 => 1,
        4182 => 1,
        4183 => 1,
        4184 => 1,
        4185 => 1,
        4250 => 1,
        4251 => 1,
        4252 => 1,
        4253 => 1,
        4254 => 2,
        4255 => 2,
        4343 => 1,
        4344 => 1,
        4345 => 1,
        4346 => 1,
        4347 => 2,
        4349 => 1,
        4350 => 1,
        4351 => 1,
        4442 => 2,
        4443 => 2,
        4444 => 2,
        4445 => 2,
        4446 => 2,
        4515 => 2,
        4516 => 2,
        4517 => 2,
        4518 => 2,
        4519 => 2,
        4602 => 2,
        4603 => 2,
        4604 => 2,
        4605 => 2,
        4606 => 2,
        4607 => 2,
        4608 => 1,
        4609 => 1,
        4610 => 1,
        4611 => 1,
        4612 => 1,
        4613 => 1,
        4614 => 1,
        4615 => 1,
        4679 => 1,
        4680 => 1,
        4682 => 1,
        4683 => 1,
        4684 => 1,
        4685 => 1,
        4688 => 1,
        4689 => 1,
        4690 => 1,
        4691 => 1,
        4692 => 1,
        4693 => 1,
        4694 => 1,
        4696 => 1,
        4698 => 1,
        4699 => 1,
        4700 => 1,
        4701 => 1,
        4743 => 1,
        4744 => 1,
        4746 => 1,
        4747 => 1,
        4748 => 1,
        4749 => 1,
        4783 => 1,
        4784 => 1,
        4786 => 1,
        4787 => 1,
        4788 => 1,
        4789 => 1,
        4792 => 1,
        4793 => 1,
        4794 => 1,
        4795 => 1,
        4796 => 1,
        4797 => 1,
        4798 => 1,
        4800 => 1,
        4802 => 1,
        4803 => 1,
        4804 => 1,
        4805 => 1,
        4808 => 1,
        4809 => 1,
        4810 => 1,
        4811 => 1,
        4812 => 1,
        4813 => 1,
        4814 => 1,
        4815 => 1,
        4816 => 1,
        4817 => 1,
        4818 => 1,
        4819 => 1,
        4820 => 1,
        4821 => 1,
        4822 => 1,
        4824 => 1,
        4825 => 1,
        4826 => 1,
        4827 => 1,
        4828 => 1,
        4829 => 1,
        4830 => 1,
        4831 => 1,
        4832 => 1,
        4833 => 1,
        4834 => 1,
        4835 => 1,
        4836 => 1,
        4837 => 1,
        4838 => 1,
        4839 => 1,
        4840 => 1,
        4841 => 1,
        4842 => 1,
        4843 => 1,
        4844 => 1,
        4845 => 1,
        4846 => 1,
        4847 => 1,
        4879 => 1,
        4880 => 1,
        4882 => 1,
        4883 => 1,
        4884 => 1,
        4885 => 1,
        4888 => 1,
        4889 => 1,
        4890 => 1,
        4891 => 1,
        4892 => 1,
        4893 => 1,
        4894 => 1,
        4895 => 1,
        4935 => 1,
        4936 => 1,
        4937 => 1,
        4938 => 1,
        4939 => 1,
        4940 => 1,
        4941 => 1,
        4942 => 1,
        4943 => 1,
        4944 => 1,
        4945 => 1,
        4946 => 1,
        4947 => 1,
        4948 => 1,
        4949 => 1,
        4950 => 1,
        4951 => 1,
        4952 => 1,
        4953 => 1,
        4954 => 1,
        4957 => 1,
        4958 => 1,
        4959 => 1,
        4960 => 2,
        4992 => 1,
        4993 => 1,
        4994 => 1,
        4995 => 1,
        4996 => 1,
        4997 => 1,
        4998 => 1,
        4999 => 1,
        5000 => 1,
        5001 => 1,
        5002 => 1,
        5003 => 1,
        5004 => 1,
        5005 => 1,
        5006 => 1,
        5007 => 1,
        5008 => 2,
        5009 => 2,
        5010 => 2,
        5011 => 2,
        5012 => 2,
        5013 => 2,
        5014 => 2,
        5015 => 2,
        5016 => 2,
        5017 => 2,
        5109 => 1,
        5120 => 2,
        5741 => 2,
        5742 => 2,
        5743 => 1,
        5744 => 1,
        5745 => 1,
        5746 => 1,
        5747 => 1,
        5748 => 1,
        5749 => 1,
        5750 => 1,
        5751 => 1,
        5752 => 1,
        5753 => 1,
        5754 => 1,
        5755 => 1,
        5756 => 1,
        5757 => 1,
        5758 => 1,
        5759 => 1,
        5761 => 1,
        5762 => 1,
        5763 => 1,
        5764 => 1,
        5765 => 1,
        5766 => 1,
        5767 => 1,
        5768 => 1,
        5769 => 1,
        5770 => 1,
        5771 => 1,
        5772 => 1,
        5773 => 1,
        5774 => 1,
        5775 => 1,
        5776 => 1,
        5777 => 1,
        5778 => 1,
        5779 => 1,
        5780 => 1,
        5781 => 1,
        5782 => 1,
        5783 => 1,
        5784 => 1,
        5785 => 1,
        5786 => 1,
        5787 => 2,
        5788 => 2,
        5867 => 2,
        5868 => 2,
        5869 => 2,
        5870 => 2,
        5871 => 2,
        5872 => 2,
        5873 => 1,
        5874 => 1,
        5875 => 1,
        5876 => 1,
        5877 => 1,
        5878 => 1,
        5879 => 1,
        5880 => 1,
        5888 => 1,
        5889 => 1,
        5890 => 1,
        5891 => 1,
        5892 => 1,
        5893 => 1,
        5894 => 1,
        5895 => 1,
        5896 => 1,
        5897 => 1,
        5898 => 1,
        5899 => 1,
        5900 => 1,
        5901 => 1,
        5902 => 1,
        5903 => 1,
        5904 => 1,
        5905 => 1,
        5906 => 1,
        5907 => 1,
        5908 => 1,
        5909 => 1,
        5919 => 1,
        5920 => 1,
        5921 => 1,
        5922 => 1,
        5923 => 1,
        5924 => 1,
        5925 => 1,
        5926 => 1,
        5927 => 1,
        5928 => 1,
        5929 => 1,
        5930 => 1,
        5931 => 1,
        5932 => 1,
        5933 => 1,
        5934 => 1,
        5935 => 1,
        5936 => 1,
        5937 => 1,
        5938 => 1,
        5939 => 1,
        5940 => 1,
        5941 => 2,
        5942 => 2,
        5952 => 1,
        5953 => 1,
        5954 => 1,
        5955 => 1,
        5956 => 1,
        5957 => 1,
        5958 => 1,
        5959 => 1,
        5960 => 1,
        5961 => 1,
        5962 => 1,
        5963 => 1,
        5964 => 1,
        5965 => 1,
        5966 => 1,
        5967 => 1,
        5968 => 1,
        5969 => 1,
        5970 => 1,
        5971 => 1,
        5984 => 1,
        5985 => 1,
        5986 => 1,
        5987 => 1,
        5988 => 1,
        5989 => 1,
        5990 => 1,
        5991 => 1,
        5992 => 1,
        5993 => 1,
        5994 => 1,
        5995 => 1,
        5996 => 1,
        5998 => 1,
        5999 => 1,
        6000 => 1,
        6002 => 1,
        6003 => 1,
        6100 => 2,
        6101 => 2,
        6102 => 2,
        6103 => 1,
        6104 => 2,
        6105 => 2,
        6106 => 2,
        6107 => 2,
        6108 => 1,
        6109 => 1,
        6112 => 1,
        6113 => 1,
        6114 => 1,
        6115 => 1,
        6116 => 1,
        6117 => 1,
        6118 => 1,
        6119 => 1,
        6120 => 1,
        6121 => 1,
        6128 => 2,
        6129 => 2,
        6130 => 2,
        6131 => 2,
        6132 => 2,
        6133 => 2,
        6134 => 2,
        6135 => 2,
        6136 => 2,
        6137 => 2,
        6144 => 2,
        6145 => 2,
        6146 => 2,
        6147 => 2,
        6148 => 2,
        6149 => 2,
        6151 => 2,
        6152 => 2,
        6153 => 2,
        6154 => 2,
        6160 => 1,
        6161 => 1,
        6162 => 1,
        6163 => 1,
        6164 => 1,
        6165 => 1,
        6166 => 1,
        6167 => 1,
        6168 => 1,
        6169 => 1,
        6264 => 1,
        6314 => 1,
        6429 => 1,
        6430 => 1,
        6432 => 1,
        6433 => 1,
        6434 => 1,
        6435 => 1,
        6436 => 1,
        6437 => 1,
        6438 => 1,
        6439 => 1,
        6440 => 1,
        6441 => 1,
        6442 => 1,
        6443 => 1,
        6448 => 1,
        6449 => 1,
        6450 => 1,
        6451 => 1,
        6452 => 1,
        6453 => 1,
        6454 => 1,
        6455 => 1,
        6456 => 1,
        6457 => 1,
        6458 => 1,
        6459 => 1,
        6464 => 2,
        6468 => 2,
        6469 => 2,
        6512 => 1,
        6513 => 1,
        6514 => 1,
        6515 => 1,
        6516 => 1,
        6570 => 1,
        6571 => 1,
        6576 => 1,
        6577 => 1,
        6578 => 1,
        6579 => 1,
        6580 => 1,
        6581 => 1,
        6582 => 1,
        6583 => 1,
        6584 => 1,
        6585 => 1,
        6586 => 1,
        6587 => 1,
        6588 => 1,
        6589 => 1,
        6590 => 1,
        6591 => 1,
        6592 => 1,
        6593 => 1,
        6594 => 1,
        6595 => 1,
        6596 => 1,
        6597 => 1,
        6598 => 1,
        6599 => 1,
        6600 => 1,
        6601 => 1,
        6608 => 1,
        6609 => 1,
        6610 => 1,
        6611 => 1,
        6612 => 1,
        6613 => 1,
        6614 => 1,
        6615 => 1,
        6616 => 1,
        6617 => 1,
        6618 => 3,
        6622 => 2,
        6623 => 2,
        6686 => 2,
        6687 => 2,
        6783 => 1,
        6784 => 1,
        6785 => 1,
        6786 => 1,
        6787 => 1,
        6788 => 1,
        6789 => 1,
        6790 => 1,
        6791 => 1,
        6792 => 1,
        6793 => 1,
        6800 => 1,
        6801 => 1,
        6802 => 1,
        6803 => 1,
        6804 => 1,
        6805 => 1,
        6806 => 1,
        6807 => 1,
        6808 => 1,
        6809 => 1,
        6816 => 2,
        6817 => 2,
        6818 => 2,
        6819 => 2,
        6820 => 2,
        6821 => 2,
        6822 => 2,
        6823 => 1,
        6824 => 2,
        6825 => 2,
        6826 => 2,
        6827 => 2,
        6828 => 2,
        6829 => 2,
        6832 => 1,
        6833 => 1,
        6834 => 1,
        6835 => 1,
        6836 => 1,
        6837 => 1,
        6838 => 1,
        6839 => 1,
        6840 => 1,
        6841 => 1,
        6842 => 1,
        6843 => 1,
        6844 => 1,
        6845 => 1,
        6846 => 2,
        6847 => 1,
        6848 => 1,
        6849 => 1,
        6850 => 1,
        6851 => 1,
        6852 => 1,
        6853 => 1,
        6854 => 1,
        6855 => 1,
        6856 => 1,
        6857 => 1,
        6858 => 1,
        6859 => 1,
        6860 => 1,
        6861 => 1,
        6862 => 1,
        6988 => 1,
        6992 => 1,
        6993 => 1,
        6994 => 1,
        6995 => 1,
        6996 => 1,
        6997 => 1,
        6998 => 1,
        6999 => 1,
        7000 => 1,
        7001 => 1,
        7002 => 2,
        7003 => 2,
        7004 => 2,
        7005 => 2,
        7006 => 2,
        7007 => 2,
        7008 => 2,
        7009 => 2,
        7010 => 2,
        7011 => 2,
        7012 => 2,
        7013 => 2,
        7014 => 2,
        7015 => 2,
        7016 => 2,
        7017 => 2,
        7018 => 2,
        7019 => 1,
        7020 => 1,
        7021 => 1,
        7022 => 1,
        7023 => 1,
        7024 => 1,
        7025 => 1,
        7026 => 1,
        7027 => 1,
        7028 => 2,
        7029 => 2,
        7030 => 2,
        7031 => 2,
        7032 => 2,
        7033 => 2,
        7034 => 2,
        7035 => 2,
        7036 => 2,
        7037 => 2,
        7038 => 2,
        7083 => 1,
        7084 => 1,
        7085 => 1,
        7086 => 1,
        7087 => 1,
        7088 => 1,
        7089 => 1,
        7090 => 1,
        7091 => 1,
        7092 => 1,
        7093 => 1,
        7094 => 1,
        7095 => 1,
        7096 => 1,
        7097 => 1,
        7098 => 1,
        7099 => 1,
        7100 => 1,
        7101 => 1,
        7102 => 1,
        7103 => 1,
        7164 => 2,
        7165 => 2,
        7166 => 2,
        7167 => 2,
        7227 => 2,
        7228 => 2,
        7229 => 2,
        7230 => 2,
        7231 => 2,
        7232 => 1,
        7233 => 1,
        7234 => 1,
        7235 => 1,
        7236 => 1,
        7237 => 1,
        7238 => 1,
        7239 => 1,
        7240 => 1,
        7241 => 1,
        7294 => 2,
        7295 => 2,
        7360 => 2,
        7361 => 2,
        7362 => 2,
        7363 => 2,
        7364 => 2,
        7365 => 2,
        7366 => 2,
        7367 => 2,
        7376 => 1,
        7377 => 1,
        7378 => 1,
        7379 => 2,
        7411 => 1,
        7412 => 1,
        7413 => 1,
        7414 => 1,
        7415 => 1,
        7416 => 1,
        7417 => 1,
        7418 => 1,
        7471 => 1,
        7483 => 1,
        7502 => 1,
        7531 => 1,
        7532 => 1,
        7533 => 1,
        7534 => 1,
        7535 => 1,
        7536 => 1,
        7537 => 1,
        7538 => 1,
        7539 => 1,
        7540 => 1,
        7541 => 1,
        7542 => 1,
        7543 => 1,
        7616 => 1,
        7617 => 1,
        7618 => 1,
        7619 => 1,
        7620 => 1,
        7621 => 1,
        7622 => 1,
        7623 => 1,
        7624 => 1,
        7625 => 1,
        7626 => 1,
        7655 => 1,
        7656 => 1,
        7657 => 1,
        7658 => 1,
        7659 => 1,
        7660 => 1,
        7661 => 1,
        7662 => 1,
        7663 => 1,
        7664 => 1,
        7665 => 1,
        7666 => 1,
        7667 => 1,
        7668 => 1,
        7669 => 1,
        7670 => 1,
        7671 => 1,
        7672 => 1,
        7673 => 1,
        7674 => 1,
        7675 => 1,
        7676 => 1,
        7677 => 1,
        7678 => 1,
        7679 => 1,
        7681 => 1,
        7683 => 1,
        7685 => 1,
        7687 => 1,
        7689 => 1,
        7691 => 1,
        7693 => 1,
        7695 => 1,
        7697 => 1,
        7699 => 1,
        7701 => 1,
        7703 => 1,
        7705 => 1,
        7707 => 1,
        7709 => 1,
        7711 => 1,
        7713 => 1,
        7715 => 1,
        7717 => 1,
        7719 => 1,
        7721 => 1,
        7723 => 1,
        7725 => 1,
        7727 => 1,
        7729 => 1,
        7731 => 1,
        7733 => 1,
        7735 => 1,
        7737 => 1,
        7739 => 1,
        7741 => 1,
        7743 => 1,
        7745 => 1,
        7747 => 1,
        7749 => 1,
        7751 => 1,
        7753 => 1,
        7755 => 1,
        7757 => 1,
        7759 => 1,
        7761 => 1,
        7763 => 1,
        7765 => 1,
        7767 => 1,
        7769 => 1,
        7771 => 1,
        7773 => 1,
        7775 => 1,
        7777 => 1,
        7779 => 1,
        7781 => 1,
        7783 => 1,
        7785 => 1,
        7787 => 1,
        7789 => 1,
        7791 => 1,
        7793 => 1,
        7795 => 1,
        7797 => 1,
        7799 => 1,
        7801 => 1,
        7803 => 1,
        7805 => 1,
        7807 => 1,
        7809 => 1,
        7811 => 1,
        7813 => 1,
        7815 => 1,
        7817 => 1,
        7819 => 1,
        7821 => 1,
        7823 => 1,
        7825 => 1,
        7827 => 1,
        7829 => 1,
        7830 => 1,
        7831 => 1,
        7832 => 1,
        7833 => 1,
        7836 => 1,
        7837 => 1,
        7839 => 1,
        7841 => 1,
        7843 => 1,
        7845 => 1,
        7847 => 1,
        7849 => 1,
        7851 => 1,
        7853 => 1,
        7855 => 1,
        7857 => 1,
        7859 => 1,
        7861 => 1,
        7863 => 1,
        7865 => 1,
        7867 => 1,
        7869 => 1,
        7871 => 1,
        7873 => 1,
        7875 => 1,
        7877 => 1,
        7879 => 1,
        7881 => 1,
        7883 => 1,
        7885 => 1,
        7887 => 1,
        7889 => 1,
        7891 => 1,
        7893 => 1,
        7895 => 1,
        7897 => 1,
        7899 => 1,
        7901 => 1,
        7903 => 1,
        7905 => 1,
        7907 => 1,
        7909 => 1,
        7911 => 1,
        7913 => 1,
        7915 => 1,
        7917 => 1,
        7919 => 1,
        7921 => 1,
        7923 => 1,
        7925 => 1,
        7927 => 1,
        7929 => 1,
        7931 => 1,
        7933 => 1,
        7935 => 1,
        7936 => 1,
        7937 => 1,
        7938 => 1,
        7939 => 1,
        7940 => 1,
        7941 => 1,
        7942 => 1,
        7943 => 1,
        7952 => 1,
        7953 => 1,
        7954 => 1,
        7955 => 1,
        7956 => 1,
        7957 => 1,
        7968 => 1,
        7969 => 1,
        7970 => 1,
        7971 => 1,
        7972 => 1,
        7973 => 1,
        7974 => 1,
        7975 => 1,
        7984 => 1,
        7985 => 1,
        7986 => 1,
        7987 => 1,
        7988 => 1,
        7989 => 1,
        7990 => 1,
        7991 => 1,
        8000 => 1,
        8001 => 1,
        8002 => 1,
        8003 => 1,
        8004 => 1,
        8005 => 1,
        8016 => 1,
        8017 => 1,
        8018 => 1,
        8019 => 1,
        8020 => 1,
        8021 => 1,
        8022 => 1,
        8023 => 1,
        8032 => 1,
        8033 => 1,
        8034 => 1,
        8035 => 1,
        8036 => 1,
        8037 => 1,
        8038 => 1,
        8039 => 1,
        8048 => 1,
        8050 => 1,
        8052 => 1,
        8054 => 1,
        8056 => 1,
        8058 => 1,
        8060 => 1,
        8112 => 1,
        8113 => 1,
        8118 => 1,
        8134 => 1,
        8144 => 1,
        8145 => 1,
        8146 => 1,
        8150 => 1,
        8151 => 1,
        8160 => 1,
        8161 => 1,
        8162 => 1,
        8164 => 1,
        8165 => 1,
        8166 => 1,
        8167 => 1,
        8182 => 1,
        8208 => 2,
        8210 => 2,
        8211 => 2,
        8212 => 2,
        8213 => 2,
        8214 => 2,
        8216 => 2,
        8217 => 2,
        8218 => 2,
        8219 => 2,
        8220 => 2,
        8221 => 2,
        8222 => 2,
        8223 => 2,
        8224 => 2,
        8225 => 2,
        8226 => 2,
        8227 => 2,
        8231 => 2,
        8240 => 2,
        8241 => 2,
        8242 => 2,
        8245 => 2,
        8248 => 2,
        8249 => 2,
        8250 => 2,
        8251 => 2,
        8253 => 2,
        8255 => 2,
        8256 => 2,
        8257 => 2,
        8258 => 2,
        8259 => 2,
        8260 => 2,
        8261 => 2,
        8262 => 2,
        8266 => 2,
        8267 => 2,
        8268 => 2,
        8269 => 2,
        8270 => 2,
        8271 => 2,
        8272 => 2,
        8273 => 2,
        8274 => 2,
        8275 => 2,
        8276 => 2,
        8277 => 2,
        8278 => 2,
        8280 => 2,
        8281 => 2,
        8282 => 2,
        8283 => 2,
        8284 => 2,
        8285 => 2,
        8286 => 2,
        8352 => 2,
        8353 => 2,
        8354 => 2,
        8355 => 2,
        8356 => 2,
        8357 => 2,
        8358 => 2,
        8359 => 2,
        8361 => 2,
        8362 => 2,
        8363 => 2,
        8364 => 2,
        8365 => 2,
        8366 => 2,
        8367 => 2,
        8368 => 2,
        8369 => 2,
        8370 => 2,
        8371 => 2,
        8372 => 2,
        8373 => 2,
        8374 => 2,
        8375 => 2,
        8376 => 2,
        8377 => 2,
        8378 => 2,
        8379 => 2,
        8380 => 2,
        8381 => 2,
        8382 => 2,
        8383 => 2,
        8384 => 2,
        8400 => 2,
        8401 => 2,
        8402 => 2,
        8403 => 2,
        8404 => 2,
        8405 => 2,
        8406 => 2,
        8407 => 2,
        8408 => 2,
        8409 => 2,
        8410 => 2,
        8411 => 2,
        8412 => 2,
        8413 => 2,
        8414 => 2,
        8415 => 2,
        8416 => 2,
        8417 => 2,
        8418 => 2,
        8419 => 2,
        8420 => 2,
        8421 => 2,
        8422 => 2,
        8423 => 2,
        8424 => 2,
        8425 => 2,
        8426 => 2,
        8427 => 2,
        8428 => 2,
        8429 => 2,
        8430 => 2,
        8431 => 2,
        8432 => 2,
        8452 => 2,
        8456 => 2,
        8468 => 2,
        8471 => 2,
        8472 => 2,
        8478 => 2,
        8479 => 2,
        8483 => 2,
        8485 => 2,
        8487 => 2,
        8489 => 2,
        8494 => 2,
        8506 => 2,
        8513 => 2,
        8514 => 2,
        8515 => 2,
        8516 => 2,
        8522 => 2,
        8523 => 2,
        8524 => 2,
        8525 => 2,
        8526 => 1,
        8527 => 2,
        8576 => 2,
        8577 => 2,
        8578 => 2,
        8580 => 1,
        8581 => 2,
        8582 => 2,
        8583 => 2,
        8584 => 2,
        8586 => 2,
        8587 => 2,
        8683 => 2,
        8684 => 2,
        8685 => 2,
        8686 => 2,
        8687 => 2,
        8688 => 2,
        8689 => 2,
        8690 => 2,
        8691 => 2,
        8692 => 2,
        8693 => 2,
        8694 => 2,
        8695 => 2,
        8696 => 2,
        8697 => 2,
        8698 => 2,
        8699 => 2,
        8700 => 2,
        8701 => 2,
        8702 => 2,
        8703 => 2,
        8750 => 2,
        8801 => 2,
        8802 => 2,
        8803 => 2,
        8804 => 2,
        8805 => 2,
        8806 => 2,
        8807 => 2,
        8808 => 2,
        8809 => 2,
        8810 => 2,
        8811 => 2,
        8812 => 2,
        8813 => 2,
        8946 => 2,
        8947 => 2,
        8948 => 2,
        8949 => 2,
        8950 => 2,
        8951 => 2,
        8952 => 2,
        8953 => 2,
        8954 => 2,
        8955 => 2,
        8956 => 2,
        8957 => 2,
        8958 => 2,
        8959 => 2,
        8960 => 2,
        8961 => 2,
        9083 => 2,
        9084 => 2,
        9167 => 2,
        9168 => 2,
        9169 => 2,
        9170 => 2,
        9171 => 2,
        9172 => 2,
        9173 => 2,
        9174 => 2,
        9175 => 2,
        9176 => 2,
        9177 => 2,
        9178 => 2,
        9179 => 2,
        9180 => 2,
        9181 => 2,
        9182 => 2,
        9183 => 2,
        9184 => 2,
        9185 => 2,
        9186 => 2,
        9187 => 2,
        9188 => 2,
        9189 => 2,
        9190 => 2,
        9191 => 2,
        9192 => 2,
        9193 => 2,
        9194 => 2,
        9195 => 2,
        9196 => 2,
        9197 => 2,
        9198 => 2,
        9199 => 2,
        9200 => 2,
        9201 => 2,
        9202 => 2,
        9203 => 2,
        9204 => 2,
        9205 => 2,
        9206 => 2,
        9207 => 2,
        9208 => 2,
        9209 => 2,
        9210 => 2,
        9211 => 2,
        9212 => 2,
        9213 => 2,
        9214 => 2,
        9215 => 2,
        9253 => 2,
        9254 => 2,
        9280 => 2,
        9281 => 2,
        9282 => 2,
        9283 => 2,
        9284 => 2,
        9285 => 2,
        9286 => 2,
        9287 => 2,
        9288 => 2,
        9289 => 2,
        9290 => 2,
        9451 => 2,
        9452 => 2,
        9453 => 2,
        9454 => 2,
        9455 => 2,
        9456 => 2,
        9457 => 2,
        9458 => 2,
        9459 => 2,
        9460 => 2,
        9461 => 2,
        9462 => 2,
        9463 => 2,
        9464 => 2,
        9465 => 2,
        9466 => 2,
        9467 => 2,
        9468 => 2,
        9469 => 2,
        9470 => 2,
        9471 => 2,
        9622 => 2,
        9623 => 2,
        9624 => 2,
        9625 => 2,
        9626 => 2,
        9627 => 2,
        9628 => 2,
        9629 => 2,
        9630 => 2,
        9631 => 2,
        9712 => 2,
        9713 => 2,
        9714 => 2,
        9715 => 2,
        9716 => 2,
        9717 => 2,
        9718 => 2,
        9719 => 2,
        9720 => 2,
        9721 => 2,
        9722 => 2,
        9723 => 2,
        9724 => 2,
        9725 => 2,
        9726 => 2,
        9727 => 2,
        9728 => 2,
        9729 => 2,
        9730 => 2,
        9731 => 2,
        9732 => 2,
        9733 => 2,
        9734 => 2,
        9735 => 2,
        9736 => 2,
        9737 => 2,
        9738 => 2,
        9739 => 2,
        9740 => 2,
        9741 => 2,
        9742 => 2,
        9743 => 2,
        9744 => 2,
        9745 => 2,
        9746 => 2,
        9747 => 2,
        9748 => 2,
        9749 => 2,
        9750 => 2,
        9751 => 2,
        9752 => 2,
        9753 => 2,
        9840 => 2,
        9841 => 2,
        9842 => 2,
        9843 => 2,
        9844 => 2,
        9845 => 2,
        9846 => 2,
        9847 => 2,
        9848 => 2,
        9849 => 2,
        9850 => 2,
        9851 => 2,
        9852 => 2,
        9853 => 2,
        9854 => 2,
        9855 => 2,
        9856 => 2,
        9857 => 2,
        9858 => 2,
        9859 => 2,
        9860 => 2,
        9861 => 2,
        9862 => 2,
        9863 => 2,
        9864 => 2,
        9865 => 2,
        9866 => 2,
        9867 => 2,
        9868 => 2,
        9869 => 2,
        9870 => 2,
        9871 => 2,
        9872 => 2,
        9873 => 2,
        9874 => 2,
        9875 => 2,
        9876 => 2,
        9877 => 2,
        9878 => 2,
        9879 => 2,
        9880 => 2,
        9881 => 2,
        9882 => 2,
        9883 => 2,
        9884 => 2,
        9885 => 2,
        9886 => 2,
        9887 => 2,
        9888 => 2,
        9889 => 2,
        9890 => 2,
        9891 => 2,
        9892 => 2,
        9893 => 2,
        9894 => 2,
        9895 => 2,
        9896 => 2,
        9897 => 2,
        9898 => 2,
        9899 => 2,
        9900 => 2,
        9901 => 2,
        9902 => 2,
        9903 => 2,
        9904 => 2,
        9905 => 2,
        9906 => 2,
        9907 => 2,
        9908 => 2,
        9909 => 2,
        9910 => 2,
        9911 => 2,
        9912 => 2,
        9913 => 2,
        9914 => 2,
        9915 => 2,
        9916 => 2,
        9917 => 2,
        9918 => 2,
        9919 => 2,
        9920 => 2,
        9921 => 2,
        9922 => 2,
        9923 => 2,
        9924 => 2,
        9925 => 2,
        9926 => 2,
        9927 => 2,
        9928 => 2,
        9929 => 2,
        9930 => 2,
        9931 => 2,
        9932 => 2,
        9933 => 2,
        9934 => 2,
        9935 => 2,
        9936 => 2,
        9937 => 2,
        9938 => 2,
        9939 => 2,
        9940 => 2,
        9941 => 2,
        9942 => 2,
        9943 => 2,
        9944 => 2,
        9945 => 2,
        9946 => 2,
        9947 => 2,
        9948 => 2,
        9949 => 2,
        9950 => 2,
        9951 => 2,
        9952 => 2,
        9953 => 2,
        9954 => 2,
        9955 => 2,
        9956 => 2,
        9957 => 2,
        9958 => 2,
        9959 => 2,
        9960 => 2,
        9961 => 2,
        9962 => 2,
        9963 => 2,
        9964 => 2,
        9965 => 2,
        9966 => 2,
        9967 => 2,
        9968 => 2,
        9969 => 2,
        9970 => 2,
        9971 => 2,
        9972 => 2,
        9973 => 2,
        9974 => 2,
        9975 => 2,
        9976 => 2,
        9977 => 2,
        9978 => 2,
        9979 => 2,
        9980 => 2,
        9981 => 2,
        9982 => 2,
        9983 => 2,
        9984 => 2,
        9985 => 2,
        9986 => 2,
        9987 => 2,
        9988 => 2,
        9989 => 2,
        9990 => 2,
        9991 => 2,
        9992 => 2,
        9993 => 2,
        9994 => 2,
        9995 => 2,
        10024 => 2,
        10060 => 2,
        10061 => 2,
        10062 => 2,
        10063 => 2,
        10064 => 2,
        10065 => 2,
        10066 => 2,
        10067 => 2,
        10068 => 2,
        10069 => 2,
        10070 => 2,
        10071 => 2,
        10072 => 2,
        10073 => 2,
        10074 => 2,
        10075 => 2,
        10076 => 2,
        10077 => 2,
        10078 => 2,
        10079 => 2,
        10080 => 2,
        10081 => 2,
        10082 => 2,
        10083 => 2,
        10084 => 2,
        10085 => 2,
        10086 => 2,
        10087 => 2,
        10088 => 2,
        10089 => 2,
        10090 => 2,
        10091 => 2,
        10092 => 2,
        10093 => 2,
        10094 => 2,
        10095 => 2,
        10096 => 2,
        10097 => 2,
        10098 => 2,
        10099 => 2,
        10100 => 2,
        10101 => 2,
        10133 => 2,
        10134 => 2,
        10135 => 2,
        10136 => 2,
        10137 => 2,
        10138 => 2,
        10139 => 2,
        10140 => 2,
        10141 => 2,
        10142 => 2,
        10143 => 2,
        10144 => 2,
        10145 => 2,
        10146 => 2,
        10147 => 2,
        10148 => 2,
        10149 => 2,
        10150 => 2,
        10151 => 2,
        10152 => 2,
        10153 => 2,
        10154 => 2,
        10155 => 2,
        10156 => 2,
        10157 => 2,
        10158 => 2,
        10159 => 2,
        10160 => 2,
        10161 => 2,
        10162 => 2,
        10163 => 2,
        10164 => 2,
        10165 => 2,
        10166 => 2,
        10167 => 2,
        10168 => 2,
        10169 => 2,
        10170 => 2,
        10171 => 2,
        10172 => 2,
        10173 => 2,
        10174 => 2,
        10175 => 2,
        10176 => 2,
        10177 => 2,
        10178 => 2,
        10179 => 2,
        10180 => 2,
        10181 => 2,
        10182 => 2,
        10183 => 2,
        10184 => 2,
        10185 => 2,
        10186 => 2,
        10187 => 2,
        10188 => 2,
        10189 => 2,
        10190 => 2,
        10191 => 2,
        10220 => 2,
        10221 => 2,
        10222 => 2,
        10223 => 2,
        10224 => 2,
        10225 => 2,
        10226 => 2,
        10227 => 2,
        10228 => 2,
        10229 => 2,
        10230 => 2,
        10231 => 2,
        10232 => 2,
        10233 => 2,
        10234 => 2,
        10235 => 2,
        10236 => 2,
        10237 => 2,
        10238 => 2,
        10239 => 2,
        11008 => 2,
        11009 => 2,
        11010 => 2,
        11011 => 2,
        11012 => 2,
        11013 => 2,
        11014 => 2,
        11015 => 2,
        11016 => 2,
        11017 => 2,
        11018 => 2,
        11019 => 2,
        11020 => 2,
        11021 => 2,
        11022 => 2,
        11023 => 2,
        11024 => 2,
        11025 => 2,
        11026 => 2,
        11027 => 2,
        11028 => 2,
        11029 => 2,
        11030 => 2,
        11031 => 2,
        11032 => 2,
        11033 => 2,
        11034 => 2,
        11035 => 2,
        11036 => 2,
        11037 => 2,
        11038 => 2,
        11039 => 2,
        11040 => 2,
        11041 => 2,
        11042 => 2,
        11043 => 2,
        11085 => 2,
        11086 => 2,
        11087 => 2,
        11088 => 2,
        11089 => 2,
        11090 => 2,
        11091 => 2,
        11092 => 2,
        11093 => 2,
        11094 => 2,
        11095 => 2,
        11096 => 2,
        11097 => 2,
        11098 => 2,
        11099 => 2,
        11100 => 2,
        11101 => 2,
        11102 => 2,
        11103 => 2,
        11104 => 2,
        11105 => 2,
        11106 => 2,
        11107 => 2,
        11108 => 2,
        11109 => 2,
        11110 => 2,
        11111 => 2,
        11112 => 2,
        11113 => 2,
        11114 => 2,
        11115 => 2,
        11116 => 2,
        11117 => 2,
        11118 => 2,
        11119 => 2,
        11120 => 2,
        11121 => 2,
        11122 => 2,
        11123 => 2,
        11159 => 2,
        11194 => 2,
        11195 => 2,
        11196 => 2,
        11197 => 2,
        11198 => 2,
        11199 => 2,
        11200 => 2,
        11201 => 2,
        11202 => 2,
        11203 => 2,
        11204 => 2,
        11205 => 2,
        11206 => 2,
        11207 => 2,
        11208 => 2,
        11209 => 2,
        11210 => 2,
        11211 => 2,
        11212 => 2,
        11213 => 2,
        11214 => 2,
        11215 => 2,
        11216 => 2,
        11217 => 2,
        11218 => 2,
        11219 => 2,
        11220 => 2,
        11221 => 2,
        11222 => 2,
        11223 => 2,
        11224 => 2,
        11225 => 2,
        11226 => 2,
        11227 => 2,
        11228 => 2,
        11229 => 2,
        11230 => 2,
        11231 => 2,
        11232 => 2,
        11233 => 2,
        11234 => 2,
        11235 => 2,
        11236 => 2,
        11237 => 2,
        11238 => 2,
        11239 => 2,
        11240 => 2,
        11241 => 2,
        11242 => 2,
        11243 => 2,
        11244 => 2,
        11245 => 2,
        11246 => 2,
        11247 => 2,
        11248 => 2,
        11249 => 2,
        11250 => 2,
        11251 => 2,
        11252 => 2,
        11253 => 2,
        11254 => 2,
        11255 => 2,
        11256 => 2,
        11257 => 2,
        11258 => 2,
        11259 => 2,
        11260 => 2,
        11261 => 2,
        11262 => 2,
        11263 => 2,
        11359 => 1,
        11361 => 1,
        11365 => 1,
        11366 => 1,
        11368 => 1,
        11370 => 1,
        11372 => 1,
        11377 => 1,
        11379 => 1,
        11380 => 1,
        11382 => 1,
        11383 => 1,
        11384 => 1,
        11385 => 1,
        11386 => 1,
        11387 => 1,
        11393 => 1,
        11395 => 1,
        11397 => 1,
        11399 => 1,
        11401 => 1,
        11403 => 1,
        11405 => 1,
        11407 => 1,
        11409 => 1,
        11411 => 1,
        11413 => 1,
        11415 => 1,
        11417 => 1,
        11419 => 1,
        11421 => 1,
        11423 => 1,
        11425 => 1,
        11427 => 1,
        11429 => 1,
        11431 => 1,
        11433 => 1,
        11435 => 1,
        11437 => 1,
        11439 => 1,
        11441 => 1,
        11443 => 1,
        11445 => 1,
        11447 => 1,
        11449 => 1,
        11451 => 1,
        11453 => 1,
        11455 => 1,
        11457 => 1,
        11459 => 1,
        11461 => 1,
        11463 => 1,
        11465 => 1,
        11467 => 1,
        11469 => 1,
        11471 => 1,
        11473 => 1,
        11475 => 1,
        11477 => 1,
        11479 => 1,
        11481 => 1,
        11483 => 1,
        11485 => 1,
        11487 => 1,
        11489 => 1,
        11491 => 1,
        11492 => 1,
        11493 => 2,
        11494 => 2,
        11495 => 2,
        11496 => 2,
        11497 => 2,
        11498 => 2,
        11500 => 1,
        11502 => 1,
        11503 => 1,
        11504 => 1,
        11505 => 1,
        11507 => 1,
        11513 => 2,
        11514 => 2,
        11515 => 2,
        11516 => 2,
        11517 => 2,
        11518 => 2,
        11519 => 2,
        11559 => 1,
        11565 => 1,
        11622 => 1,
        11623 => 1,
        11632 => 2,
        11647 => 1,
        11648 => 1,
        11649 => 1,
        11650 => 1,
        11651 => 1,
        11652 => 1,
        11653 => 1,
        11654 => 1,
        11655 => 1,
        11656 => 1,
        11657 => 1,
        11658 => 1,
        11659 => 1,
        11660 => 1,
        11661 => 1,
        11662 => 1,
        11663 => 1,
        11664 => 1,
        11665 => 1,
        11666 => 1,
        11667 => 1,
        11668 => 1,
        11669 => 1,
        11670 => 1,
        11680 => 1,
        11681 => 1,
        11682 => 1,
        11683 => 1,
        11684 => 1,
        11685 => 1,
        11686 => 1,
        11688 => 1,
        11689 => 1,
        11690 => 1,
        11691 => 1,
        11692 => 1,
        11693 => 1,
        11694 => 1,
        11696 => 1,
        11697 => 1,
        11698 => 1,
        11699 => 1,
        11700 => 1,
        11701 => 1,
        11702 => 1,
        11704 => 1,
        11705 => 1,
        11706 => 1,
        11707 => 1,
        11708 => 1,
        11709 => 1,
        11710 => 1,
        11712 => 1,
        11713 => 1,
        11714 => 1,
        11715 => 1,
        11716 => 1,
        11717 => 1,
        11718 => 1,
        11720 => 1,
        11721 => 1,
        11722 => 1,
        11723 => 1,
        11724 => 1,
        11725 => 1,
        11726 => 1,
        11728 => 1,
        11729 => 1,
        11730 => 1,
        11731 => 1,
        11732 => 1,
        11733 => 1,
        11734 => 1,
        11736 => 1,
        11737 => 1,
        11738 => 1,
        11739 => 1,
        11740 => 1,
        11741 => 1,
        11742 => 1,
        11776 => 2,
        11777 => 2,
        11778 => 2,
        11779 => 2,
        11780 => 2,
        11781 => 2,
        11782 => 2,
        11783 => 2,
        11784 => 2,
        11785 => 2,
        11786 => 2,
        11787 => 2,
        11788 => 2,
        11789 => 2,
        11790 => 2,
        11791 => 2,
        11792 => 2,
        11793 => 2,
        11794 => 2,
        11795 => 2,
        11796 => 2,
        11797 => 2,
        11798 => 2,
        11799 => 2,
        11800 => 2,
        11801 => 2,
        11802 => 2,
        11803 => 2,
        11804 => 2,
        11805 => 2,
        11806 => 2,
        11807 => 2,
        11808 => 2,
        11809 => 2,
        11810 => 2,
        11811 => 2,
        11812 => 2,
        11813 => 2,
        11814 => 2,
        11815 => 2,
        11816 => 2,
        11817 => 2,
        11818 => 2,
        11819 => 2,
        11820 => 2,
        11821 => 2,
        11822 => 2,
        11823 => 1,
        11824 => 2,
        11825 => 2,
        11826 => 2,
        11827 => 2,
        11828 => 2,
        11829 => 2,
        11830 => 2,
        11831 => 2,
        11832 => 2,
        11833 => 2,
        11834 => 2,
        11835 => 2,
        11836 => 2,
        11837 => 2,
        11838 => 2,
        11839 => 2,
        11840 => 2,
        11841 => 2,
        11842 => 2,
        11843 => 2,
        11844 => 2,
        11845 => 2,
        11846 => 2,
        11847 => 2,
        11848 => 2,
        11849 => 2,
        11850 => 2,
        11851 => 2,
        11852 => 2,
        11853 => 2,
        11854 => 2,
        11855 => 2,
        11856 => 2,
        11857 => 2,
        11858 => 2,
        11859 => 2,
        11860 => 2,
        11861 => 2,
        11862 => 2,
        11863 => 2,
        11864 => 2,
        11865 => 2,
        11866 => 2,
        11867 => 2,
        11868 => 2,
        11869 => 2,
        11904 => 2,
        11905 => 2,
        11906 => 2,
        11907 => 2,
        11908 => 2,
        11909 => 2,
        11910 => 2,
        11911 => 2,
        11912 => 2,
        11913 => 2,
        11914 => 2,
        11915 => 2,
        11916 => 2,
        11917 => 2,
        11918 => 2,
        11919 => 2,
        11920 => 2,
        11921 => 2,
        11922 => 2,
        11923 => 2,
        11924 => 2,
        11925 => 2,
        11926 => 2,
        11927 => 2,
        11928 => 2,
        11929 => 2,
        11931 => 2,
        11932 => 2,
        11933 => 2,
        11934 => 2,
        12289 => 2,
        12291 => 2,
        12292 => 2,
        12293 => 1,
        12294 => 1,
        12295 => 1,
        12330 => 1,
        12331 => 1,
        12332 => 1,
        12333 => 1,
        12334 => 2,
        12335 => 2,
        12336 => 2,
        12337 => 2,
        12338 => 2,
        12339 => 2,
        12340 => 2,
        12341 => 2,
        12343 => 2,
        12347 => 2,
        12348 => 1,
        12349 => 2,
        12350 => 2,
        12351 => 2,
        12437 => 1,
        12438 => 1,
        12441 => 1,
        12442 => 1,
        12445 => 1,
        12446 => 1,
        12448 => 2,
        12589 => 1,
        12590 => 1,
        12591 => 1,
        12688 => 2,
        12689 => 2,
        12704 => 1,
        12705 => 1,
        12706 => 1,
        12707 => 1,
        12708 => 1,
        12709 => 1,
        12710 => 1,
        12711 => 1,
        12712 => 1,
        12713 => 1,
        12714 => 1,
        12715 => 1,
        12716 => 1,
        12717 => 1,
        12718 => 1,
        12719 => 1,
        12720 => 1,
        12721 => 1,
        12722 => 1,
        12723 => 1,
        12724 => 1,
        12725 => 1,
        12726 => 1,
        12727 => 1,
        12728 => 1,
        12729 => 1,
        12730 => 1,
        12731 => 1,
        12732 => 1,
        12733 => 1,
        12734 => 1,
        12735 => 1,
        12736 => 2,
        12737 => 2,
        12738 => 2,
        12739 => 2,
        12740 => 2,
        12741 => 2,
        12742 => 2,
        12743 => 2,
        12744 => 2,
        12745 => 2,
        12746 => 2,
        12747 => 2,
        12748 => 2,
        12749 => 2,
        12750 => 2,
        12751 => 2,
        12752 => 2,
        12753 => 2,
        12754 => 2,
        12755 => 2,
        12756 => 2,
        12757 => 2,
        12758 => 2,
        12759 => 2,
        12760 => 2,
        12761 => 2,
        12762 => 2,
        12763 => 2,
        12764 => 2,
        12765 => 2,
        12766 => 2,
        12767 => 2,
        12768 => 2,
        12769 => 2,
        12770 => 2,
        12771 => 2,
        12784 => 1,
        12785 => 1,
        12786 => 1,
        12787 => 1,
        12788 => 1,
        12789 => 1,
        12790 => 1,
        12791 => 1,
        12792 => 1,
        12793 => 1,
        12794 => 1,
        12795 => 1,
        12796 => 1,
        12797 => 1,
        12798 => 1,
        12799 => 1,
        12872 => 2,
        12873 => 2,
        12874 => 2,
        12875 => 2,
        12876 => 2,
        12877 => 2,
        12878 => 2,
        12879 => 2,
        12927 => 2,
        19894 => 1,
        19895 => 1,
        19896 => 1,
        19897 => 1,
        19898 => 1,
        19899 => 1,
        19900 => 1,
        19901 => 1,
        19902 => 1,
        19903 => 1,
        40870 => 1,
        40871 => 1,
        40872 => 1,
        40873 => 1,
        40874 => 1,
        40875 => 1,
        40876 => 1,
        40877 => 1,
        40878 => 1,
        40879 => 1,
        40880 => 1,
        40881 => 1,
        40882 => 1,
        40883 => 1,
        40884 => 1,
        40885 => 1,
        40886 => 1,
        40887 => 1,
        40888 => 1,
        40889 => 1,
        40890 => 1,
        40891 => 1,
        40892 => 1,
        40893 => 1,
        40894 => 1,
        40895 => 1,
        40896 => 1,
        40897 => 1,
        40898 => 1,
        40899 => 1,
        40900 => 1,
        40901 => 1,
        40902 => 1,
        40903 => 1,
        40904 => 1,
        40905 => 1,
        40906 => 1,
        40907 => 1,
        40908 => 1,
        40909 => 1,
        40910 => 1,
        40911 => 1,
        40912 => 1,
        40913 => 1,
        40914 => 1,
        40915 => 1,
        40916 => 1,
        40917 => 1,
        40918 => 1,
        40919 => 1,
        40920 => 1,
        40921 => 1,
        40922 => 1,
        40923 => 1,
        40924 => 1,
        40925 => 1,
        40926 => 1,
        40927 => 1,
        40928 => 1,
        40929 => 1,
        40930 => 1,
        40931 => 1,
        40932 => 1,
        40933 => 1,
        40934 => 1,
        40935 => 1,
        40936 => 1,
        40937 => 1,
        40938 => 1,
        40939 => 1,
        40940 => 1,
        40941 => 1,
        40942 => 1,
        40943 => 1,
        40944 => 1,
        40945 => 1,
        40946 => 1,
        40947 => 1,
        40948 => 1,
        40949 => 1,
        40950 => 1,
        40951 => 1,
        40952 => 1,
        40953 => 1,
        40954 => 1,
        40955 => 1,
        40956 => 1,
        40957 => 1,
        40958 => 1,
        40959 => 1,
        42128 => 2,
        42129 => 2,
        42130 => 2,
        42131 => 2,
        42132 => 2,
        42133 => 2,
        42134 => 2,
        42135 => 2,
        42136 => 2,
        42137 => 2,
        42138 => 2,
        42139 => 2,
        42140 => 2,
        42141 => 2,
        42142 => 2,
        42143 => 2,
        42144 => 2,
        42145 => 2,
        42146 => 2,
        42147 => 2,
        42148 => 2,
        42149 => 2,
        42150 => 2,
        42151 => 2,
        42152 => 2,
        42153 => 2,
        42154 => 2,
        42155 => 2,
        42156 => 2,
        42157 => 2,
        42158 => 2,
        42159 => 2,
        42160 => 2,
        42161 => 2,
        42162 => 2,
        42163 => 2,
        42164 => 2,
        42165 => 2,
        42166 => 2,
        42167 => 2,
        42168 => 2,
        42169 => 2,
        42170 => 2,
        42171 => 2,
        42172 => 2,
        42173 => 2,
        42174 => 2,
        42175 => 2,
        42176 => 2,
        42177 => 2,
        42178 => 2,
        42179 => 2,
        42180 => 2,
        42181 => 2,
        42182 => 2,
        42238 => 2,
        42239 => 2,
        42509 => 2,
        42510 => 2,
        42511 => 2,
        42561 => 1,
        42563 => 1,
        42565 => 1,
        42567 => 1,
        42569 => 1,
        42571 => 1,
        42573 => 1,
        42575 => 1,
        42577 => 1,
        42579 => 1,
        42581 => 1,
        42583 => 1,
        42585 => 1,
        42587 => 1,
        42589 => 1,
        42591 => 1,
        42593 => 1,
        42595 => 1,
        42597 => 1,
        42599 => 1,
        42601 => 1,
        42603 => 1,
        42605 => 1,
        42606 => 1,
        42607 => 1,
        42608 => 2,
        42609 => 2,
        42610 => 2,
        42611 => 2,
        42612 => 1,
        42613 => 1,
        42614 => 1,
        42615 => 1,
        42616 => 1,
        42617 => 1,
        42618 => 1,
        42619 => 1,
        42620 => 1,
        42621 => 1,
        42622 => 2,
        42623 => 1,
        42625 => 1,
        42627 => 1,
        42629 => 1,
        42631 => 1,
        42633 => 1,
        42635 => 1,
        42637 => 1,
        42639 => 1,
        42641 => 1,
        42643 => 1,
        42645 => 1,
        42647 => 1,
        42649 => 1,
        42651 => 1,
        42654 => 1,
        42655 => 1,
        42726 => 2,
        42727 => 2,
        42728 => 2,
        42729 => 2,
        42730 => 2,
        42731 => 2,
        42732 => 2,
        42733 => 2,
        42734 => 2,
        42735 => 2,
        42736 => 1,
        42737 => 1,
        42738 => 2,
        42739 => 2,
        42740 => 2,
        42741 => 2,
        42742 => 2,
        42743 => 2,
        42752 => 2,
        42753 => 2,
        42754 => 2,
        42755 => 2,
        42756 => 2,
        42757 => 2,
        42758 => 2,
        42759 => 2,
        42760 => 2,
        42761 => 2,
        42762 => 2,
        42763 => 2,
        42764 => 2,
        42765 => 2,
        42766 => 2,
        42767 => 2,
        42768 => 2,
        42769 => 2,
        42770 => 2,
        42771 => 2,
        42772 => 2,
        42773 => 2,
        42774 => 2,
        42775 => 1,
        42776 => 1,
        42777 => 1,
        42778 => 1,
        42779 => 1,
        42780 => 1,
        42781 => 1,
        42782 => 1,
        42783 => 1,
        42784 => 2,
        42785 => 2,
        42787 => 1,
        42789 => 1,
        42791 => 1,
        42793 => 1,
        42795 => 1,
        42797 => 1,
        42799 => 1,
        42800 => 1,
        42801 => 1,
        42803 => 1,
        42805 => 1,
        42807 => 1,
        42809 => 1,
        42811 => 1,
        42813 => 1,
        42815 => 1,
        42817 => 1,
        42819 => 1,
        42821 => 1,
        42823 => 1,
        42825 => 1,
        42827 => 1,
        42829 => 1,
        42831 => 1,
        42833 => 1,
        42835 => 1,
        42837 => 1,
        42839 => 1,
        42841 => 1,
        42843 => 1,
        42845 => 1,
        42847 => 1,
        42849 => 1,
        42851 => 1,
        42853 => 1,
        42855 => 1,
        42857 => 1,
        42859 => 1,
        42861 => 1,
        42863 => 1,
        42865 => 1,
        42866 => 1,
        42867 => 1,
        42868 => 1,
        42869 => 1,
        42870 => 1,
        42871 => 1,
        42872 => 1,
        42874 => 1,
        42876 => 1,
        42879 => 1,
        42881 => 1,
        42883 => 1,
        42885 => 1,
        42887 => 1,
        42888 => 1,
        42889 => 2,
        42890 => 2,
        42892 => 1,
        42894 => 1,
        42895 => 1,
        42897 => 1,
        42899 => 1,
        42900 => 1,
        42901 => 1,
        42903 => 1,
        42905 => 1,
        42907 => 1,
        42909 => 1,
        42911 => 1,
        42913 => 1,
        42915 => 1,
        42917 => 1,
        42919 => 1,
        42921 => 1,
        42927 => 1,
        42933 => 1,
        42935 => 1,
        42937 => 1,
        42939 => 1,
        42941 => 1,
        42943 => 1,
        42945 => 1,
        42947 => 1,
        42952 => 1,
        42954 => 1,
        42961 => 1,
        42963 => 1,
        42965 => 1,
        42967 => 1,
        42969 => 1,
        42998 => 1,
        42999 => 1,
        43002 => 1,
        43003 => 1,
        43004 => 1,
        43005 => 1,
        43006 => 1,
        43007 => 1,
        43048 => 2,
        43049 => 2,
        43050 => 2,
        43051 => 2,
        43052 => 1,
        43056 => 2,
        43057 => 2,
        43058 => 2,
        43059 => 2,
        43060 => 2,
        43061 => 2,
        43062 => 2,
        43063 => 2,
        43064 => 2,
        43065 => 2,
        43124 => 2,
        43125 => 2,
        43126 => 2,
        43127 => 2,
        43205 => 1,
        43214 => 2,
        43215 => 2,
        43216 => 1,
        43217 => 1,
        43218 => 1,
        43219 => 1,
        43220 => 1,
        43221 => 1,
        43222 => 1,
        43223 => 1,
        43224 => 1,
        43225 => 1,
        43232 => 1,
        43233 => 1,
        43234 => 1,
        43235 => 1,
        43236 => 1,
        43237 => 1,
        43238 => 1,
        43239 => 1,
        43240 => 1,
        43241 => 1,
        43242 => 1,
        43243 => 1,
        43244 => 1,
        43245 => 1,
        43246 => 1,
        43247 => 1,
        43248 => 1,
        43249 => 1,
        43250 => 1,
        43251 => 1,
        43252 => 1,
        43253 => 1,
        43254 => 1,
        43255 => 1,
        43256 => 2,
        43257 => 2,
        43258 => 2,
        43259 => 1,
        43260 => 2,
        43261 => 1,
        43262 => 1,
        43263 => 1,
        43310 => 2,
        43311 => 2,
        43359 => 2,
        43457 => 2,
        43458 => 2,
        43459 => 2,
        43460 => 2,
        43461 => 2,
        43462 => 2,
        43463 => 2,
        43464 => 2,
        43465 => 2,
        43466 => 2,
        43467 => 2,
        43468 => 2,
        43469 => 2,
        43471 => 1,
        43472 => 1,
        43473 => 1,
        43474 => 1,
        43475 => 1,
        43476 => 1,
        43477 => 1,
        43478 => 1,
        43479 => 1,
        43480 => 1,
        43481 => 1,
        43486 => 2,
        43487 => 2,
        43584 => 1,
        43585 => 1,
        43586 => 1,
        43587 => 1,
        43588 => 1,
        43589 => 1,
        43590 => 1,
        43591 => 1,
        43592 => 1,
        43593 => 1,
        43594 => 1,
        43595 => 1,
        43596 => 1,
        43597 => 1,
        43600 => 1,
        43601 => 1,
        43602 => 1,
        43603 => 1,
        43604 => 1,
        43605 => 1,
        43606 => 1,
        43607 => 1,
        43608 => 1,
        43609 => 1,
        43612 => 2,
        43613 => 2,
        43614 => 2,
        43615 => 2,
        43616 => 1,
        43617 => 1,
        43618 => 1,
        43619 => 1,
        43620 => 1,
        43621 => 1,
        43622 => 1,
        43623 => 1,
        43624 => 1,
        43625 => 1,
        43626 => 1,
        43627 => 1,
        43628 => 1,
        43629 => 1,
        43630 => 1,
        43631 => 1,
        43632 => 1,
        43633 => 1,
        43634 => 1,
        43635 => 1,
        43636 => 1,
        43637 => 1,
        43638 => 1,
        43639 => 2,
        43640 => 2,
        43641 => 2,
        43642 => 1,
        43643 => 1,
        43644 => 1,
        43645 => 1,
        43646 => 1,
        43647 => 1,
        43739 => 1,
        43740 => 1,
        43741 => 1,
        43742 => 2,
        43743 => 2,
        43744 => 1,
        43745 => 1,
        43746 => 1,
        43747 => 1,
        43748 => 1,
        43749 => 1,
        43750 => 1,
        43751 => 1,
        43752 => 1,
        43753 => 1,
        43754 => 1,
        43755 => 1,
        43756 => 1,
        43757 => 1,
        43758 => 1,
        43759 => 1,
        43760 => 2,
        43761 => 2,
        43762 => 1,
        43763 => 1,
        43764 => 1,
        43765 => 1,
        43766 => 1,
        43777 => 1,
        43778 => 1,
        43779 => 1,
        43780 => 1,
        43781 => 1,
        43782 => 1,
        43785 => 1,
        43786 => 1,
        43787 => 1,
        43788 => 1,
        43789 => 1,
        43790 => 1,
        43793 => 1,
        43794 => 1,
        43795 => 1,
        43796 => 1,
        43797 => 1,
        43798 => 1,
        43808 => 1,
        43809 => 1,
        43810 => 1,
        43811 => 1,
        43812 => 1,
        43813 => 1,
        43814 => 1,
        43816 => 1,
        43817 => 1,
        43818 => 1,
        43819 => 1,
        43820 => 1,
        43821 => 1,
        43822 => 1,
        43867 => 2,
        43872 => 1,
        43873 => 1,
        43874 => 1,
        43875 => 1,
        43876 => 1,
        43877 => 1,
        43878 => 1,
        43879 => 1,
        43880 => 1,
        43882 => 2,
        43883 => 2,
        44011 => 2,
        44012 => 1,
        44013 => 1,
        44016 => 1,
        44017 => 1,
        44018 => 1,
        44019 => 1,
        44020 => 1,
        44021 => 1,
        44022 => 1,
        44023 => 1,
        44024 => 1,
        44025 => 1,
        55216 => 2,
        55217 => 2,
        55218 => 2,
        55219 => 2,
        55220 => 2,
        55221 => 2,
        55222 => 2,
        55223 => 2,
        55224 => 2,
        55225 => 2,
        55226 => 2,
        55227 => 2,
        55228 => 2,
        55229 => 2,
        55230 => 2,
        55231 => 2,
        55232 => 2,
        55233 => 2,
        55234 => 2,
        55235 => 2,
        55236 => 2,
        55237 => 2,
        55238 => 2,
        64014 => 1,
        64015 => 1,
        64017 => 1,
        64019 => 1,
        64020 => 1,
        64031 => 1,
        64033 => 1,
        64035 => 1,
        64036 => 1,
        64039 => 1,
        64040 => 1,
        64041 => 1,
        64286 => 1,
        64434 => 2,
        64435 => 2,
        64436 => 2,
        64437 => 2,
        64438 => 2,
        64439 => 2,
        64440 => 2,
        64441 => 2,
        64442 => 2,
        64443 => 2,
        64444 => 2,
        64445 => 2,
        64446 => 2,
        64447 => 2,
        64448 => 2,
        64449 => 2,
        64450 => 2,
        64830 => 2,
        64831 => 2,
        64832 => 2,
        64833 => 2,
        64834 => 2,
        64835 => 2,
        64836 => 2,
        64837 => 2,
        64838 => 2,
        64839 => 2,
        64840 => 2,
        64841 => 2,
        64842 => 2,
        64843 => 2,
        64844 => 2,
        64845 => 2,
        64846 => 2,
        64847 => 2,
        64975 => 2,
        65021 => 2,
        65022 => 2,
        65023 => 2,
        65056 => 1,
        65057 => 1,
        65058 => 1,
        65059 => 1,
        65060 => 1,
        65061 => 1,
        65062 => 1,
        65063 => 1,
        65064 => 1,
        65065 => 1,
        65066 => 1,
        65067 => 1,
        65068 => 1,
        65069 => 1,
        65070 => 1,
        65071 => 1,
        65093 => 2,
        65094 => 2,
        65139 => 1,
        65536 => 1,
        65537 => 1,
        65538 => 1,
        65539 => 1,
        65540 => 1,
        65541 => 1,
        65542 => 1,
        65543 => 1,
        65544 => 1,
        65545 => 1,
        65546 => 1,
        65547 => 1,
        65549 => 1,
        65550 => 1,
        65551 => 1,
        65552 => 1,
        65553 => 1,
        65554 => 1,
        65555 => 1,
        65556 => 1,
        65557 => 1,
        65558 => 1,
        65559 => 1,
        65560 => 1,
        65561 => 1,
        65562 => 1,
        65563 => 1,
        65564 => 1,
        65565 => 1,
        65566 => 1,
        65567 => 1,
        65568 => 1,
        65569 => 1,
        65570 => 1,
        65571 => 1,
        65572 => 1,
        65573 => 1,
        65574 => 1,
        65576 => 1,
        65577 => 1,
        65578 => 1,
        65579 => 1,
        65580 => 1,
        65581 => 1,
        65582 => 1,
        65583 => 1,
        65584 => 1,
        65585 => 1,
        65586 => 1,
        65587 => 1,
        65588 => 1,
        65589 => 1,
        65590 => 1,
        65591 => 1,
        65592 => 1,
        65593 => 1,
        65594 => 1,
        65596 => 1,
        65597 => 1,
        65599 => 1,
        65600 => 1,
        65601 => 1,
        65602 => 1,
        65603 => 1,
        65604 => 1,
        65605 => 1,
        65606 => 1,
        65607 => 1,
        65608 => 1,
        65609 => 1,
        65610 => 1,
        65611 => 1,
        65612 => 1,
        65613 => 1,
        65616 => 1,
        65617 => 1,
        65618 => 1,
        65619 => 1,
        65620 => 1,
        65621 => 1,
        65622 => 1,
        65623 => 1,
        65624 => 1,
        65625 => 1,
        65626 => 1,
        65627 => 1,
        65628 => 1,
        65629 => 1,
        65792 => 2,
        65793 => 2,
        65794 => 2,
        65847 => 2,
        65848 => 2,
        65849 => 2,
        65850 => 2,
        65851 => 2,
        65852 => 2,
        65853 => 2,
        65854 => 2,
        65855 => 2,
        65931 => 2,
        65932 => 2,
        65933 => 2,
        65934 => 2,
        65936 => 2,
        65937 => 2,
        65938 => 2,
        65939 => 2,
        65940 => 2,
        65941 => 2,
        65942 => 2,
        65943 => 2,
        65944 => 2,
        65945 => 2,
        65946 => 2,
        65947 => 2,
        65948 => 2,
        65952 => 2,
        66045 => 1,
        66272 => 1,
        66273 => 2,
        66274 => 2,
        66275 => 2,
        66276 => 2,
        66277 => 2,
        66278 => 2,
        66279 => 2,
        66280 => 2,
        66281 => 2,
        66282 => 2,
        66283 => 2,
        66284 => 2,
        66285 => 2,
        66286 => 2,
        66287 => 2,
        66288 => 2,
        66289 => 2,
        66290 => 2,
        66291 => 2,
        66292 => 2,
        66293 => 2,
        66294 => 2,
        66295 => 2,
        66296 => 2,
        66297 => 2,
        66298 => 2,
        66299 => 2,
        66335 => 1,
        66336 => 2,
        66337 => 2,
        66338 => 2,
        66339 => 2,
        66349 => 1,
        66350 => 1,
        66351 => 1,
        66352 => 1,
        66353 => 1,
        66354 => 1,
        66355 => 1,
        66356 => 1,
        66357 => 1,
        66358 => 1,
        66359 => 1,
        66360 => 1,
        66361 => 1,
        66362 => 1,
        66363 => 1,
        66364 => 1,
        66365 => 1,
        66366 => 1,
        66367 => 1,
        66368 => 1,
        66369 => 2,
        66370 => 1,
        66371 => 1,
        66372 => 1,
        66373 => 1,
        66374 => 1,
        66375 => 1,
        66376 => 1,
        66377 => 1,
        66378 => 2,
        66463 => 2,
        66504 => 1,
        66505 => 1,
        66506 => 1,
        66507 => 1,
        66508 => 1,
        66509 => 1,
        66510 => 1,
        66511 => 1,
        66512 => 2,
        66513 => 2,
        66514 => 2,
        66515 => 2,
        66516 => 2,
        66517 => 2,
        66720 => 1,
        66721 => 1,
        66722 => 1,
        66723 => 1,
        66724 => 1,
        66725 => 1,
        66726 => 1,
        66727 => 1,
        66728 => 1,
        66729 => 1,
        66927 => 2,
        66967 => 1,
        66968 => 1,
        66969 => 1,
        66970 => 1,
        66971 => 1,
        66972 => 1,
        66973 => 1,
        66974 => 1,
        66975 => 1,
        66976 => 1,
        66977 => 1,
        66979 => 1,
        66980 => 1,
        66981 => 1,
        66982 => 1,
        66983 => 1,
        66984 => 1,
        66985 => 1,
        66986 => 1,
        66987 => 1,
        66988 => 1,
        66989 => 1,
        66990 => 1,
        66991 => 1,
        66992 => 1,
        66993 => 1,
        66995 => 1,
        66996 => 1,
        66997 => 1,
        66998 => 1,
        66999 => 1,
        67000 => 1,
        67001 => 1,
        67003 => 1,
        67004 => 1,
        67392 => 1,
        67393 => 1,
        67394 => 1,
        67395 => 1,
        67396 => 1,
        67397 => 1,
        67398 => 1,
        67399 => 1,
        67400 => 1,
        67401 => 1,
        67402 => 1,
        67403 => 1,
        67404 => 1,
        67405 => 1,
        67406 => 1,
        67407 => 1,
        67408 => 1,
        67409 => 1,
        67410 => 1,
        67411 => 1,
        67412 => 1,
        67413 => 1,
        67424 => 1,
        67425 => 1,
        67426 => 1,
        67427 => 1,
        67428 => 1,
        67429 => 1,
        67430 => 1,
        67431 => 1,
        67456 => 1,
        67584 => 1,
        67585 => 1,
        67586 => 1,
        67587 => 1,
        67588 => 1,
        67589 => 1,
        67592 => 1,
        67639 => 1,
        67640 => 1,
        67644 => 1,
        67647 => 1,
        67648 => 1,
        67649 => 1,
        67650 => 1,
        67651 => 1,
        67652 => 1,
        67653 => 1,
        67654 => 1,
        67655 => 1,
        67656 => 1,
        67657 => 1,
        67658 => 1,
        67659 => 1,
        67660 => 1,
        67661 => 1,
        67662 => 1,
        67663 => 1,
        67664 => 1,
        67665 => 1,
        67666 => 1,
        67667 => 1,
        67668 => 1,
        67669 => 1,
        67671 => 2,
        67672 => 2,
        67673 => 2,
        67674 => 2,
        67675 => 2,
        67676 => 2,
        67677 => 2,
        67678 => 2,
        67679 => 2,
        67680 => 1,
        67681 => 1,
        67682 => 1,
        67683 => 1,
        67684 => 1,
        67685 => 1,
        67686 => 1,
        67687 => 1,
        67688 => 1,
        67689 => 1,
        67690 => 1,
        67691 => 1,
        67692 => 1,
        67693 => 1,
        67694 => 1,
        67695 => 1,
        67696 => 1,
        67697 => 1,
        67698 => 1,
        67699 => 1,
        67700 => 1,
        67701 => 1,
        67702 => 1,
        67703 => 2,
        67704 => 2,
        67705 => 2,
        67706 => 2,
        67707 => 2,
        67708 => 2,
        67709 => 2,
        67710 => 2,
        67711 => 2,
        67751 => 2,
        67752 => 2,
        67753 => 2,
        67754 => 2,
        67755 => 2,
        67756 => 2,
        67757 => 2,
        67758 => 2,
        67759 => 2,
        67808 => 1,
        67809 => 1,
        67810 => 1,
        67811 => 1,
        67812 => 1,
        67813 => 1,
        67814 => 1,
        67815 => 1,
        67816 => 1,
        67817 => 1,
        67818 => 1,
        67819 => 1,
        67820 => 1,
        67821 => 1,
        67822 => 1,
        67823 => 1,
        67824 => 1,
        67825 => 1,
        67826 => 1,
        67828 => 1,
        67829 => 1,
        67835 => 2,
        67836 => 2,
        67837 => 2,
        67838 => 2,
        67839 => 2,
        67840 => 1,
        67841 => 1,
        67842 => 1,
        67843 => 1,
        67844 => 1,
        67845 => 1,
        67846 => 1,
        67847 => 1,
        67848 => 1,
        67849 => 1,
        67850 => 1,
        67851 => 1,
        67852 => 1,
        67853 => 1,
        67854 => 1,
        67855 => 1,
        67856 => 1,
        67857 => 1,
        67858 => 1,
        67859 => 1,
        67860 => 1,
        67861 => 1,
        67862 => 2,
        67863 => 2,
        67864 => 2,
        67865 => 2,
        67866 => 2,
        67867 => 2,
        67871 => 2,
        67872 => 1,
        67873 => 1,
        67874 => 1,
        67875 => 1,
        67876 => 1,
        67877 => 1,
        67878 => 1,
        67879 => 1,
        67880 => 1,
        67881 => 1,
        67882 => 1,
        67883 => 1,
        67884 => 1,
        67885 => 1,
        67886 => 1,
        67887 => 1,
        67888 => 1,
        67889 => 1,
        67890 => 1,
        67891 => 1,
        67892 => 1,
        67893 => 1,
        67894 => 1,
        67895 => 1,
        67896 => 1,
        67897 => 1,
        67903 => 2,
        68028 => 2,
        68029 => 2,
        68030 => 1,
        68031 => 1,
        68032 => 2,
        68033 => 2,
        68034 => 2,
        68035 => 2,
        68036 => 2,
        68037 => 2,
        68038 => 2,
        68039 => 2,
        68040 => 2,
        68041 => 2,
        68042 => 2,
        68043 => 2,
        68044 => 2,
        68045 => 2,
        68046 => 2,
        68047 => 2,
        68096 => 1,
        68097 => 1,
        68098 => 1,
        68099 => 1,
        68101 => 1,
        68102 => 1,
        68108 => 1,
        68109 => 1,
        68110 => 1,
        68111 => 1,
        68112 => 1,
        68113 => 1,
        68114 => 1,
        68115 => 1,
        68117 => 1,
        68118 => 1,
        68119 => 1,
        68121 => 1,
        68122 => 1,
        68123 => 1,
        68124 => 1,
        68125 => 1,
        68126 => 1,
        68127 => 1,
        68128 => 1,
        68129 => 1,
        68130 => 1,
        68131 => 1,
        68132 => 1,
        68133 => 1,
        68134 => 1,
        68135 => 1,
        68136 => 1,
        68137 => 1,
        68138 => 1,
        68139 => 1,
        68140 => 1,
        68141 => 1,
        68142 => 1,
        68143 => 1,
        68144 => 1,
        68145 => 1,
        68146 => 1,
        68147 => 1,
        68148 => 1,
        68149 => 1,
        68152 => 1,
        68153 => 1,
        68154 => 1,
        68159 => 1,
        68160 => 2,
        68161 => 2,
        68162 => 2,
        68163 => 2,
        68164 => 2,
        68165 => 2,
        68166 => 2,
        68167 => 2,
        68168 => 2,
        68176 => 2,
        68177 => 2,
        68178 => 2,
        68179 => 2,
        68180 => 2,
        68181 => 2,
        68182 => 2,
        68183 => 2,
        68184 => 2,
        68221 => 2,
        68222 => 2,
        68223 => 2,
        68253 => 2,
        68254 => 2,
        68255 => 2,
        68288 => 1,
        68289 => 1,
        68290 => 1,
        68291 => 1,
        68292 => 1,
        68293 => 1,
        68294 => 1,
        68295 => 1,
        68296 => 2,
        68331 => 2,
        68332 => 2,
        68333 => 2,
        68334 => 2,
        68335 => 2,
        68336 => 2,
        68337 => 2,
        68338 => 2,
        68339 => 2,
        68340 => 2,
        68341 => 2,
        68342 => 2,
        68409 => 2,
        68410 => 2,
        68411 => 2,
        68412 => 2,
        68413 => 2,
        68414 => 2,
        68415 => 2,
        68416 => 1,
        68417 => 1,
        68418 => 1,
        68419 => 1,
        68420 => 1,
        68421 => 1,
        68422 => 1,
        68423 => 1,
        68424 => 1,
        68425 => 1,
        68426 => 1,
        68427 => 1,
        68428 => 1,
        68429 => 1,
        68430 => 1,
        68431 => 1,
        68432 => 1,
        68433 => 1,
        68434 => 1,
        68435 => 1,
        68436 => 1,
        68437 => 1,
        68440 => 2,
        68441 => 2,
        68442 => 2,
        68443 => 2,
        68444 => 2,
        68445 => 2,
        68446 => 2,
        68447 => 2,
        68448 => 1,
        68449 => 1,
        68450 => 1,
        68451 => 1,
        68452 => 1,
        68453 => 1,
        68454 => 1,
        68455 => 1,
        68456 => 1,
        68457 => 1,
        68458 => 1,
        68459 => 1,
        68460 => 1,
        68461 => 1,
        68462 => 1,
        68463 => 1,
        68464 => 1,
        68465 => 1,
        68466 => 1,
        68472 => 2,
        68473 => 2,
        68474 => 2,
        68475 => 2,
        68476 => 2,
        68477 => 2,
        68478 => 2,
        68479 => 2,
        68480 => 1,
        68481 => 1,
        68482 => 1,
        68483 => 1,
        68484 => 1,
        68485 => 1,
        68486 => 1,
        68487 => 1,
        68488 => 1,
        68489 => 1,
        68490 => 1,
        68491 => 1,
        68492 => 1,
        68493 => 1,
        68494 => 1,
        68495 => 1,
        68496 => 1,
        68497 => 1,
        68505 => 2,
        68506 => 2,
        68507 => 2,
        68508 => 2,
        68521 => 2,
        68522 => 2,
        68523 => 2,
        68524 => 2,
        68525 => 2,
        68526 => 2,
        68527 => 2,
        68858 => 2,
        68859 => 2,
        68860 => 2,
        68861 => 2,
        68862 => 2,
        68863 => 2,
        68912 => 1,
        68913 => 1,
        68914 => 1,
        68915 => 1,
        68916 => 1,
        68917 => 1,
        68918 => 1,
        68919 => 1,
        68920 => 1,
        68921 => 1,
        69291 => 1,
        69292 => 1,
        69293 => 2,
        69296 => 1,
        69297 => 1,
        69373 => 1,
        69374 => 1,
        69375 => 1,
        69405 => 2,
        69406 => 2,
        69407 => 2,
        69408 => 2,
        69409 => 2,
        69410 => 2,
        69411 => 2,
        69412 => 2,
        69413 => 2,
        69414 => 2,
        69415 => 1,
        69457 => 2,
        69458 => 2,
        69459 => 2,
        69460 => 2,
        69461 => 2,
        69462 => 2,
        69463 => 2,
        69464 => 2,
        69465 => 2,
        69488 => 1,
        69489 => 1,
        69490 => 1,
        69491 => 1,
        69492 => 1,
        69493 => 1,
        69494 => 1,
        69495 => 1,
        69496 => 1,
        69497 => 1,
        69498 => 1,
        69499 => 1,
        69500 => 1,
        69501 => 1,
        69502 => 1,
        69503 => 1,
        69504 => 1,
        69505 => 1,
        69506 => 1,
        69507 => 1,
        69508 => 1,
        69509 => 1,
        69510 => 2,
        69511 => 2,
        69512 => 2,
        69513 => 2,
        69552 => 1,
        69553 => 1,
        69554 => 1,
        69555 => 1,
        69556 => 1,
        69557 => 1,
        69558 => 1,
        69559 => 1,
        69560 => 1,
        69561 => 1,
        69562 => 1,
        69563 => 1,
        69564 => 1,
        69565 => 1,
        69566 => 1,
        69567 => 1,
        69568 => 1,
        69569 => 1,
        69570 => 1,
        69571 => 1,
        69572 => 1,
        69573 => 2,
        69574 => 2,
        69575 => 2,
        69576 => 2,
        69577 => 2,
        69578 => 2,
        69579 => 2,
        69600 => 1,
        69601 => 1,
        69602 => 1,
        69603 => 1,
        69604 => 1,
        69605 => 1,
        69606 => 1,
        69607 => 1,
        69608 => 1,
        69609 => 1,
        69610 => 1,
        69611 => 1,
        69612 => 1,
        69613 => 1,
        69614 => 1,
        69615 => 1,
        69616 => 1,
        69617 => 1,
        69618 => 1,
        69619 => 1,
        69620 => 1,
        69621 => 1,
        69622 => 1,
        69703 => 2,
        69704 => 2,
        69705 => 2,
        69706 => 2,
        69707 => 2,
        69708 => 2,
        69709 => 2,
        69714 => 2,
        69715 => 2,
        69716 => 2,
        69717 => 2,
        69718 => 2,
        69719 => 2,
        69720 => 2,
        69721 => 2,
        69722 => 2,
        69723 => 2,
        69724 => 2,
        69725 => 2,
        69726 => 2,
        69727 => 2,
        69728 => 2,
        69729 => 2,
        69730 => 2,
        69731 => 2,
        69732 => 2,
        69733 => 2,
        69734 => 1,
        69735 => 1,
        69736 => 1,
        69737 => 1,
        69738 => 1,
        69739 => 1,
        69740 => 1,
        69741 => 1,
        69742 => 1,
        69743 => 1,
        69744 => 1,
        69745 => 1,
        69746 => 1,
        69747 => 1,
        69748 => 1,
        69749 => 1,
        69759 => 1,
        69819 => 2,
        69820 => 2,
        69822 => 2,
        69823 => 2,
        69824 => 2,
        69825 => 2,
        69826 => 1,
        69840 => 1,
        69841 => 1,
        69842 => 1,
        69843 => 1,
        69844 => 1,
        69845 => 1,
        69846 => 1,
        69847 => 1,
        69848 => 1,
        69849 => 1,
        69850 => 1,
        69851 => 1,
        69852 => 1,
        69853 => 1,
        69854 => 1,
        69855 => 1,
        69856 => 1,
        69857 => 1,
        69858 => 1,
        69859 => 1,
        69860 => 1,
        69861 => 1,
        69862 => 1,
        69863 => 1,
        69864 => 1,
        69872 => 1,
        69873 => 1,
        69874 => 1,
        69875 => 1,
        69876 => 1,
        69877 => 1,
        69878 => 1,
        69879 => 1,
        69880 => 1,
        69881 => 1,
        69942 => 1,
        69943 => 1,
        69944 => 1,
        69945 => 1,
        69946 => 1,
        69947 => 1,
        69948 => 1,
        69949 => 1,
        69950 => 1,
        69951 => 1,
        69952 => 2,
        69953 => 2,
        69954 => 2,
        69955 => 2,
        69956 => 1,
        69957 => 1,
        69958 => 1,
        69959 => 1,
        70004 => 2,
        70005 => 2,
        70006 => 1,
        70085 => 2,
        70086 => 2,
        70087 => 2,
        70088 => 2,
        70089 => 1,
        70090 => 1,
        70091 => 1,
        70092 => 1,
        70093 => 2,
        70094 => 1,
        70095 => 1,
        70096 => 1,
        70097 => 1,
        70098 => 1,
        70099 => 1,
        70100 => 1,
        70101 => 1,
        70102 => 1,
        70103 => 1,
        70104 => 1,
        70105 => 1,
        70106 => 1,
        70107 => 2,
        70108 => 1,
        70109 => 2,
        70110 => 2,
        70111 => 2,
        70113 => 2,
        70114 => 2,
        70115 => 2,
        70116 => 2,
        70117 => 2,
        70118 => 2,
        70119 => 2,
        70120 => 2,
        70121 => 2,
        70122 => 2,
        70123 => 2,
        70124 => 2,
        70125 => 2,
        70126 => 2,
        70127 => 2,
        70128 => 2,
        70129 => 2,
        70130 => 2,
        70131 => 2,
        70132 => 2,
        70144 => 1,
        70145 => 1,
        70146 => 1,
        70147 => 1,
        70148 => 1,
        70149 => 1,
        70150 => 1,
        70151 => 1,
        70152 => 1,
        70153 => 1,
        70154 => 1,
        70155 => 1,
        70156 => 1,
        70157 => 1,
        70158 => 1,
        70159 => 1,
        70160 => 1,
        70161 => 1,
        70200 => 2,
        70201 => 2,
        70202 => 2,
        70203 => 2,
        70204 => 2,
        70205 => 2,
        70206 => 1,
        70207 => 1,
        70208 => 1,
        70209 => 1,
        70272 => 1,
        70273 => 1,
        70274 => 1,
        70275 => 1,
        70276 => 1,
        70277 => 1,
        70278 => 1,
        70280 => 1,
        70282 => 1,
        70283 => 1,
        70284 => 1,
        70285 => 1,
        70287 => 1,
        70288 => 1,
        70289 => 1,
        70290 => 1,
        70291 => 1,
        70292 => 1,
        70293 => 1,
        70294 => 1,
        70295 => 1,
        70296 => 1,
        70297 => 1,
        70298 => 1,
        70299 => 1,
        70300 => 1,
        70301 => 1,
        70303 => 1,
        70304 => 1,
        70305 => 1,
        70306 => 1,
        70307 => 1,
        70308 => 1,
        70309 => 1,
        70310 => 1,
        70311 => 1,
        70312 => 1,
        70313 => 2,
        70384 => 1,
        70385 => 1,
        70386 => 1,
        70387 => 1,
        70388 => 1,
        70389 => 1,
        70390 => 1,
        70391 => 1,
        70392 => 1,
        70393 => 1,
        70400 => 1,
        70401 => 1,
        70402 => 1,
        70403 => 1,
        70405 => 1,
        70406 => 1,
        70407 => 1,
        70408 => 1,
        70409 => 1,
        70410 => 1,
        70411 => 1,
        70412 => 1,
        70415 => 1,
        70416 => 1,
        70419 => 1,
        70420 => 1,
        70421 => 1,
        70422 => 1,
        70423 => 1,
        70424 => 1,
        70425 => 1,
        70426 => 1,
        70427 => 1,
        70428 => 1,
        70429 => 1,
        70430 => 1,
        70431 => 1,
        70432 => 1,
        70433 => 1,
        70434 => 1,
        70435 => 1,
        70436 => 1,
        70437 => 1,
        70438 => 1,
        70439 => 1,
        70440 => 1,
        70442 => 1,
        70443 => 1,
        70444 => 1,
        70445 => 1,
        70446 => 1,
        70447 => 1,
        70448 => 1,
        70450 => 1,
        70451 => 1,
        70453 => 1,
        70454 => 1,
        70455 => 1,
        70456 => 1,
        70457 => 1,
        70459 => 1,
        70460 => 1,
        70461 => 1,
        70462 => 1,
        70463 => 1,
        70464 => 1,
        70465 => 1,
        70466 => 1,
        70467 => 1,
        70468 => 1,
        70471 => 1,
        70472 => 1,
        70475 => 1,
        70476 => 1,
        70477 => 1,
        70480 => 1,
        70487 => 1,
        70493 => 1,
        70494 => 1,
        70495 => 1,
        70496 => 1,
        70497 => 1,
        70498 => 1,
        70499 => 1,
        70502 => 1,
        70503 => 1,
        70504 => 1,
        70505 => 1,
        70506 => 1,
        70507 => 1,
        70508 => 1,
        70512 => 1,
        70513 => 1,
        70514 => 1,
        70515 => 1,
        70516 => 1,
        70731 => 2,
        70732 => 2,
        70733 => 2,
        70734 => 2,
        70735 => 2,
        70736 => 1,
        70737 => 1,
        70738 => 1,
        70739 => 1,
        70740 => 1,
        70741 => 1,
        70742 => 1,
        70743 => 1,
        70744 => 1,
        70745 => 1,
        70746 => 2,
        70747 => 2,
        70749 => 2,
        70750 => 1,
        70751 => 1,
        70752 => 1,
        70753 => 1,
        70854 => 2,
        70855 => 1,
        70864 => 1,
        70865 => 1,
        70866 => 1,
        70867 => 1,
        70868 => 1,
        70869 => 1,
        70870 => 1,
        70871 => 1,
        70872 => 1,
        70873 => 1,
        71096 => 1,
        71097 => 1,
        71098 => 1,
        71099 => 1,
        71100 => 1,
        71101 => 1,
        71102 => 1,
        71103 => 1,
        71104 => 1,
        71105 => 2,
        71106 => 2,
        71107 => 2,
        71108 => 2,
        71109 => 2,
        71110 => 2,
        71111 => 2,
        71112 => 2,
        71113 => 2,
        71114 => 2,
        71115 => 2,
        71116 => 2,
        71117 => 2,
        71118 => 2,
        71119 => 2,
        71120 => 2,
        71121 => 2,
        71122 => 2,
        71123 => 2,
        71124 => 2,
        71125 => 2,
        71126 => 2,
        71127 => 2,
        71128 => 1,
        71129 => 1,
        71130 => 1,
        71131 => 1,
        71132 => 1,
        71133 => 1,
        71233 => 2,
        71234 => 2,
        71235 => 2,
        71236 => 1,
        71248 => 1,
        71249 => 1,
        71250 => 1,
        71251 => 1,
        71252 => 1,
        71253 => 1,
        71254 => 1,
        71255 => 1,
        71256 => 1,
        71257 => 1,
        71264 => 2,
        71265 => 2,
        71266 => 2,
        71267 => 2,
        71268 => 2,
        71269 => 2,
        71270 => 2,
        71271 => 2,
        71272 => 2,
        71273 => 2,
        71274 => 2,
        71275 => 2,
        71276 => 2,
        71352 => 1,
        71353 => 2,
        71360 => 1,
        71361 => 1,
        71362 => 1,
        71363 => 1,
        71364 => 1,
        71365 => 1,
        71366 => 1,
        71367 => 1,
        71368 => 1,
        71369 => 1,
        71424 => 1,
        71425 => 1,
        71426 => 1,
        71427 => 1,
        71428 => 1,
        71429 => 1,
        71430 => 1,
        71431 => 1,
        71432 => 1,
        71433 => 1,
        71434 => 1,
        71435 => 1,
        71436 => 1,
        71437 => 1,
        71438 => 1,
        71439 => 1,
        71440 => 1,
        71441 => 1,
        71442 => 1,
        71443 => 1,
        71444 => 1,
        71445 => 1,
        71446 => 1,
        71447 => 1,
        71448 => 1,
        71449 => 1,
        71450 => 1,
        71453 => 1,
        71454 => 1,
        71455 => 1,
        71456 => 1,
        71457 => 1,
        71458 => 1,
        71459 => 1,
        71460 => 1,
        71461 => 1,
        71462 => 1,
        71463 => 1,
        71464 => 1,
        71465 => 1,
        71466 => 1,
        71467 => 1,
        71472 => 1,
        71473 => 1,
        71474 => 1,
        71475 => 1,
        71476 => 1,
        71477 => 1,
        71478 => 1,
        71479 => 1,
        71480 => 1,
        71481 => 1,
        71482 => 2,
        71483 => 2,
        71484 => 2,
        71485 => 2,
        71486 => 2,
        71487 => 2,
        71488 => 1,
        71489 => 1,
        71490 => 1,
        71491 => 1,
        71492 => 1,
        71493 => 1,
        71494 => 1,
        71739 => 2,
        71914 => 2,
        71915 => 2,
        71916 => 2,
        71917 => 2,
        71918 => 2,
        71919 => 2,
        71920 => 2,
        71921 => 2,
        71922 => 2,
        71935 => 1,
        71936 => 1,
        71937 => 1,
        71938 => 1,
        71939 => 1,
        71940 => 1,
        71941 => 1,
        71942 => 1,
        71945 => 1,
        71948 => 1,
        71949 => 1,
        71950 => 1,
        71951 => 1,
        71952 => 1,
        71953 => 1,
        71954 => 1,
        71955 => 1,
        71957 => 1,
        71958 => 1,
        71991 => 1,
        71992 => 1,
        71995 => 1,
        71996 => 1,
        71997 => 1,
        71998 => 1,
        71999 => 1,
        72000 => 1,
        72001 => 1,
        72002 => 1,
        72003 => 1,
        72004 => 2,
        72005 => 2,
        72006 => 2,
        72016 => 1,
        72017 => 1,
        72018 => 1,
        72019 => 1,
        72020 => 1,
        72021 => 1,
        72022 => 1,
        72023 => 1,
        72024 => 1,
        72025 => 1,
        72096 => 1,
        72097 => 1,
        72098 => 1,
        72099 => 1,
        72100 => 1,
        72101 => 1,
        72102 => 1,
        72103 => 1,
        72154 => 1,
        72155 => 1,
        72156 => 1,
        72157 => 1,
        72158 => 1,
        72159 => 1,
        72160 => 1,
        72161 => 1,
        72162 => 2,
        72163 => 1,
        72164 => 1,
        72255 => 2,
        72256 => 2,
        72257 => 2,
        72258 => 2,
        72259 => 2,
        72260 => 2,
        72261 => 2,
        72262 => 2,
        72263 => 1,
        72324 => 1,
        72325 => 1,
        72326 => 1,
        72327 => 1,
        72328 => 1,
        72329 => 1,
        72330 => 1,
        72331 => 1,
        72332 => 1,
        72333 => 1,
        72334 => 1,
        72335 => 1,
        72336 => 1,
        72337 => 1,
        72338 => 1,
        72339 => 1,
        72340 => 1,
        72341 => 1,
        72342 => 1,
        72343 => 1,
        72344 => 1,
        72345 => 1,
        72346 => 2,
        72347 => 2,
        72348 => 2,
        72349 => 1,
        72350 => 2,
        72351 => 2,
        72352 => 2,
        72353 => 2,
        72354 => 2,
        72368 => 1,
        72369 => 1,
        72370 => 1,
        72371 => 1,
        72372 => 1,
        72373 => 1,
        72374 => 1,
        72375 => 1,
        72376 => 1,
        72377 => 1,
        72378 => 1,
        72379 => 1,
        72380 => 1,
        72381 => 1,
        72382 => 1,
        72383 => 1,
        72448 => 2,
        72449 => 2,
        72450 => 2,
        72451 => 2,
        72452 => 2,
        72453 => 2,
        72454 => 2,
        72455 => 2,
        72456 => 2,
        72457 => 2,
        72704 => 1,
        72705 => 1,
        72706 => 1,
        72707 => 1,
        72708 => 1,
        72709 => 1,
        72710 => 1,
        72711 => 1,
        72712 => 1,
        72760 => 1,
        72761 => 1,
        72762 => 1,
        72763 => 1,
        72764 => 1,
        72765 => 1,
        72766 => 1,
        72767 => 1,
        72768 => 1,
        72769 => 2,
        72770 => 2,
        72771 => 2,
        72772 => 2,
        72773 => 2,
        72784 => 1,
        72785 => 1,
        72786 => 1,
        72787 => 1,
        72788 => 1,
        72789 => 1,
        72790 => 1,
        72791 => 1,
        72792 => 1,
        72793 => 1,
        72794 => 2,
        72795 => 2,
        72796 => 2,
        72797 => 2,
        72798 => 2,
        72799 => 2,
        72800 => 2,
        72801 => 2,
        72802 => 2,
        72803 => 2,
        72804 => 2,
        72805 => 2,
        72806 => 2,
        72807 => 2,
        72808 => 2,
        72809 => 2,
        72810 => 2,
        72811 => 2,
        72812 => 2,
        72816 => 2,
        72817 => 2,
        72850 => 1,
        72851 => 1,
        72852 => 1,
        72853 => 1,
        72854 => 1,
        72855 => 1,
        72856 => 1,
        72857 => 1,
        72858 => 1,
        72859 => 1,
        72860 => 1,
        72861 => 1,
        72862 => 1,
        72863 => 1,
        72864 => 1,
        72865 => 1,
        72866 => 1,
        72867 => 1,
        72868 => 1,
        72869 => 1,
        72870 => 1,
        72871 => 1,
        72873 => 1,
        72874 => 1,
        72875 => 1,
        72876 => 1,
        72877 => 1,
        72878 => 1,
        72879 => 1,
        72880 => 1,
        72881 => 1,
        72882 => 1,
        72883 => 1,
        72884 => 1,
        72885 => 1,
        72886 => 1,
        72960 => 1,
        72961 => 1,
        72962 => 1,
        72963 => 1,
        72964 => 1,
        72965 => 1,
        72966 => 1,
        72968 => 1,
        72969 => 1,
        73018 => 1,
        73020 => 1,
        73021 => 1,
        73023 => 1,
        73024 => 1,
        73025 => 1,
        73026 => 1,
        73027 => 1,
        73028 => 1,
        73029 => 1,
        73030 => 1,
        73031 => 1,
        73040 => 1,
        73041 => 1,
        73042 => 1,
        73043 => 1,
        73044 => 1,
        73045 => 1,
        73046 => 1,
        73047 => 1,
        73048 => 1,
        73049 => 1,
        73056 => 1,
        73057 => 1,
        73058 => 1,
        73059 => 1,
        73060 => 1,
        73061 => 1,
        73063 => 1,
        73064 => 1,
        73104 => 1,
        73105 => 1,
        73107 => 1,
        73108 => 1,
        73109 => 1,
        73110 => 1,
        73111 => 1,
        73112 => 1,
        73120 => 1,
        73121 => 1,
        73122 => 1,
        73123 => 1,
        73124 => 1,
        73125 => 1,
        73126 => 1,
        73127 => 1,
        73128 => 1,
        73129 => 1,
        73440 => 1,
        73441 => 1,
        73442 => 1,
        73443 => 1,
        73444 => 1,
        73445 => 1,
        73446 => 1,
        73447 => 1,
        73448 => 1,
        73449 => 1,
        73450 => 1,
        73451 => 1,
        73452 => 1,
        73453 => 1,
        73454 => 1,
        73455 => 1,
        73456 => 1,
        73457 => 1,
        73458 => 1,
        73459 => 1,
        73460 => 1,
        73461 => 1,
        73462 => 1,
        73463 => 2,
        73464 => 2,
        73472 => 1,
        73473 => 1,
        73474 => 1,
        73475 => 1,
        73476 => 1,
        73477 => 1,
        73478 => 1,
        73479 => 1,
        73480 => 1,
        73481 => 1,
        73482 => 1,
        73483 => 1,
        73484 => 1,
        73485 => 1,
        73486 => 1,
        73487 => 1,
        73488 => 1,
        73534 => 1,
        73535 => 1,
        73536 => 1,
        73537 => 1,
        73538 => 1,
        73539 => 2,
        73540 => 2,
        73541 => 2,
        73542 => 2,
        73543 => 2,
        73544 => 2,
        73545 => 2,
        73546 => 2,
        73547 => 2,
        73548 => 2,
        73549 => 2,
        73550 => 2,
        73551 => 2,
        73552 => 1,
        73553 => 1,
        73554 => 1,
        73555 => 1,
        73556 => 1,
        73557 => 1,
        73558 => 1,
        73559 => 1,
        73560 => 1,
        73561 => 1,
        73648 => 1,
        73727 => 2,
        74649 => 1,
        74851 => 2,
        74852 => 2,
        74853 => 2,
        74854 => 2,
        74855 => 2,
        74856 => 2,
        74857 => 2,
        74858 => 2,
        74859 => 2,
        74860 => 2,
        74861 => 2,
        74862 => 2,
        74864 => 2,
        74865 => 2,
        74866 => 2,
        74867 => 2,
        74868 => 2,
        77809 => 2,
        77810 => 2,
        78895 => 1,
        78912 => 1,
        78913 => 1,
        78914 => 1,
        78915 => 1,
        78916 => 1,
        78917 => 1,
        78918 => 1,
        78919 => 1,
        78920 => 1,
        78921 => 1,
        78922 => 1,
        78923 => 1,
        78924 => 1,
        78925 => 1,
        78926 => 1,
        78927 => 1,
        78928 => 1,
        78929 => 1,
        78930 => 1,
        78931 => 1,
        78932 => 1,
        78933 => 1,
        92768 => 1,
        92769 => 1,
        92770 => 1,
        92771 => 1,
        92772 => 1,
        92773 => 1,
        92774 => 1,
        92775 => 1,
        92776 => 1,
        92777 => 1,
        92782 => 2,
        92783 => 2,
        92864 => 1,
        92865 => 1,
        92866 => 1,
        92867 => 1,
        92868 => 1,
        92869 => 1,
        92870 => 1,
        92871 => 1,
        92872 => 1,
        92873 => 1,
        92912 => 1,
        92913 => 1,
        92914 => 1,
        92915 => 1,
        92916 => 1,
        92917 => 2,
        92983 => 2,
        92984 => 2,
        92985 => 2,
        92986 => 2,
        92987 => 2,
        92988 => 2,
        92989 => 2,
        92990 => 2,
        92991 => 2,
        92992 => 1,
        92993 => 1,
        92994 => 1,
        92995 => 1,
        92996 => 2,
        92997 => 2,
        93008 => 1,
        93009 => 1,
        93010 => 1,
        93011 => 1,
        93012 => 1,
        93013 => 1,
        93014 => 1,
        93015 => 1,
        93016 => 1,
        93017 => 1,
        93019 => 2,
        93020 => 2,
        93021 => 2,
        93022 => 2,
        93023 => 2,
        93024 => 2,
        93025 => 2,
        93027 => 1,
        93028 => 1,
        93029 => 1,
        93030 => 1,
        93031 => 1,
        93032 => 1,
        93033 => 1,
        93034 => 1,
        93035 => 1,
        93036 => 1,
        93037 => 1,
        93038 => 1,
        93039 => 1,
        93040 => 1,
        93041 => 1,
        93042 => 1,
        93043 => 1,
        93044 => 1,
        93045 => 1,
        93046 => 1,
        93047 => 1,
        93053 => 1,
        93054 => 1,
        93055 => 1,
        93056 => 1,
        93057 => 1,
        93058 => 1,
        93059 => 1,
        93060 => 1,
        93061 => 1,
        93062 => 1,
        93063 => 1,
        93064 => 1,
        93065 => 1,
        93066 => 1,
        93067 => 1,
        93068 => 1,
        93069 => 1,
        93070 => 1,
        93071 => 1,
        93824 => 2,
        93825 => 2,
        93826 => 2,
        93827 => 2,
        93828 => 2,
        93829 => 2,
        93830 => 2,
        93831 => 2,
        93832 => 2,
        93833 => 2,
        93834 => 2,
        93835 => 2,
        93836 => 2,
        93837 => 2,
        93838 => 2,
        93839 => 2,
        93840 => 2,
        93841 => 2,
        93842 => 2,
        93843 => 2,
        93844 => 2,
        93845 => 2,
        93846 => 2,
        93847 => 2,
        93848 => 2,
        93849 => 2,
        93850 => 2,
        94021 => 1,
        94022 => 1,
        94023 => 1,
        94024 => 1,
        94025 => 1,
        94026 => 1,
        94031 => 1,
        94079 => 1,
        94080 => 1,
        94081 => 1,
        94082 => 1,
        94083 => 1,
        94084 => 1,
        94085 => 1,
        94086 => 1,
        94087 => 1,
        94095 => 1,
        94096 => 1,
        94097 => 1,
        94098 => 1,
        94099 => 1,
        94100 => 1,
        94101 => 1,
        94102 => 1,
        94103 => 1,
        94104 => 1,
        94105 => 1,
        94106 => 1,
        94107 => 1,
        94108 => 1,
        94109 => 1,
        94110 => 1,
        94111 => 1,
        94176 => 1,
        94177 => 1,
        94178 => 2,
        94179 => 1,
        94180 => 1,
        94192 => 1,
        94193 => 1,
        100333 => 1,
        100334 => 1,
        100335 => 1,
        100336 => 1,
        100337 => 1,
        100338 => 1,
        100339 => 1,
        100340 => 1,
        100341 => 1,
        100342 => 1,
        100343 => 1,
        101632 => 1,
        101633 => 1,
        101634 => 1,
        101635 => 1,
        101636 => 1,
        101637 => 1,
        101638 => 1,
        101639 => 1,
        101640 => 1,
        110576 => 1,
        110577 => 1,
        110578 => 1,
        110579 => 1,
        110581 => 1,
        110582 => 1,
        110583 => 1,
        110584 => 1,
        110585 => 1,
        110586 => 1,
        110587 => 1,
        110589 => 1,
        110590 => 1,
        110592 => 1,
        110593 => 1,
        110879 => 1,
        110880 => 1,
        110881 => 1,
        110882 => 1,
        110898 => 1,
        110928 => 1,
        110929 => 1,
        110930 => 1,
        110933 => 1,
        110948 => 1,
        110949 => 1,
        110950 => 1,
        110951 => 1,
        113776 => 1,
        113777 => 1,
        113778 => 1,
        113779 => 1,
        113780 => 1,
        113781 => 1,
        113782 => 1,
        113783 => 1,
        113784 => 1,
        113785 => 1,
        113786 => 1,
        113787 => 1,
        113788 => 1,
        113792 => 1,
        113793 => 1,
        113794 => 1,
        113795 => 1,
        113796 => 1,
        113797 => 1,
        113798 => 1,
        113799 => 1,
        113800 => 1,
        113808 => 1,
        113809 => 1,
        113810 => 1,
        113811 => 1,
        113812 => 1,
        113813 => 1,
        113814 => 1,
        113815 => 1,
        113816 => 1,
        113817 => 1,
        113820 => 2,
        113821 => 1,
        113822 => 1,
        113823 => 2,
        118576 => 1,
        118577 => 1,
        118578 => 1,
        118579 => 1,
        118580 => 1,
        118581 => 1,
        118582 => 1,
        118583 => 1,
        118584 => 1,
        118585 => 1,
        118586 => 1,
        118587 => 1,
        118588 => 1,
        118589 => 1,
        118590 => 1,
        118591 => 1,
        118592 => 1,
        118593 => 1,
        118594 => 1,
        118595 => 1,
        118596 => 1,
        118597 => 1,
        118598 => 1,
        119081 => 2,
        119141 => 2,
        119142 => 2,
        119143 => 2,
        119144 => 2,
        119145 => 2,
        119146 => 2,
        119147 => 2,
        119148 => 2,
        119149 => 2,
        119150 => 2,
        119151 => 2,
        119152 => 2,
        119153 => 2,
        119154 => 2,
        119262 => 2,
        119263 => 2,
        119264 => 2,
        119265 => 2,
        119266 => 2,
        119267 => 2,
        119268 => 2,
        119269 => 2,
        119270 => 2,
        119271 => 2,
        119272 => 2,
        119273 => 2,
        119274 => 2,
        119488 => 2,
        119489 => 2,
        119490 => 2,
        119491 => 2,
        119492 => 2,
        119493 => 2,
        119494 => 2,
        119495 => 2,
        119496 => 2,
        119497 => 2,
        119498 => 2,
        119499 => 2,
        119500 => 2,
        119501 => 2,
        119502 => 2,
        119503 => 2,
        119504 => 2,
        119505 => 2,
        119506 => 2,
        119507 => 2,
        119520 => 2,
        119521 => 2,
        119522 => 2,
        119523 => 2,
        119524 => 2,
        119525 => 2,
        119526 => 2,
        119527 => 2,
        119528 => 2,
        119529 => 2,
        119530 => 2,
        119531 => 2,
        119532 => 2,
        119533 => 2,
        119534 => 2,
        119535 => 2,
        119536 => 2,
        119537 => 2,
        119538 => 2,
        119539 => 2,
        119648 => 2,
        119649 => 2,
        119650 => 2,
        119651 => 2,
        119652 => 2,
        119653 => 2,
        119654 => 2,
        119655 => 2,
        119656 => 2,
        119657 => 2,
        119658 => 2,
        119659 => 2,
        119660 => 2,
        119661 => 2,
        119662 => 2,
        119663 => 2,
        119664 => 2,
        119665 => 2,
        119666 => 2,
        119667 => 2,
        119668 => 2,
        119669 => 2,
        119670 => 2,
        119671 => 2,
        119672 => 2,
        121399 => 2,
        121400 => 2,
        121401 => 2,
        121402 => 2,
        121453 => 2,
        121454 => 2,
        121455 => 2,
        121456 => 2,
        121457 => 2,
        121458 => 2,
        121459 => 2,
        121460 => 2,
        121461 => 1,
        121462 => 2,
        121463 => 2,
        121464 => 2,
        121465 => 2,
        121466 => 2,
        121467 => 2,
        121468 => 2,
        121469 => 2,
        121470 => 2,
        121471 => 2,
        121472 => 2,
        121473 => 2,
        121474 => 2,
        121475 => 2,
        121476 => 1,
        121477 => 2,
        121478 => 2,
        121479 => 2,
        121480 => 2,
        121481 => 2,
        121482 => 2,
        121483 => 2,
        121499 => 1,
        121500 => 1,
        121501 => 1,
        121502 => 1,
        121503 => 1,
        121505 => 1,
        121506 => 1,
        121507 => 1,
        121508 => 1,
        121509 => 1,
        121510 => 1,
        121511 => 1,
        121512 => 1,
        121513 => 1,
        121514 => 1,
        121515 => 1,
        121516 => 1,
        121517 => 1,
        121518 => 1,
        121519 => 1,
        122661 => 1,
        122662 => 1,
        122663 => 1,
        122664 => 1,
        122665 => 1,
        122666 => 1,
        122880 => 1,
        122881 => 1,
        122882 => 1,
        122883 => 1,
        122884 => 1,
        122885 => 1,
        122886 => 1,
        122888 => 1,
        122889 => 1,
        122890 => 1,
        122891 => 1,
        122892 => 1,
        122893 => 1,
        122894 => 1,
        122895 => 1,
        122896 => 1,
        122897 => 1,
        122898 => 1,
        122899 => 1,
        122900 => 1,
        122901 => 1,
        122902 => 1,
        122903 => 1,
        122904 => 1,
        122907 => 1,
        122908 => 1,
        122909 => 1,
        122910 => 1,
        122911 => 1,
        122912 => 1,
        122913 => 1,
        122915 => 1,
        122916 => 1,
        122918 => 1,
        122919 => 1,
        122920 => 1,
        122921 => 1,
        122922 => 1,
        123023 => 1,
        123184 => 1,
        123185 => 1,
        123186 => 1,
        123187 => 1,
        123188 => 1,
        123189 => 1,
        123190 => 1,
        123191 => 1,
        123192 => 1,
        123193 => 1,
        123194 => 1,
        123195 => 1,
        123196 => 1,
        123197 => 1,
        123200 => 1,
        123201 => 1,
        123202 => 1,
        123203 => 1,
        123204 => 1,
        123205 => 1,
        123206 => 1,
        123207 => 1,
        123208 => 1,
        123209 => 1,
        123214 => 1,
        123215 => 2,
        123647 => 2,
        124896 => 1,
        124897 => 1,
        124898 => 1,
        124899 => 1,
        124900 => 1,
        124901 => 1,
        124902 => 1,
        124904 => 1,
        124905 => 1,
        124906 => 1,
        124907 => 1,
        124909 => 1,
        124910 => 1,
        124912 => 1,
        124913 => 1,
        124914 => 1,
        124915 => 1,
        124916 => 1,
        124917 => 1,
        124918 => 1,
        124919 => 1,
        124920 => 1,
        124921 => 1,
        124922 => 1,
        124923 => 1,
        124924 => 1,
        124925 => 1,
        124926 => 1,
        125127 => 2,
        125128 => 2,
        125129 => 2,
        125130 => 2,
        125131 => 2,
        125132 => 2,
        125133 => 2,
        125134 => 2,
        125135 => 2,
        125136 => 1,
        125137 => 1,
        125138 => 1,
        125139 => 1,
        125140 => 1,
        125141 => 1,
        125142 => 1,
        125259 => 1,
        125264 => 1,
        125265 => 1,
        125266 => 1,
        125267 => 1,
        125268 => 1,
        125269 => 1,
        125270 => 1,
        125271 => 1,
        125272 => 1,
        125273 => 1,
        125278 => 2,
        125279 => 2,
        126704 => 2,
        126705 => 2,
        127136 => 2,
        127137 => 2,
        127138 => 2,
        127139 => 2,
        127140 => 2,
        127141 => 2,
        127142 => 2,
        127143 => 2,
        127144 => 2,
        127145 => 2,
        127146 => 2,
        127147 => 2,
        127148 => 2,
        127149 => 2,
        127150 => 2,
        127153 => 2,
        127154 => 2,
        127155 => 2,
        127156 => 2,
        127157 => 2,
        127158 => 2,
        127159 => 2,
        127160 => 2,
        127161 => 2,
        127162 => 2,
        127163 => 2,
        127164 => 2,
        127165 => 2,
        127166 => 2,
        127167 => 2,
        127169 => 2,
        127170 => 2,
        127171 => 2,
        127172 => 2,
        127173 => 2,
        127174 => 2,
        127175 => 2,
        127176 => 2,
        127177 => 2,
        127178 => 2,
        127179 => 2,
        127180 => 2,
        127181 => 2,
        127182 => 2,
        127183 => 2,
        127185 => 2,
        127186 => 2,
        127187 => 2,
        127188 => 2,
        127189 => 2,
        127190 => 2,
        127191 => 2,
        127192 => 2,
        127193 => 2,
        127194 => 2,
        127195 => 2,
        127196 => 2,
        127197 => 2,
        127198 => 2,
        127199 => 2,
        127200 => 2,
        127201 => 2,
        127202 => 2,
        127203 => 2,
        127204 => 2,
        127205 => 2,
        127206 => 2,
        127207 => 2,
        127208 => 2,
        127209 => 2,
        127210 => 2,
        127211 => 2,
        127212 => 2,
        127213 => 2,
        127214 => 2,
        127215 => 2,
        127216 => 2,
        127217 => 2,
        127218 => 2,
        127219 => 2,
        127220 => 2,
        127221 => 2,
        127243 => 2,
        127244 => 2,
        127245 => 2,
        127246 => 2,
        127247 => 2,
        127279 => 2,
        127312 => 2,
        127313 => 2,
        127314 => 2,
        127315 => 2,
        127316 => 2,
        127317 => 2,
        127318 => 2,
        127319 => 2,
        127320 => 2,
        127321 => 2,
        127322 => 2,
        127323 => 2,
        127324 => 2,
        127325 => 2,
        127326 => 2,
        127327 => 2,
        127328 => 2,
        127329 => 2,
        127330 => 2,
        127331 => 2,
        127332 => 2,
        127333 => 2,
        127334 => 2,
        127335 => 2,
        127336 => 2,
        127337 => 2,
        127341 => 2,
        127342 => 2,
        127343 => 2,
        127344 => 2,
        127345 => 2,
        127346 => 2,
        127347 => 2,
        127348 => 2,
        127349 => 2,
        127350 => 2,
        127351 => 2,
        127352 => 2,
        127353 => 2,
        127354 => 2,
        127355 => 2,
        127356 => 2,
        127357 => 2,
        127358 => 2,
        127359 => 2,
        127360 => 2,
        127361 => 2,
        127362 => 2,
        127363 => 2,
        127364 => 2,
        127365 => 2,
        127366 => 2,
        127367 => 2,
        127368 => 2,
        127369 => 2,
        127370 => 2,
        127371 => 2,
        127372 => 2,
        127373 => 2,
        127374 => 2,
        127375 => 2,
        127377 => 2,
        127378 => 2,
        127379 => 2,
        127380 => 2,
        127381 => 2,
        127382 => 2,
        127383 => 2,
        127384 => 2,
        127385 => 2,
        127386 => 2,
        127387 => 2,
        127388 => 2,
        127389 => 2,
        127390 => 2,
        127391 => 2,
        127392 => 2,
        127393 => 2,
        127394 => 2,
        127395 => 2,
        127396 => 2,
        127397 => 2,
        127398 => 2,
        127399 => 2,
        127400 => 2,
        127401 => 2,
        127402 => 2,
        127403 => 2,
        127404 => 2,
        127405 => 2,
        127462 => 2,
        127463 => 2,
        127464 => 2,
        127465 => 2,
        127466 => 2,
        127467 => 2,
        127468 => 2,
        127469 => 2,
        127470 => 2,
        127471 => 2,
        127472 => 2,
        127473 => 2,
        127474 => 2,
        127475 => 2,
        127476 => 2,
        127477 => 2,
        127478 => 2,
        127479 => 2,
        127480 => 2,
        127481 => 2,
        127482 => 2,
        127483 => 2,
        127484 => 2,
        127485 => 2,
        127486 => 2,
        127487 => 2,
        127584 => 2,
        127585 => 2,
        127586 => 2,
        127587 => 2,
        127588 => 2,
        127589 => 2,
        127777 => 2,
        127778 => 2,
        127779 => 2,
        127780 => 2,
        127781 => 2,
        127782 => 2,
        127783 => 2,
        127784 => 2,
        127785 => 2,
        127786 => 2,
        127787 => 2,
        127788 => 2,
        127789 => 2,
        127790 => 2,
        127791 => 2,
        127792 => 2,
        127793 => 2,
        127794 => 2,
        127795 => 2,
        127796 => 2,
        127797 => 2,
        127798 => 2,
        127869 => 2,
        127870 => 2,
        127871 => 2,
        127872 => 2,
        127873 => 2,
        127874 => 2,
        127875 => 2,
        127876 => 2,
        127877 => 2,
        127878 => 2,
        127879 => 2,
        127880 => 2,
        127881 => 2,
        127882 => 2,
        127883 => 2,
        127884 => 2,
        127885 => 2,
        127886 => 2,
        127887 => 2,
        127888 => 2,
        127889 => 2,
        127890 => 2,
        127891 => 2,
        127892 => 2,
        127893 => 2,
        127894 => 2,
        127895 => 2,
        127896 => 2,
        127897 => 2,
        127898 => 2,
        127899 => 2,
        127900 => 2,
        127901 => 2,
        127902 => 2,
        127903 => 2,
        127941 => 2,
        127942 => 2,
        127943 => 2,
        127944 => 2,
        127945 => 2,
        127946 => 2,
        127947 => 2,
        127948 => 2,
        127949 => 2,
        127950 => 2,
        127951 => 2,
        127952 => 2,
        127953 => 2,
        127954 => 2,
        127955 => 2,
        127956 => 2,
        127957 => 2,
        127958 => 2,
        127959 => 2,
        127960 => 2,
        127961 => 2,
        127962 => 2,
        127963 => 2,
        127964 => 2,
        127965 => 2,
        127966 => 2,
        127967 => 2,
        127968 => 2,
        127969 => 2,
        127970 => 2,
        127971 => 2,
        127972 => 2,
        127973 => 2,
        127974 => 2,
        127975 => 2,
        127976 => 2,
        127977 => 2,
        127978 => 2,
        127979 => 2,
        127980 => 2,
        127981 => 2,
        127982 => 2,
        127983 => 2,
        127984 => 2,
        127985 => 2,
        127986 => 2,
        127987 => 2,
        127988 => 2,
        127989 => 2,
        127990 => 2,
        127991 => 2,
        127992 => 2,
        127993 => 2,
        127994 => 2,
        127995 => 2,
        127996 => 2,
        127997 => 2,
        127998 => 2,
        127999 => 2,
        128063 => 2,
        128064 => 2,
        128065 => 2,
        128248 => 2,
        128249 => 2,
        128250 => 2,
        128251 => 2,
        128252 => 2,
        128253 => 2,
        128254 => 2,
        128255 => 2,
        128318 => 2,
        128319 => 2,
        128320 => 2,
        128321 => 2,
        128322 => 2,
        128323 => 2,
        128324 => 2,
        128325 => 2,
        128326 => 2,
        128327 => 2,
        128328 => 2,
        128329 => 2,
        128330 => 2,
        128331 => 2,
        128332 => 2,
        128333 => 2,
        128334 => 2,
        128335 => 2,
        128336 => 2,
        128337 => 2,
        128338 => 2,
        128339 => 2,
        128340 => 2,
        128341 => 2,
        128342 => 2,
        128343 => 2,
        128344 => 2,
        128345 => 2,
        128346 => 2,
        128347 => 2,
        128348 => 2,
        128349 => 2,
        128350 => 2,
        128351 => 2,
        128352 => 2,
        128353 => 2,
        128354 => 2,
        128355 => 2,
        128356 => 2,
        128357 => 2,
        128358 => 2,
        128359 => 2,
        128360 => 2,
        128361 => 2,
        128362 => 2,
        128363 => 2,
        128364 => 2,
        128365 => 2,
        128366 => 2,
        128367 => 2,
        128368 => 2,
        128369 => 2,
        128370 => 2,
        128371 => 2,
        128372 => 2,
        128373 => 2,
        128374 => 2,
        128375 => 2,
        128376 => 2,
        128377 => 2,
        128378 => 2,
        128420 => 2,
        128507 => 2,
        128508 => 2,
        128509 => 2,
        128510 => 2,
        128511 => 2,
        128512 => 2,
        128513 => 2,
        128514 => 2,
        128515 => 2,
        128516 => 2,
        128517 => 2,
        128518 => 2,
        128519 => 2,
        128520 => 2,
        128521 => 2,
        128522 => 2,
        128523 => 2,
        128524 => 2,
        128525 => 2,
        128526 => 2,
        128527 => 2,
        128528 => 2,
        128529 => 2,
        128530 => 2,
        128531 => 2,
        128532 => 2,
        128533 => 2,
        128534 => 2,
        128535 => 2,
        128536 => 2,
        128537 => 2,
        128538 => 2,
        128539 => 2,
        128540 => 2,
        128541 => 2,
        128542 => 2,
        128543 => 2,
        128544 => 2,
        128545 => 2,
        128546 => 2,
        128547 => 2,
        128548 => 2,
        128549 => 2,
        128550 => 2,
        128551 => 2,
        128552 => 2,
        128553 => 2,
        128554 => 2,
        128555 => 2,
        128556 => 2,
        128557 => 2,
        128558 => 2,
        128559 => 2,
        128560 => 2,
        128561 => 2,
        128562 => 2,
        128563 => 2,
        128564 => 2,
        128565 => 2,
        128566 => 2,
        128567 => 2,
        128568 => 2,
        128569 => 2,
        128570 => 2,
        128571 => 2,
        128572 => 2,
        128573 => 2,
        128574 => 2,
        128575 => 2,
        128576 => 2,
        128577 => 2,
        128578 => 2,
        128579 => 2,
        128580 => 2,
        128581 => 2,
        128582 => 2,
        128583 => 2,
        128584 => 2,
        128585 => 2,
        128586 => 2,
        128587 => 2,
        128588 => 2,
        128589 => 2,
        128590 => 2,
        128591 => 2,
        128710 => 2,
        128711 => 2,
        128712 => 2,
        128713 => 2,
        128714 => 2,
        128715 => 2,
        128716 => 2,
        128717 => 2,
        128718 => 2,
        128719 => 2,
        128720 => 2,
        128721 => 2,
        128722 => 2,
        128723 => 2,
        128724 => 2,
        128725 => 2,
        128726 => 2,
        128727 => 2,
        128732 => 2,
        128733 => 2,
        128734 => 2,
        128735 => 2,
        128736 => 2,
        128737 => 2,
        128738 => 2,
        128739 => 2,
        128740 => 2,
        128741 => 2,
        128742 => 2,
        128743 => 2,
        128744 => 2,
        128745 => 2,
        128746 => 2,
        128747 => 2,
        128748 => 2,
        128752 => 2,
        128753 => 2,
        128754 => 2,
        128755 => 2,
        128756 => 2,
        128757 => 2,
        128758 => 2,
        128759 => 2,
        128760 => 2,
        128761 => 2,
        128762 => 2,
        128763 => 2,
        128764 => 2,
        128884 => 2,
        128885 => 2,
        128886 => 2,
        128891 => 2,
        128892 => 2,
        128893 => 2,
        128894 => 2,
        128895 => 2,
        128981 => 2,
        128982 => 2,
        128983 => 2,
        128984 => 2,
        128985 => 2,
        128992 => 2,
        128993 => 2,
        128994 => 2,
        128995 => 2,
        128996 => 2,
        128997 => 2,
        128998 => 2,
        128999 => 2,
        129000 => 2,
        129001 => 2,
        129002 => 2,
        129003 => 2,
        129008 => 2,
        129024 => 2,
        129025 => 2,
        129026 => 2,
        129027 => 2,
        129028 => 2,
        129029 => 2,
        129030 => 2,
        129031 => 2,
        129032 => 2,
        129033 => 2,
        129034 => 2,
        129035 => 2,
        129104 => 2,
        129105 => 2,
        129106 => 2,
        129107 => 2,
        129108 => 2,
        129109 => 2,
        129110 => 2,
        129111 => 2,
        129112 => 2,
        129113 => 2,
        129200 => 2,
        129201 => 2,
        129280 => 2,
        129281 => 2,
        129282 => 2,
        129283 => 2,
        129284 => 2,
        129285 => 2,
        129286 => 2,
        129287 => 2,
        129288 => 2,
        129289 => 2,
        129290 => 2,
        129291 => 2,
        129292 => 2,
        129293 => 2,
        129294 => 2,
        129295 => 2,
        129296 => 2,
        129297 => 2,
        129298 => 2,
        129299 => 2,
        129300 => 2,
        129301 => 2,
        129302 => 2,
        129303 => 2,
        129304 => 2,
        129305 => 2,
        129306 => 2,
        129307 => 2,
        129308 => 2,
        129309 => 2,
        129310 => 2,
        129311 => 2,
        129312 => 2,
        129313 => 2,
        129314 => 2,
        129315 => 2,
        129316 => 2,
        129317 => 2,
        129318 => 2,
        129319 => 2,
        129320 => 2,
        129321 => 2,
        129322 => 2,
        129323 => 2,
        129324 => 2,
        129325 => 2,
        129326 => 2,
        129327 => 2,
        129328 => 2,
        129329 => 2,
        129330 => 2,
        129331 => 2,
        129332 => 2,
        129333 => 2,
        129334 => 2,
        129335 => 2,
        129336 => 2,
        129337 => 2,
        129338 => 2,
        129339 => 2,
        129340 => 2,
        129341 => 2,
        129342 => 2,
        129343 => 2,
        129344 => 2,
        129345 => 2,
        129346 => 2,
        129347 => 2,
        129348 => 2,
        129349 => 2,
        129350 => 2,
        129351 => 2,
        129352 => 2,
        129353 => 2,
        129354 => 2,
        129355 => 2,
        129356 => 2,
        129357 => 2,
        129358 => 2,
        129359 => 2,
        129360 => 2,
        129361 => 2,
        129362 => 2,
        129363 => 2,
        129364 => 2,
        129365 => 2,
        129366 => 2,
        129367 => 2,
        129368 => 2,
        129369 => 2,
        129370 => 2,
        129371 => 2,
        129372 => 2,
        129373 => 2,
        129374 => 2,
        129375 => 2,
        129376 => 2,
        129377 => 2,
        129378 => 2,
        129379 => 2,
        129380 => 2,
        129381 => 2,
        129382 => 2,
        129383 => 2,
        129384 => 2,
        129385 => 2,
        129386 => 2,
        129387 => 2,
        129388 => 2,
        129389 => 2,
        129390 => 2,
        129391 => 2,
        129392 => 2,
        129393 => 2,
        129394 => 2,
        129395 => 2,
        129396 => 2,
        129397 => 2,
        129398 => 2,
        129399 => 2,
        129400 => 2,
        129401 => 2,
        129402 => 2,
        129403 => 2,
        129404 => 2,
        129405 => 2,
        129406 => 2,
        129407 => 2,
        129408 => 2,
        129409 => 2,
        129410 => 2,
        129411 => 2,
        129412 => 2,
        129413 => 2,
        129414 => 2,
        129415 => 2,
        129416 => 2,
        129417 => 2,
        129418 => 2,
        129419 => 2,
        129420 => 2,
        129421 => 2,
        129422 => 2,
        129423 => 2,
        129424 => 2,
        129425 => 2,
        129426 => 2,
        129427 => 2,
        129428 => 2,
        129429 => 2,
        129430 => 2,
        129431 => 2,
        129432 => 2,
        129433 => 2,
        129434 => 2,
        129435 => 2,
        129436 => 2,
        129437 => 2,
        129438 => 2,
        129439 => 2,
        129440 => 2,
        129441 => 2,
        129442 => 2,
        129443 => 2,
        129444 => 2,
        129445 => 2,
        129446 => 2,
        129447 => 2,
        129448 => 2,
        129449 => 2,
        129450 => 2,
        129451 => 2,
        129452 => 2,
        129453 => 2,
        129454 => 2,
        129455 => 2,
        129456 => 2,
        129457 => 2,
        129458 => 2,
        129459 => 2,
        129460 => 2,
        129461 => 2,
        129462 => 2,
        129463 => 2,
        129464 => 2,
        129465 => 2,
        129466 => 2,
        129467 => 2,
        129468 => 2,
        129469 => 2,
        129470 => 2,
        129471 => 2,
        129472 => 2,
        129473 => 2,
        129474 => 2,
        129475 => 2,
        129476 => 2,
        129477 => 2,
        129478 => 2,
        129479 => 2,
        129480 => 2,
        129481 => 2,
        129482 => 2,
        129483 => 2,
        129484 => 2,
        129485 => 2,
        129486 => 2,
        129487 => 2,
        129488 => 2,
        129489 => 2,
        129490 => 2,
        129491 => 2,
        129492 => 2,
        129493 => 2,
        129494 => 2,
        129495 => 2,
        129496 => 2,
        129497 => 2,
        129498 => 2,
        129499 => 2,
        129500 => 2,
        129501 => 2,
        129502 => 2,
        129503 => 2,
        129504 => 2,
        129505 => 2,
        129506 => 2,
        129507 => 2,
        129508 => 2,
        129509 => 2,
        129510 => 2,
        129511 => 2,
        129512 => 2,
        129513 => 2,
        129514 => 2,
        129515 => 2,
        129516 => 2,
        129517 => 2,
        129518 => 2,
        129519 => 2,
        129520 => 2,
        129521 => 2,
        129522 => 2,
        129523 => 2,
        129524 => 2,
        129525 => 2,
        129526 => 2,
        129527 => 2,
        129528 => 2,
        129529 => 2,
        129530 => 2,
        129531 => 2,
        129532 => 2,
        129533 => 2,
        129534 => 2,
        129535 => 2,
        129632 => 2,
        129633 => 2,
        129634 => 2,
        129635 => 2,
        129636 => 2,
        129637 => 2,
        129638 => 2,
        129639 => 2,
        129640 => 2,
        129641 => 2,
        129642 => 2,
        129643 => 2,
        129644 => 2,
        129645 => 2,
        129648 => 2,
        129649 => 2,
        129650 => 2,
        129651 => 2,
        129652 => 2,
        129653 => 2,
        129654 => 2,
        129655 => 2,
        129656 => 2,
        129657 => 2,
        129658 => 2,
        129659 => 2,
        129660 => 2,
        129664 => 2,
        129665 => 2,
        129666 => 2,
        129667 => 2,
        129668 => 2,
        129669 => 2,
        129670 => 2,
        129671 => 2,
        129672 => 2,
        129680 => 2,
        129681 => 2,
        129682 => 2,
        129683 => 2,
        129684 => 2,
        129685 => 2,
        129686 => 2,
        129687 => 2,
        129688 => 2,
        129689 => 2,
        129690 => 2,
        129691 => 2,
        129692 => 2,
        129693 => 2,
        129694 => 2,
        129695 => 2,
        129696 => 2,
        129697 => 2,
        129698 => 2,
        129699 => 2,
        129700 => 2,
        129701 => 2,
        129702 => 2,
        129703 => 2,
        129704 => 2,
        129705 => 2,
        129706 => 2,
        129707 => 2,
        129708 => 2,
        129709 => 2,
        129710 => 2,
        129711 => 2,
        129712 => 2,
        129713 => 2,
        129714 => 2,
        129715 => 2,
        129716 => 2,
        129717 => 2,
        129718 => 2,
        129719 => 2,
        129720 => 2,
        129721 => 2,
        129722 => 2,
        129723 => 2,
        129724 => 2,
        129725 => 2,
        129727 => 2,
        129728 => 2,
        129729 => 2,
        129730 => 2,
        129731 => 2,
        129732 => 2,
        129733 => 2,
        129742 => 2,
        129743 => 2,
        129744 => 2,
        129745 => 2,
        129746 => 2,
        129747 => 2,
        129748 => 2,
        129749 => 2,
        129750 => 2,
        129751 => 2,
        129752 => 2,
        129753 => 2,
        129754 => 2,
        129755 => 2,
        129760 => 2,
        129761 => 2,
        129762 => 2,
        129763 => 2,
        129764 => 2,
        129765 => 2,
        129766 => 2,
        129767 => 2,
        129768 => 2,
        129776 => 2,
        129777 => 2,
        129778 => 2,
        129779 => 2,
        129780 => 2,
        129781 => 2,
        129782 => 2,
        129783 => 2,
        129784 => 2,
        173783 => 1,
        173784 => 1,
        173785 => 1,
        173786 => 1,
        173787 => 1,
        173788 => 1,
        173789 => 1,
        173790 => 1,
        173791 => 1,
        177973 => 1,
        177974 => 1,
        177975 => 1,
        177976 => 1,
        177977 => 1,
    );

    protected static $validDisallowedSTD3 = array(
        0 => 1,
        1 => 1,
        2 => 1,
        3 => 1,
        4 => 1,
        5 => 1,
        6 => 1,
        7 => 1,
        8 => 1,
        9 => 1,
        10 => 1,
        11 => 1,
        12 => 1,
        13 => 1,
        14 => 1,
        15 => 1,
        16 => 1,
        17 => 1,
        18 => 1,
        19 => 1,
        20 => 1,
        21 => 1,
        22 => 1,
        23 => 1,
        24 => 1,
        25 => 1,
        26 => 1,
        27 => 1,
        28 => 1,
        29 => 1,
        30 => 1,
        31 => 1,
        32 => 1,
        33 => 1,
        34 => 1,
        35 => 1,
        36 => 1,
        37 => 1,
        38 => 1,
        39 => 1,
        40 => 1,
        41 => 1,
        42 => 1,
        43 => 1,
        44 => 1,
        47 => 1,
        58 => 1,
        59 => 1,
        60 => 1,
        61 => 1,
        62 => 1,
        63 => 1,
        64 => 1,
        91 => 1,
        92 => 1,
        93 => 1,
        94 => 1,
        95 => 1,
        96 => 1,
        123 => 1,
        124 => 1,
        125 => 1,
        126 => 1,
        127 => 1,
        8800 => 1,
        8814 => 1,
        8815 => 1,
    );

    /**
     * Check if a code point is valid.
     *
     * @param int $codepoint
     * @param int[] $exclude
     * @param bool $useSTD3ASCIIRules
     *
     * @return bool
     */
    public static function isValid($codepoint, array $exclude = array(), $useSTD3ASCIIRules = true)
    {
        $excluded = null;
        if (isset(static::$validSingle[$codepoint])) {
            $excluded = static::$validSingle[$codepoint];
        } elseif (!$useSTD3ASCIIRules && isset(static::$validDisallowedSTD3[$codepoint])) {
            $excluded = static::$validDisallowedSTD3[$codepoint];
        } elseif (
            ($codepoint >= 592 && $codepoint <= 680 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 768 && $codepoint <= 831 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 1072 && $codepoint <= 1103 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 1377 && $codepoint <= 1414 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 1657 && $codepoint <= 1719 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 1808 && $codepoint <= 1836 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 1872 && $codepoint <= 1901 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 1920 && $codepoint <= 1968 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 1984 && $codepoint <= 2037 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 2048 && $codepoint <= 2093 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 2112 && $codepoint <= 2139 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 2309 && $codepoint <= 2361 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 3585 && $codepoint <= 3634 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 4096 && $codepoint <= 4129 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 4186 && $codepoint <= 4249 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 4304 && $codepoint <= 4342 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 4352 && $codepoint <= 4441 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 4449 && $codepoint <= 4514 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 4520 && $codepoint <= 4601 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 4616 && $codepoint <= 4678 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 4704 && $codepoint <= 4742 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 4752 && $codepoint <= 4782 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 4848 && $codepoint <= 4878 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 4896 && $codepoint <= 4934 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 4961 && $codepoint <= 4988 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 5024 && $codepoint <= 5108 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 5121 && $codepoint <= 5740 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 5792 && $codepoint <= 5866 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 6016 && $codepoint <= 6067 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 6070 && $codepoint <= 6099 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 6176 && $codepoint <= 6263 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 6272 && $codepoint <= 6313 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 6320 && $codepoint <= 6389 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 6400 && $codepoint <= 6428 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 6470 && $codepoint <= 6509 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 6528 && $codepoint <= 6569 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 6624 && $codepoint <= 6655 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 6656 && $codepoint <= 6683 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 6688 && $codepoint <= 6750 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 6752 && $codepoint <= 6780 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 6912 && $codepoint <= 6987 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 7040 && $codepoint <= 7082 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 7104 && $codepoint <= 7155 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 7168 && $codepoint <= 7223 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 7245 && $codepoint <= 7293 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 7380 && $codepoint <= 7410 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 7424 && $codepoint <= 7467 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 7545 && $codepoint <= 7578 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 7627 && $codepoint <= 7654 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 8592 && $codepoint <= 8682 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 8704 && $codepoint <= 8747 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 8753 && $codepoint <= 8799 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 8816 && $codepoint <= 8945 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 8962 && $codepoint <= 9000 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 9003 && $codepoint <= 9082 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 9085 && $codepoint <= 9114 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 9115 && $codepoint <= 9166 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 9216 && $codepoint <= 9252 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 9472 && $codepoint <= 9621 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 9632 && $codepoint <= 9711 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 9754 && $codepoint <= 9839 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 9996 && $codepoint <= 10023 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 10025 && $codepoint <= 10059 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 10102 && $codepoint <= 10132 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 10192 && $codepoint <= 10219 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 10240 && $codepoint <= 10495 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 10496 && $codepoint <= 10763 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 10765 && $codepoint <= 10867 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 10871 && $codepoint <= 10971 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 10973 && $codepoint <= 11007 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 11044 && $codepoint <= 11084 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 11126 && $codepoint <= 11157 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 11160 && $codepoint <= 11193 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 11312 && $codepoint <= 11358 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 11520 && $codepoint <= 11557 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 11568 && $codepoint <= 11621 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 11744 && $codepoint <= 11775 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 11936 && $codepoint <= 12018 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 12296 && $codepoint <= 12329 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 12353 && $codepoint <= 12436 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 12449 && $codepoint <= 12542 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 12549 && $codepoint <= 12588 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 13312 && $codepoint <= 19893 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 19904 && $codepoint <= 19967 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 19968 && $codepoint <= 40869 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 40960 && $codepoint <= 42124 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 42192 && $codepoint <= 42237 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 42240 && $codepoint <= 42508 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 42512 && $codepoint <= 42539 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 42656 && $codepoint <= 42725 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 43008 && $codepoint <= 43047 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 43072 && $codepoint <= 43123 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 43136 && $codepoint <= 43204 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 43264 && $codepoint <= 43309 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 43312 && $codepoint <= 43347 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 43360 && $codepoint <= 43388 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 43392 && $codepoint <= 43456 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 43488 && $codepoint <= 43518 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 43520 && $codepoint <= 43574 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 43648 && $codepoint <= 43714 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 43824 && $codepoint <= 43866 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 43968 && $codepoint <= 44010 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 44032 && $codepoint <= 55203 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 55243 && $codepoint <= 55291 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 65664 && $codepoint <= 65786 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 65799 && $codepoint <= 65843 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 65856 && $codepoint <= 65930 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 66000 && $codepoint <= 66044 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 66176 && $codepoint <= 66204 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 66208 && $codepoint <= 66256 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 66304 && $codepoint <= 66334 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 66384 && $codepoint <= 66426 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 66432 && $codepoint <= 66461 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 66464 && $codepoint <= 66499 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 66600 && $codepoint <= 66637 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 66638 && $codepoint <= 66717 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 66776 && $codepoint <= 66811 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 66816 && $codepoint <= 66855 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 66864 && $codepoint <= 66915 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 67072 && $codepoint <= 67382 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 67594 && $codepoint <= 67637 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 67712 && $codepoint <= 67742 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 67968 && $codepoint <= 68023 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 68050 && $codepoint <= 68095 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 68192 && $codepoint <= 68220 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 68224 && $codepoint <= 68252 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 68297 && $codepoint <= 68326 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 68352 && $codepoint <= 68405 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 68608 && $codepoint <= 68680 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 68800 && $codepoint <= 68850 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 68864 && $codepoint <= 68903 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 69216 && $codepoint <= 69246 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 69248 && $codepoint <= 69289 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 69376 && $codepoint <= 69404 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 69424 && $codepoint <= 69456 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 69632 && $codepoint <= 69702 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 69760 && $codepoint <= 69818 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 69888 && $codepoint <= 69940 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 69968 && $codepoint <= 70003 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 70016 && $codepoint <= 70084 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 70163 && $codepoint <= 70199 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 70320 && $codepoint <= 70378 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 70656 && $codepoint <= 70730 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 70784 && $codepoint <= 70853 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 71040 && $codepoint <= 71093 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 71168 && $codepoint <= 71232 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 71296 && $codepoint <= 71351 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 71680 && $codepoint <= 71738 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 71872 && $codepoint <= 71913 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 71960 && $codepoint <= 71989 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 72106 && $codepoint <= 72151 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 72192 && $codepoint <= 72254 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 72272 && $codepoint <= 72323 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 72384 && $codepoint <= 72440 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 72714 && $codepoint <= 72758 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 72818 && $codepoint <= 72847 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 72971 && $codepoint <= 73014 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 73066 && $codepoint <= 73102 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 73490 && $codepoint <= 73530 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 73664 && $codepoint <= 73713 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 73728 && $codepoint <= 74606 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 74607 && $codepoint <= 74648 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 74752 && $codepoint <= 74850 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 74880 && $codepoint <= 75075 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 77712 && $codepoint <= 77808 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 77824 && $codepoint <= 78894 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 82944 && $codepoint <= 83526 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 92160 && $codepoint <= 92728 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 92736 && $codepoint <= 92766 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 92784 && $codepoint <= 92862 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 92880 && $codepoint <= 92909 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 92928 && $codepoint <= 92982 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 93792 && $codepoint <= 93823 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 93952 && $codepoint <= 94020 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 94032 && $codepoint <= 94078 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 94208 && $codepoint <= 100332 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 100352 && $codepoint <= 101106 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 101107 && $codepoint <= 101589 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 110594 && $codepoint <= 110878 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 110960 && $codepoint <= 111355 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 113664 && $codepoint <= 113770 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 118528 && $codepoint <= 118573 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 118608 && $codepoint <= 118723 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 118784 && $codepoint <= 119029 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 119040 && $codepoint <= 119078 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 119082 && $codepoint <= 119133 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 119163 && $codepoint <= 119226 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 119233 && $codepoint <= 119261 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 119296 && $codepoint <= 119365 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 119552 && $codepoint <= 119638 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 120832 && $codepoint <= 121343 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 121344 && $codepoint <= 121398 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 121403 && $codepoint <= 121452 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 122624 && $codepoint <= 122654 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 123136 && $codepoint <= 123180 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 123536 && $codepoint <= 123566 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 123584 && $codepoint <= 123641 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 124112 && $codepoint <= 124153 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 124928 && $codepoint <= 125124 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 125218 && $codepoint <= 125258 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 126065 && $codepoint <= 126132 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 126209 && $codepoint <= 126269 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 126976 && $codepoint <= 127019 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 127024 && $codepoint <= 127123 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 127744 && $codepoint <= 127776 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 127799 && $codepoint <= 127868 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 127904 && $codepoint <= 127940 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 128000 && $codepoint <= 128062 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 128066 && $codepoint <= 128247 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 128256 && $codepoint <= 128317 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 128379 && $codepoint <= 128419 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 128421 && $codepoint <= 128506 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 128592 && $codepoint <= 128639 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 128640 && $codepoint <= 128709 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 128768 && $codepoint <= 128883 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 128896 && $codepoint <= 128980 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 129040 && $codepoint <= 129095 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 129120 && $codepoint <= 129159 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 129168 && $codepoint <= 129197 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 129536 && $codepoint <= 129619 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 129792 && $codepoint <= 129938 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 129940 && $codepoint <= 129994 && $excluded = static::EXCLUDE_ALWAYS)
            || ($codepoint >= 131072 && $codepoint <= 173782 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 173824 && $codepoint <= 177972 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 177984 && $codepoint <= 178205 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 178208 && $codepoint <= 183969 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 183984 && $codepoint <= 191456 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 196608 && $codepoint <= 201546 && $excluded = static::EXCLUDE_NO)
            || ($codepoint >= 201552 && $codepoint <= 205743 && $excluded = static::EXCLUDE_NO)
        ) {
            // noop
        }
        $result = false;
        if ($excluded !== null && !in_array($excluded, $exclude)) {
            $result = true;
        }

        return $result;
    }
}
