<?php

/**
 * Comarch invoices synchronization file.
 *
 * The file is part of the paid functionality. Using the file is allowed only after purchasing a subscription.
 * File modification allowed only with the consent of the system producer.
 *
 * @package Integration
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 */

namespace App\Integrations\Comarch\Xl\Synchronizer;

/**
 * Comarch invoices synchronization class.
 */
class Invoices extends \App\Integrations\Comarch\Synchronizer
{
	/** {@inheritdoc} */
	const LIMIT_NAME = 'invoices_limit';

	/** {@inheritdoc} */
	public function process(): void
	{
		$mapModel = $this->getMapModel();
		if (\App\Module::isModuleActive($mapModel->getModule())) {
			$direction = (int) $this->config->get('direction_invoices');
			if ($this->config->get('master')) {
				if (self::DIRECTION_TWO_WAY === $direction || self::DIRECTION_YF_TO_API === $direction) {
					$this->runQueue('export');
					$this->export();
				}
				if (self::DIRECTION_TWO_WAY === $direction || self::DIRECTION_API_TO_YF === $direction) {
					$this->runQueue('import');
					$this->import();
				}
			} else {
				if (self::DIRECTION_TWO_WAY === $direction || self::DIRECTION_API_TO_YF === $direction) {
					$this->runQueue('import');
					$this->import();
				}
				if (self::DIRECTION_TWO_WAY === $direction || self::DIRECTION_YF_TO_API === $direction) {
					$this->runQueue('export');
					$this->export();
				}
			}
		}
	}

	/**
	 * Import invoices from Comarch.
	 *
	 * @return void
	 */
	public function import(): void
	{
		$this->processStart('import');
		$i = 0;
		try {
			$page = $this->lastScan['page'] ?? 1;
			$load = true;
			$finish = false;
			$limit = $this->config->get(self::LIMIT_NAME);
			while ($load) {
				// if ($rows = $this->getFromApi('Document/GetById/3540')) {
				if ($rows = $this->getFromApi('Document/?typDokumentu=2033&page=' . $page . '&' . $this->getFromApiCond())) {
					foreach ($rows as $id => $row) {
						$this->importItem($row);
						$this->config->setScan('import' . $this->name, 'id', $id);
						++$i;
					}
					++$page;
					if (\is_callable($this->controller->bathCallback)) {
						$load = \call_user_func($this->controller->bathCallback, 'import' . $this->name);
					}
					if ($limit !== \count($rows)) {
						$finish = true;
					}
				} else {
					$finish = true;
				}
				if ($finish || !$load) {
					$load = false;
					if ($finish) {
						$this->config->setEndScan('import' . $this->name, $this->lastScan['start_date']);
					} else {
						$this->config->setScan('import' . $this->name, 'page', $page);
					}
				}
			}
		} catch (\Throwable $ex) {
			$this->logError('import ' . $this->name, null, $ex);
		}
		$this->processEnd('import', ['imported' => $i]);
	}

	/** {@inheritdoc} */
	public function getByIdFromApi(int $apiId): array
	{
		if ($rows = $this->getFromApi('Document/GetById/' . $apiId)) {
			return $rows[0];
		}
		return [];
	}

	/** {@inheritdoc} */
	protected function getExportQuery(): \App\Db\Query
	{
		$query = parent::getExportQuery();
		$query->andWhere(['comarch_invoice_status' => 'PLL_COMARCH_TO_SEND']);
		return $query;
	}
}
