<?php

/**
 * KSeF Single Invoice Client class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Client;

use App\BatchMethod;
use App\Integrations\KSeF\Exception\KSeFXmlException;
use App\Integrations\KSeF\KSeF;
use App\Integrations\KSeF\Model\Client\DownloadQueue;
use App\Integrations\KSeF\Model\Enum\ImportType;
use App\Integrations\KSeF\Model\Enum\Status;
use App\Integrations\KSeF\Repository\DownloadQueueRepository;
use App\Integrations\KSeF\Repository\QueueRepository;
use App\Integrations\KSeF\Service\Connector\ConnectorService;
use App\Integrations\KSeF\Service\Invoice\InvoiceService;
use N1ebieski\KSEFClient\Actions\DecryptDocument\DecryptDocumentAction;
use N1ebieski\KSEFClient\Actions\DecryptDocument\DecryptDocumentHandler;
use N1ebieski\KSEFClient\Factories\EncryptionKeyFactory;

/**
 * KSeF Single Invoice Client class.
 */
final class BatchInvoiceClient
{
	/** @var int Default max invoices per session */
	private const DEFAULT_MAX_INVOICES_PER_SESSION = 50;

	/**
	 * Send invoices in batch mode.
	 *
	 * @param int    $settingId
	 * @param Status $status
	 * @param int    $limit     Max invoices per session (optional)
	 *
	 * @return void
	 */
	public function sendInvoicesInBatchMode(int $settingId, Status $status, int $limit = self::DEFAULT_MAX_INVOICES_PER_SESSION): void
	{
		$connectorService = new ConnectorService();
		$queueRepository = new QueueRepository();
		$invoicesFromQueue = [];
		try {
			$connectorService->createConnection($settingId, EncryptionKeyFactory::makeRandom());
			$invoicesFromQueue = $queueRepository->getByStatus($status, $limit);

			$invoices = [];
			$i = 1;
			foreach ($invoicesFromQueue as $queueObject) {
				$invoiceRecordModel = \Vtiger_Record_Model::getInstanceById($queueObject->getInvoiceRecordId());
				$queueObject->setOrdinalNumber($i++);
				$invoices[$queueObject->getOrdinalNumber()] = (new InvoiceService($invoiceRecordModel))->makeInvoice();

				$queueRepository->update($queueObject);
			}
			if (!empty($invoices)) {
				$sendResponse = $connectorService->sendBatch($invoices);

				foreach ($invoicesFromQueue as $queueObject) {
					$queueObject
						->setStatus(Status::IN_PROGRESS->value)
						->setSessionKey($sendResponse->object()->referenceNumber);
					$queueRepository->update($queueObject);
				}
			}
		} catch (\Throwable $t) {
			KSeF::getLogger()->error($t->getMessage(), ['e' => $t]);
			foreach ($invoicesFromQueue as $queueObject) {
				$queueObject
					->setStatus(Status::ERROR->value)
					->setSessionKey($sendResponse?->object()->referenceNumber);
				$queueRepository->update($queueObject);
			}
		}
		if (isset($sendResponse)) {
			$connectorService->closeBatchSession($sendResponse->object()->referenceNumber);
		}
	}

	/**
	 * Verify and update invoices status.
	 *
	 * @param int $settingId
	 *
	 * @return void
	 */
	public function updateInvoicesStatus(int $settingId): void
	{
		$connectorService = new ConnectorService();
		$queueRepository = new QueueRepository();

		$connectorService->createConnection($settingId, EncryptionKeyFactory::makeRandom());

		$sessionKeys = $queueRepository->findSessionKeys(Status::IN_PROGRESS, $settingId);

		foreach ($sessionKeys as $sessionKey) {
			try {
				$statusResponse = $connectorService->statusBatch(
					$sessionKey,
				);

				foreach ($statusResponse->object()->invoices as $invoice) {
					InvoiceService::updateInvoiceKSeFStatus(
						$sessionKey,
						$invoice->ordinalNumber,
						$invoice->ksefNumber,
						$invoice->referenceNumber,
						$invoice->acquisitionDate,
						$invoice->invoicingMode,
						$invoice->status,
					);
				}
			} catch (\Throwable $t) {
				KSeF::getLogger()->error($t->getMessage(), ['e' => $t, 'configId' => $settingId, 'sessionKey' => $sessionKey]);
			}
		}
	}

	/**
	 * Send request for batch import of invoices from KSeF.
	 *
	 * @param int        $settingId
	 * @param \DateTime  $from
	 * @param \DateTime  $to
	 * @param ImportType $type
	 *
	 * @return void
	 */
	public function requestBatchImport(
		int $settingId,
		\DateTime $from = new \DateTime('-1 day'),
		\DateTime $to = new \DateTime(),
		ImportType $type = ImportType::COST,
	): void {
		$connectorService = new ConnectorService();
		$queueRepository = new DownloadQueueRepository();

		$from->setTimezone(new \DateTimeZone('UTC'));
		$to->setTimezone(new \DateTimeZone('UTC'));

		$encryptionKey = EncryptionKeyFactory::makeRandom();
		$queueObject = (new DownloadQueue())
			->setStatus(Status::NEW->value)
			->setSettingsId($settingId)
			->setKey($encryptionKey->key)
			->setIv($encryptionKey->iv)
			->setModule($type->module());

		try {
			$connectorService->createConnection($settingId, $encryptionKey);

			$response = $connectorService->requestImport($from, $to, $type);
			$queueObject->setReferenceNumber($response->object()->referenceNumber);
			$queueRepository->new($queueObject);
		} catch (\Throwable $t) {
			KSeF::getLogger()->error($t->getMessage(), ['e' => $t, 'params' => ['configId' => $settingId]]);
			$queueObject->setStatus(Status::ERROR->value);
			$queueRepository->new($queueObject);
		}
	}

	/**
	 * Send request for download of invoices pack from KSeF.
	 *
	 * @param int           $settingId
	 * @param DownloadQueue $downloadQueue
	 *
	 * @return void
	 */
	public function downloadBatchImport(int $settingId, DownloadQueue $downloadQueue): void
	{
		$connectorService = new ConnectorService();
		$queueRepository = new DownloadQueueRepository();

		try {
			if (!is_dir(KSeF::ZIP_PATH)) {
				mkdir(KSeF::ZIP_PATH, 0o755, true);
			}
			$connectorService->createConnection($settingId, $downloadQueue->getEncryptionKey());
			$response = $connectorService->downloadImport($downloadQueue->getReferenceNumber());

			if (100 === $response->object()->status->code) {
				return;
			}
			if (200 === $response->object()->status->code) {
				$decryptDocumentHandler = new DecryptDocumentHandler();
				$zipContents = '';
				foreach ($response->object()->package?->parts as $part) {
					$contents = $decryptDocumentHandler->handle(new DecryptDocumentAction(
						encryptionKey: $downloadQueue->getEncryptionKey(),
						document: file_get_contents($part->url)
					));

					$zipContents .= $contents;
				}
				if (!empty($zipContents)) {
					file_put_contents(KSeF::ZIP_PATH . "{$downloadQueue->getReferenceNumber()}.zip", $zipContents);
					$downloadQueue->setStatus(Status::IN_PROGRESS->value);
				} else {
					$downloadQueue->setStatus(Status::CANCELED->value);
				}
			} else {
				throw new KSeFXmlException(\sprintf('Error downloading package - %s [%s]', $response->object()->status->message, $response->object()->status->code));
			}

			$queueRepository->update($downloadQueue);
		} catch (\Throwable $t) {
			KSeF::getLogger()->error($t->getMessage(), ['e' => $t, 'params' => [
				'configId' => $settingId,
				'referenceNumber' => $downloadQueue->getReferenceNumber(),
			]]);
			$downloadQueue->setStatus(Status::ERROR->value);
			$queueRepository->update($downloadQueue);
		}
	}
}
