<?php

/**
 * KSeF Single Invoice Client class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Client;

use App\Integrations\KSeF\Exception\KSeFClientException;
use App\Integrations\KSeF\Exception\KSeFConfigurationException;
use App\Integrations\KSeF\KSeF;
use App\Integrations\KSeF\Model\Enum\Mode;
use App\Integrations\KSeF\Model\Enum\Status;
use App\Integrations\KSeF\Repository\QueueRepository;
use App\Integrations\KSeF\Service\Connector\ConnectorService;
use App\Integrations\KSeF\Service\Invoice\InvoiceService;
use N1ebieski\KSEFClient\Factories\EncryptionKeyFactory;

/**
 * KSeF Single Invoice Client class.
 */
final class SingleInvoiceClient
{
	/** @var int Default max invoices per session */
	private const DEFAULT_MAX_INVOICES_PER_SESSION = 100;

	/**
	 * Send invoices in single mode.
	 *
	 * @param int      $settingId
	 * @param Status   $status
	 * @param int|null $limit
	 *
	 * @throws KSeFClientException
	 * @throws KSeFConfigurationException
	 *
	 * @return void
	 */
	public function sendInvoicesInSingleMode(int $settingId, Status $status, ?int $limit = self::DEFAULT_MAX_INVOICES_PER_SESSION): void
	{
		$connectorService = new ConnectorService();
		$queueRepository = new QueueRepository();

		$connectorService->createConnection($settingId, EncryptionKeyFactory::makeRandom());
		$openResponse = $connectorService->openSession();

		$invoicesFromQueue = $queueRepository->getByStatus($status, $limit);

		foreach ($invoicesFromQueue as $queueObject) {
			try {
				$invoiceRecordModel = \Vtiger_Record_Model::getInstanceById($queueObject->getInvoiceRecordId());
				$invoice = (new InvoiceService($invoiceRecordModel))->makeInvoice();
				$sendResponse = $connectorService->send($invoice, $openResponse->object()->referenceNumber);
				$queueObject
					->setStatus(Status::IN_PROGRESS->value)
					->setSessionKey($openResponse->object()->referenceNumber)
					->setInvoiceKey($sendResponse->object()->referenceNumber);
				$queueRepository->update($queueObject);
			} catch (\Throwable $t) {
				KSeF::getLogger()->error($t->getMessage(), ['e' => $t, 'params' => ['record' => $queueObject?->getInvoiceRecordId(), 'configId' => $settingId]]);
				$queueObject
					->setStatus(Status::ERROR->value)
					->setSessionKey($openResponse->object()->referenceNumber);
				$queueRepository->update($queueObject);
			}
		}

		if (isset($openResponse)) {
			$connectorService->closeSession($openResponse->object()->referenceNumber);
		}
	}

	/**
	 * Verify and update invoices status.
	 *
	 * @param int      $settingId
	 * @param int|null $limit
	 *
	 * @throws KSeFConfigurationException
	 *
	 * @return void
	 */
	public function updateInvoicesStatus(int $settingId, ?int $limit = self::DEFAULT_MAX_INVOICES_PER_SESSION): void
	{
		$connectorService = new ConnectorService();
		$queueRepository = new QueueRepository();

		$connectorService->createConnection($settingId, EncryptionKeyFactory::makeRandom());
		$invoicesFromQueue = $queueRepository->getByStatus(Status::IN_PROGRESS, $limit);

		foreach ($invoicesFromQueue as $queueObject) {
			try {
				$statusResponse = $connectorService->status(
					$queueObject->getSessionKey(),
					$queueObject->getInvoiceKey(),
				);

				if (200 === $statusResponse->object()->status->code) {
					$queueObject
						->setStatus(Status::DONE->value)
						->setKsefNumber($statusResponse->object()->ksefNumber)
						->setDateAcquisition((new \DateTime($statusResponse->object()->acquisitionDate))->format('Y-m-d H:i:s'))
						->setMode(Mode::fromString($statusResponse->object()->invoicingMode));
				} else {
					KSeF::getLogger()->error(
						'Invoice processing did not complete successfully.',
						[
							'params' => [
								'record' => $queueObject?->getInvoiceRecordId(),
								'configId' => $settingId,
								'responseCode' => $statusResponse->object()->status->code,
								'responseDesc' => $statusResponse->object()->status->description,
							],
						]
					);
					$queueObject->setStatus(Status::REJECTED->value);
				}
				$queueRepository->update($queueObject);
			} catch (\Throwable $t) {
				KSeF::getLogger()->error($t->getMessage(), ['e' => $t, 'params' => ['record' => $queueObject?->getInvoiceRecordId()]]);
				$queueObject->setStatus(Status::ERROR->value);
				$queueRepository->update($queueObject);
			}
		}
	}
}
