<?php

/**
 * KSeF Certificate Model class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Model\Certificate;

use App\Integrations\KSeF\Model\Certificate\Enum\CertificateType;
use App\Json;

final class Certificate
{
	public function __construct(
		public readonly string $content,
		public readonly CertificateType $type,
		/** @var string Hexadecimal serial number */
		public readonly string $serialNumber,
		public readonly ?\DateTimeImmutable $validFrom = null,
		public readonly ?\DateTimeImmutable $validTo = null,
	) {}

	/**
	 * Create Certificate object from file.
	 *
	 * @param string          $path
	 * @param CertificateType $type
	 */
	public static function fromFile(string $path, CertificateType $type = CertificateType::Authentication): self
	{
		if (!is_file($path) || !is_readable($path)) {
			throw new \RuntimeException("Certificate file not readable: {$path}");
		}

		$content = file_get_contents($path);
		if (false === $content) {
			throw new \RuntimeException("Unable to read certificate file: {$path}");
		}

		$res = @openssl_x509_read($content);
		$info = $res ? @openssl_x509_parse($res, false) : null;

		$validFrom = isset($info['validFrom_time_t']) ? new \DateTimeImmutable('@' . $info['validFrom_time_t']) : null;
		$validTo = isset($info['validTo_time_t']) ? new \DateTimeImmutable('@' . $info['validTo_time_t']) : null;

		return new self(
			$content,
			$type,
			$info['serialNumberHex'] ?? '',
			$validFrom,
			$validTo,
		);
	}

	public static function fromField(?string $certJson, CertificateType $type = CertificateType::Authentication): self
	{
		$path = $certJson ? Json::decode($certJson)[0]['path'] ?? '' : '';

		return self::fromFile($path, $type);
	}

	public function getContent(): string
	{
		return $this->content;
	}

	/**
	 * @return \DateTimeImmutable|null
	 */
	public function getValidFrom(): ?\DateTimeImmutable
	{
		return $this->validFrom;
	}

	public function getValidTo(): ?\DateTimeImmutable
	{
		return $this->validTo;
	}

	public function getType(): CertificateType
	{
		return $this->type;
	}

	public function getValidFromDate(): string
	{
		return $this->validFrom->format('Y-m-d H:i:s');
	}

	public function getSerialNumber(): string
	{
		return $this->serialNumber;
	}

	public function getValidToDate(): string
	{
		return $this->validTo->format('Y-m-d H:i:s');
	}
}
