<?php

/**
 * KSeF Setting Model class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Model\Connector;

use App\Integrations\KSeF\Model\Certificate\Certificate;
use App\Json;

/**
 * KSeF Setting Model class.
 */
final class Setting
{
	/** @var string API URL */
	private string $apiUrl = 'https://api.ksef.mf.gov.pl/v2';

	/**
	 * Constructor with promoted properties.
	 *
	 * @param int         $recordId
	 * @param string      $identifier
	 * @param Certificate $certificate
	 * @param string      $passphrase
	 * @param string|null $privateKey
	 * @param int|null    $companyId
	 */
	public function __construct(
		private readonly int $recordId,
		private readonly string $identifier,
		private readonly Certificate $certificate,
		#[\SensitiveParameter]
		private readonly string $passphrase,
		#[\SensitiveParameter]
		private readonly ?string $privateKey = null,
		private readonly ?int $companyId = 0,
	) {}

	public static function fromData(#[\SensitiveParameter] array $data): self
	{
		$privateKey = $data['key'] ?? null;
		return new self(
			recordId: $data['id'] ?? 0,
			identifier: $data['identifier'],
			certificate: Certificate::fromField($data['cert']),
			passphrase: $data['passphrase'],
			privateKey: Json::isEmpty($privateKey) ? null : file_get_contents(Json::decode($privateKey)[0]['path'] ?? ''),
			companyId: (int) $data['company'] ?? 0,
		);
	}

	/**
	 * Get record ID.
	 *
	 * @return int
	 */
	public function getId(): int
	{
		return $this->recordId;
	}

	/**
	 * Get certificate.
	 *
	 * @return Certificate
	 */
	public function getCertificate(): Certificate
	{
		return $this->certificate;
	}

	/**
	 * Get private key.
	 *
	 * @return string|null
	 */
	public function getPrivateKey(): ?string
	{
		return $this->privateKey;
	}

	/**
	 * Get passphrase.
	 *
	 * @return string
	 */
	public function getPassphrase(): string
	{
		return $this->passphrase;
	}

	/**
	 * Get API URL.
	 *
	 * @return string
	 */
	public function getApiUrl(): string
	{
		return $this->apiUrl;
	}

	/**
	 * Set API URL.
	 *
	 * @param string $apiUrl
	 *
	 * @return self
	 */
	public function setApiUrl(string $apiUrl): self
	{
		$this->apiUrl = $apiUrl;
		return $this;
	}

	/**
	 * Get identifier.
	 *
	 * @return Identifier|null
	 */
	public function getIdentifier(): ?string
	{
		return $this->identifier;
	}

	public function getCompanyId(): int
	{
		return $this->companyId;
	}
}
