<?php
/**
 * KSeF Abstract Mapper class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Model\Mapper;

/**
 * KSeF Abstract Mapper class.
 */
abstract class AbstractMapper
{
	/** @var string Database table name */
	public const TABLENAME = null;

	/** @var string YetiForce module name */
	public const MODULE = null;

	/** @var Pair[] Field mapping pairs */
	protected array $pairs = [];

	/** @var \Vtiger_Record_Model Record being mapped */
	protected \Vtiger_Record_Model $record;

	/**
	 * Add a pair to the mapper.
	 *
	 * @param Pair $pair
	 *
	 * @return self
	 */
	public function addPair(Pair $pair): self
	{
		$this->pairs[$pair->getKey()] = $pair;

		return $this;
	}

	/**
	 * Get pair by key.
	 *
	 * @param string $key
	 *
	 * @return Pair|null
	 */
	public function get(string $key): ?Pair
	{
		return $this->pairs[$key] ?? null;
	}

	/**
	 * Get all pairs.
	 *
	 * @return Pair[]
	 */
	public function getAll(): array
	{
		return $this->pairs;
	}

	/**
	 * Get table name.
	 *
	 * @return string|null
	 */
	public function getTableName(): ?string
	{
		return static::TABLENAME;
	}

	/**
	 * Get module name.
	 *
	 * @return string|null
	 */
	public function getModule(): ?string
	{
		return static::MODULE;
	}

	/**
	 * Set record model.
	 *
	 * @param \Vtiger_Record_Model $record
	 *
	 * @return self
	 */
	public function setRecord(\Vtiger_Record_Model $record): self
	{
		$this->record = $record;
		return $this;
	}

	/**
	 * Get record model.
	 *
	 * @return \Vtiger_Record_Model
	 */
	public function getRecord(): \Vtiger_Record_Model
	{
		return $this->record;
	}
}
