<?php
/**
 * KSeF Mapped Field Pair class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Model\Mapper;

/**
 * KSeF Mapped Field Pair class.
 */
final class Pair
{
	/** @var string|null Unique identifier key for the mapping pair */
	private ?string $key = null;

	/** @var string|null KSeF field name in the XML structure */
	private ?string $ksefField = null;

	/** @var string|null Human-readable name/label for the mapping */
	private ?string $name = null;

	/** @var string|null YetiForce inventory field name */
	private ?string $inventoryField = null;

	/** @var string|null YetiForce field name */
	private ?string $yfField = null;

	/** @var string|null YetiForce field id */
	private ?string $field = null;

	/** * @var bool Tracks if any property has been modified */
	private bool $isChanged = false;

	/**
	 * Set mapping key.
	 *
	 * @param string|null $key
	 *
	 * @return self
	 */
	public function setKey(?string $key): self
	{
		$this->markChanged($this->key, $key);
		$this->key = $key;

		return $this;
	}

	/**
	 * Get mapping key.
	 *
	 * @return string|null
	 */
	public function getKey(): ?string
	{
		return $this->key;
	}

	/**
	 * Set KSeF field name.
	 *
	 * @param string|null $ksefField
	 *
	 * @return self
	 */
	public function setKsefField(?string $ksefField): self
	{
		$this->markChanged($this->ksefField, $ksefField);
		$this->ksefField = $ksefField;

		return $this;
	}

	/**
	 * Get KSeF field name.
	 *
	 * @return string|null
	 */
	public function getKsefField(): ?string
	{
		return $this->ksefField;
	}

	/**
	 * Set mapping name.
	 *
	 * @param string|null $name
	 *
	 * @return self
	 */
	public function setName(?string $name): self
	{
		$this->markChanged($this->name, $name);
		$this->name = $name;

		return $this;
	}

	/**
	 * Get mapping name.
	 *
	 * @return string|null
	 */
	public function getName(): ?string
	{
		return $this->name;
	}

	/**
	 * Set YetiForce inventory field name.
	 *
	 * @param string|null $inventoryField
	 *
	 * @return self
	 */
	public function setInventoryField(?string $inventoryField): self
	{
		$this->markChanged($this->inventoryField, $inventoryField);
		$this->inventoryField = $inventoryField;

		return $this;
	}

	/**
	 * Get YetiForce inventory field name.
	 *
	 * @return string|null
	 */
	public function getInventoryField(): ?string
	{
		return $this->inventoryField;
	}

	/**
	 * Get YetiForce field name.
	 *
	 * @return string
	 */
	public function getField(): string
	{
		return $this->field ?? '';
	}

	/**
	 * Set YetiForce field id.
	 *
	 * @param string|null $field
	 * @return self
	 */
	public function setField(?string $field): self
	{
		$this->markChanged($this->field, $field);
		$this->field = $field;

		return $this;
	}

	/**
	 * Check if any property has been modified.
	 *
	 * @return bool
	 */
	public function isChanged(): bool
	{
		return $this->isChanged;
	}

	/**
	 * Mark pair as changed if value differs.
	 *
	 * @param mixed $oldValue
	 * @param mixed $newValue
	 *
	 * @return void
	 */
	private function markChanged(mixed $oldValue, mixed $newValue): void
	{
		if ($this->isChanged || $oldValue !== $newValue) {
			$this->isChanged = true;
		}
	}
}
