<?php

/**
 * KSeF Certificate repository class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Repository;

use App\Db\Query;
use App\Integrations\KSeF\KSeF;
use App\Integrations\KSeF\Model\Certificate\Enum\CertificateStatus;
use App\Integrations\KSeF\Model\Connector\Setting;

/**
 * KSeF Certificate repository class.
 */
final class CertificateRepository
{
	/**
	 * Retrieve ids of active certificates.
	 *
	 * @return int[]
	 */
	public function findActiveSettingIds(): array
	{
		$now = (new \DateTime())->format('Y-m-d H:i:s');

		$query = new Query();
		$query
			->from(KSeF::SETTINGS_TABLE_NAME)
			->limit(1000)
			->select(['id'])
			->where(['status' => CertificateStatus::Active->getLabel()])
			->andWhere(['<=', 'date_valid_from', $now])
			->andWhere(['>=', 'date_valid_to', $now]);

		return $query->column() ?: [];
	}

	/**
	 * Retrieve data for certificates by expiration dates.
	 *
	 * @param \DateTime[] $dates Array with datetime objects
	 *
	 * @return array
	 */
	public function findByExpirationDates(array $dates): array
	{
		$query = new Query();
		$query->from(KSeF::SETTINGS_TABLE_NAME);
		$query->limit(1000);

		foreach ($dates as $date) {
			$startOfDay = $date->format('Y-m-d') . ' 00:00:00';
			$endOfDay = $date->format('Y-m-d') . ' 23:59:59';
			$query->orWhere(['between', 'date_valid_to', $startOfDay, $endOfDay]);
		}

		return $query->all();
	}

	public function findByCompanyId(int $companyId): ?Setting
	{
		$now = (new \DateTime())->format('Y-m-d H:i:s');

		$query = new Query();
		$query->from(KSeF::SETTINGS_TABLE_NAME)
			->where(['company' => $companyId, 'status' => CertificateStatus::Active->getLabel()])
			->andWhere(['<=', 'date_valid_from', $now])
			->andWhere(['>=', 'date_valid_to', $now]);

		$result = $query->one();

		return $result ? Setting::fromData($result) : null;
	}

	public function findById(int $id): ?Setting
	{
		$query = new Query();
		$result = $query->from(KSeF::SETTINGS_TABLE_NAME)
			->where(['id' => $id])
			->one();

		return $result ? Setting::fromData($result) : null;
	}
}
