<?php
/**
 * KSeF Download Queue repository class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Repository;

use App\Db\Query;
use App\Encryption;
use App\Integrations\KSeF\KSeF;
use App\Integrations\KSeF\Model\Client\DownloadQueue;
use App\Integrations\KSeF\Model\Enum\Status;

/**
 * KSeF Download Queue repository class.
 */
final class DownloadQueueRepository
{
	/**
	 * Load queue records by status.
	 *
	 * @param Status $status status of the queue records to load
	 * @param int    $limit  Number of records to load
	 *
	 * @return DownloadQueue[] array of DownloadQueue objects
	 */
	public function getByStatus(Status $status, int $limit): array
	{
		$queueObjects = [];

		$dataReader = (new Query())
			->from(KSeF::DOWNLOAD_REQUEST_TABLE_NAME)
			->where(['status' => $status->value])
			->limit($limit)
			->createCommand()
			->query();

		while ($row = $dataReader->read()) {
			$queueObjects[] = $this->rowToObject($row);
		}

		return $queueObjects;
	}

	/**
	 * Load queue records by reference number.
	 *
	 * @param string $referenceNumber reference number
	 * @param int    $limit  Number of records to load
	 *
	 * @return DownloadQueue[] array of DownloadQueue objects
	 */
	public function getByReferenceNumber(string $referenceNumber, int $limit): array
	{
		$queueObjects = [];

		$dataReader = (new Query())
			->from(KSeF::DOWNLOAD_REQUEST_TABLE_NAME)
			->where(['reference_number' => $referenceNumber])
			->limit($limit)
			->createCommand()
			->query();

		while ($row = $dataReader->read()) {
			$queueObjects[] = $this->rowToObject($row);
		}

		return $queueObjects;
	}

	/**
	 * Check if queue record exists by KSeF reference number.
	 *
	 * @param string $referenceNumber reference number
	 *
	 * @return bool
	 */
	public function exists(string $referenceNumber): bool
	{
		return (new Query())
			->from(KSeF::DOWNLOAD_REQUEST_TABLE_NAME)
			->where(['reference_number' => $referenceNumber])
			->exists();
	}

	/**
	 * Update queue record.
	 *
	 * @param DownloadQueue $queueObject Queue object to update
	 *
	 * @return void
	 */
	public function update(DownloadQueue $queueObject): void
	{
		$queueObject->setDateMod((new \DateTime())->format('Y-m-d H:i:s'));

		(new Query())
			->createCommand()
			->update(
				KSeF::DOWNLOAD_REQUEST_TABLE_NAME,
				$this->objectToRow($queueObject),
				['id' => $queueObject->getId()]
			)->execute();
	}

	/**
	 * Add new queue record.
	 *
	 * @param DownloadQueue $queueObject Queue object to add
	 *
	 * @return void
	 */
	public function new(DownloadQueue $queueObject): void
	{
		$queueObject->setDateInput((new \DateTime())->format('Y-m-d H:i:s'));

		(new Query())
			->createCommand()
			->insert(
				KSeF::DOWNLOAD_REQUEST_TABLE_NAME,
				$this->objectToRow($queueObject),
			)->execute();
	}

	/**
	 * Transform database row to DownloadQueue object.
	 *
	 * @param array $row Database row
	 *
	 * @return DownloadQueue
	 */
	private function rowToObject(array $row): DownloadQueue
	{
		$encryption = Encryption::getInstance();
		return (new DownloadQueue())
			->setId($row['id'])
			->setStatus((int) $row['status'])
			->setSettingsId($row['settings_id'])
			->setReferenceNumber($row['reference_number'])
			->setDateInput($row['date_input'])
			->setDateMod($row['date_mod'])
			->setModule($row['module'])
			->setKey(!empty($row['key']) ? base64_decode($encryption->decrypt($row['key'])) : null)
			->setIv(!empty($row['iv']) ? base64_decode($encryption->decrypt($row['iv'])) : null);
	}

	/**
	 * Transform DownloadQueue object to database row.
	 *
	 * @param DownloadQueue $object DownloadQueue object
	 *
	 * @return array $row Database row
	 */
	private function objectToRow(DownloadQueue $object): array
	{
		$encryption = Encryption::getInstance();
		return [
			'status' => $object->getStatus(),
			'settings_id' => $object->getSettingsId(),
			'reference_number' => $object->getReferenceNumber(),
			'date_input' => $object->getDateInput(),
			'date_mod' => date('Y-m-d H:i:s'),
			'module' => $object->getModule(),
			'key' => $object->getKey() ? $encryption->encrypt(base64_encode($object->getKey())) : null,
			'iv' => $object->getIv() ? $encryption->encrypt(base64_encode($object->getIv())) : null,
		];
	}
}
