<?php

/**
 * KSeF Queue repository class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Repository;

use App\Db\Query;
use App\EventHandler;
use App\Integrations\KSeF\KSeF;
use App\Integrations\KSeF\Model\Client\InvoiceQueue;
use App\Integrations\KSeF\Model\Enum\Status;

/**
 * KSeF Queue repository class.
 */
final class QueueRepository
{
	/**
	 * Load queue records by status.
	 *
	 * @param Status $status status of the queue records to load
	 * @param int    $limit  Number of records to load
	 *
	 * @return InvoiceQueue[] array of InvoiceQueue objects
	 */
	public function getByStatus(Status $status, int $limit): array
	{
		$queueObjects = [];

		$dataReader = (new Query())
			->from(KSeF::QUEUE_TABLE_NAME)
			->where(['status' => $status->value])
			->limit($limit)
			->createCommand()
			->query();

		while ($row = $dataReader->read()) {
			$queueObjects[] = $this->rowToObject($row);
		}

		return $queueObjects;
	}

	/**
	 * Check if queue record exists by invoice record id.
	 *
	 * @param int $recordId invoice record id
	 *
	 * @return bool
	 */
	public function exists(int $recordId): bool
	{
		return (new Query())
			->from(KSeF::QUEUE_TABLE_NAME)
			->where(['invoice_record_id' => $recordId])
			->exists();
	}

	public function getLastStatus(int $recordId): ?int
	{
		$result = (new Query())
			->from(KSeF::QUEUE_TABLE_NAME)
			->select(['status'])
			->where(['invoice_record_id' => $recordId])
			->orderBy(['date_mod' => SORT_DESC])
			->scalar();

		return is_numeric($result) ? $result : null;
	}

	/**
	 * Update queue record.
	 *
	 * @param InvoiceQueue $queueObject Queue object to update
	 *
	 * @return void
	 */
	public function update(InvoiceQueue $queueObject): void
	{
		$queueObject->setDateMod((new \DateTime())->format('Y-m-d H:i:s'));

		(new Query())
			->createCommand()
			->update(
				KSeF::QUEUE_TABLE_NAME,
				$this->objectToRow($queueObject),
				['id' => $queueObject->getId()]
			)->execute();

		$this->triggerAfterUpdateEvent($queueObject);
	}

	/**
	 * Add new queue record.
	 *
	 * @param InvoiceQueue $queueObject Queue object to add
	 *
	 * @return bool
	 */
	public function new(InvoiceQueue $queueObject): bool
	{
		$queueObject->setDateInput((new \DateTime())->format('Y-m-d H:i:s'));

		$result = (new Query())
			->createCommand()
			->insert(
				KSeF::QUEUE_TABLE_NAME,
				$this->objectToRow($queueObject),
			)->execute();

		$this->triggerAfterUpdateEvent($queueObject);

		return (bool) $result;
	}

	/**
	 * Find setting IDs in queue by status.
	 *
	 * @param Status $status Status to filter
	 *
	 * @return int[]
	 */
	public function findSettingIds(Status $status): array
	{
		return (new Query())
			->from(KSeF::QUEUE_TABLE_NAME)
			->select(['settings_id'])
			->distinct()
			->where(['status' => $status->value])
			->column();
	}

	/**
	 * Find session keys in queue by status and setting ID.
	 *
	 * @param Status $status    Status to filter
	 * @param int    $settingId
	 *
	 * @return array
	 */
	public function findSessionKeys(Status $status, int $settingId): array
	{
		return (new Query())
			->from(KSeF::QUEUE_TABLE_NAME)
			->select(['session_key'])
			->distinct()
			->where(['status' => $status->value, 'settings_id' => $settingId])
			->column();
	}

	/**
	 * Find session keys in queue by status and setting ID.
	 *
	 * @param string $sessionKey    Session key to filter
	 * @param int    $ordinalNumber Ordinal number to filter
	 *
	 * @return InvoiceQueue|null
	 */
	public function findOneBySession(string $sessionKey, int $ordinalNumber): ?InvoiceQueue
	{
		$data = (new Query())
			->from(KSeF::QUEUE_TABLE_NAME)
			->where(['session_key' => $sessionKey, 'ordinal_number' => $ordinalNumber])
			->limit(1)
			->one();

		return empty($data) ? null : $this->rowToObject($data);
	}

	/**
	 * Trigger after update event.
	 *
	 * @param InvoiceQueue $queueObject Queue object
	 *
	 * @return void
	 */
	private function triggerAfterUpdateEvent(InvoiceQueue $queueObject): void
	{
		$eventHandler = new EventHandler();
		$eventHandler->setParams(['queueObject' => $queueObject]);
		$eventHandler->trigger('InvoiceQueueAfterUpdate');
	}

	/**
	 * Transform database row to InvoiceQueue object.
	 *
	 * @param array $row Database row
	 *
	 * @return InvoiceQueue
	 */
	private function rowToObject(array $row): InvoiceQueue
	{
		return (new InvoiceQueue())
			->setId($row['id'])
			->setInvoiceRecordId($row['invoice_record_id'])
			->setStatus($row['status'])
			->setSettingsId($row['settings_id'])
			->setMode($row['mode'])
			->setSessionKey($row['session_key'])
			->setInvoiceKey($row['invoice_key'])
			->setKsefNumber($row['ksef_number'])
			->setDateInput($row['date_input'])
			->setDateMod($row['date_mod'])
			->setDateAcquisition($row['date_acquisition'])
			->setOrdinalNumber($row['ordinal_number']);
	}

	/**
	 * Transform InvoiceQueue object to database row.
	 *
	 * @param InvoiceQueue $object InvoiceQueue object
	 *
	 * @return array $row Database row
	 */
	private function objectToRow(InvoiceQueue $object): array
	{
		return [
			'invoice_record_id' => $object->getInvoiceRecordId(),
			'status' => $object->getStatus(),
			'settings_id' => $object->getSettingsId(),
			'mode' => $object->getMode(),
			'session_key' => $object->getSessionKey(),
			'invoice_key' => $object->getInvoiceKey(),
			'ksef_number' => $object->getKsefNumber(),
			'date_acquisition' => $object->getDateAcquisition(),
			'date_input' => $object->getDateInput(),
			'ordinal_number' => $object->getOrdinalNumber(),
			'date_mod' => date('Y-m-d H:i:s'),
		];
	}
}
