<?php

/**
 * KSeF Invoice Abstract Builder class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Service\Builder;

use App\Integrations\KSeF\Exception\KSeFInvoiceBuilderException;
use App\Integrations\KSeF\Service\Builder\Rules\RuleInterface;
use App\Integrations\KSeF\Service\Mapper\MapperService;
use N1ebieski\KSEFClient\Exceptions\RuleValidationException;
use N1ebieski\KSEFClient\Support\AbstractDTO;

/**
 * KSeF Invoice Abstract Builder class.
 */
abstract class AbstractBuilder
{
	/** @var MapperService Mapper service for field mapping operations */
	protected MapperService $mapperService;

	/** @var RuleInterface[] Array of rules to apply during building */
	protected array $rules = [];

	/** @var array Data array to hold mapped values */
	protected array $data = [];

	/**
	 * Constructor.
	 *
	 * @param MapperService $mapperService
	 */
	public function __construct(MapperService $mapperService)
	{
		$this->mapperService = $mapperService;
	}

	/**
	 * Add a rule to the builder.
	 *
	 * @param RuleInterface $rule The rule to add.
	 *
	 * @return self
	 */
	public function addRule(RuleInterface $rule): self
	{
		$this->rules[] = $rule;
		return $this;
	}

	/**
	 * Fill data from the record using the defined rules.
	 *
	 * @return self
	 */
	public function fillFromRecord(): self
	{
		foreach ($this->rules as $rule) {
			$rule->apply($this->data, $this->mapperService);
		}

		return $this;
	}

	/**
	 * Build and return the N1ebieski\KSEFClient\Support\AbstractDTO.
	 *
	 * @return AbstractDTO
	 */
	abstract public function build(): AbstractDTO;

	public function buildValue(string $className, string $key, mixed $optional = null)
	{
		try {
			if (null !== $optional && ($this->data[$key] ?? '') === '') {
				return $optional;
			}

			$value = $this->data[$key] ?? null;
			return new $className($value);
		} catch (\InvalidArgumentException|RuleValidationException $e) {
			$fieldName = $this->mapperService->getMapper()->get($key)->getField();
			$recordId = $this->mapperService->getRecord()->getId();
			throw new KSeFInvoiceBuilderException(\sprintf('%s|"%s" => %s|record => %s', $e->getMessage(), $fieldName, $key, $recordId));
		}
	}

	/**
	 * Helper method to check if any of the specified keys have non-empty values in the data array.
	 *
	 * @param array $keys The keys to check.
	 *
	 * @return bool True if any key has a non-empty value, false otherwise.
	 */
	protected function hasAnyValue(array $keys): bool
	{
		foreach ($keys as $key) {
			if (!empty($this->data[$key])) {
				return true;
			}
		}
		return false;
	}
}
