<?php
/**
 * KSeF Builder FaPartialAdvanceRule rule class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Service\Builder\Rules;

use App\Integrations\KSeF\Model\Enum\Type;
use App\Integrations\KSeF\Service\Mapper\MapperService;
use App\QueryGenerator;
use N1ebieski\KSEFClient\DTOs\Requests\Sessions\ZaliczkaCzesciowa;
use N1ebieski\KSEFClient\Support\Optional;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\KursWalutyZW;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\P_15Z;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\P_6Z;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\RodzajFaktury;

/**
 * KSeF Builder FaPartialAdvanceRule rule class.
 */
class FaPartialAdvanceRule implements RuleInterface
{
	private ?MapperService $mapperService = null;

	/** {@inheritDoc} */
	public function apply(array &$target, MapperService $mapperService): void
	{
		$this->mapperService = $mapperService;

		$partialAdvanceNode = [];
		if (
			RodzajFaktury::Zal === $target['//Fa/RodzajFaktury']
			&& $this->isPartialAdvance($mapperService->getRecord())
		) {
			$partialAdvances = $this->getRelatedPayments($mapperService->getRecord()->getId());
			foreach ($partialAdvances as $partialAdvance) {
				$date = $this->mapperService->mapField('//Fa/Platnosc/DataZaplaty', Type::DATE, $partialAdvance);
				$value = $this->mapperService->mapField('//Fa/P_15Z', Type::FLOAT, $partialAdvance);
				$currencyRate = $this->mapperService->mapField('fa_kursWalutyZW', Type::FLOAT, $partialAdvance);

				$partialAdvanceNode[] = new ZaliczkaCzesciowa(
					p_6Z: new P_6Z($date),
					p_15Z: new P_15Z($value),
					kursWalutyZW: (!empty($currencyRate)) ? new KursWalutyZW($currencyRate) : new Optional(),
				);
			}
		}

		$target['fa_zaliczkaCzesciowa'] = $partialAdvanceNode;
	}

	/**
	 * @return bool
	 */
	private function isPartialAdvance(): bool
	{
		$result = false;

		$paymentSum = $this->mapperService->mapField('yf_payment_sum', Type::FLOAT);
		$totalValue = $this->mapperService->mapField('//Fa/P_15', Type::FLOAT);

		if ($paymentSum > 0 && $paymentSum < $totalValue) {
			$result = true;
		}

		return $result;
	}

	/**
	 * @param int $recordId
	 *
	 * @return \Vtiger_Record_Model[]
	 */
	private function getRelatedPayments(int $recordId): array
	{
		$query = new QueryGenerator('PaymentsIn');
		$query->permissions = false;
		$query->setFields(['paymentsinid']);
		$query->addCondition('finvoiceid', $recordId, 'e');
		$query->addCondition('paymentsin_status', 'PLL_PAID', 'e');

		$results = [];
		foreach ($query->createQuery()->all() as $item) {
			$results[] = \Vtiger_Record_Model::getInstanceById($item['paymentsinid']);
		}

		return $results;
	}
}
