<?php

/**
 * KSeF Builder FaRowRule rule class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Service\Builder\Rules;

use App\Integrations\KSeF\Model\Enum\Type;
use App\Integrations\KSeF\Service\Mapper\MapperService;
use N1ebieski\KSEFClient\DTOs\Requests\Sessions\FaWiersz;
use N1ebieski\KSEFClient\DTOs\Requests\Sessions\Zamowienie;
use N1ebieski\KSEFClient\DTOs\Requests\Sessions\ZamowienieWiersz;
use N1ebieski\KSEFClient\Support\Optional;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\KursWaluty;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\NrWierszaFa;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\NrWierszaZam;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\P_10;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\P_11;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\P_11A;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\P_11NettoZ;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\P_11Vat;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\P_11VatZ;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\P_12;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\P_12Z;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\P_7;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\P_7Z;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\P_8A;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\P_8AZ;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\P_8B;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\P_8BZ;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\P_9A;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\P_9AZ;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\RodzajFaktury;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\StanPrzed;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\WartoscZamowienia;

/**
 * KSeF Builder FaRowRule rule class.
 */
class FaRowRule implements RuleInterface
{
	private ?MapperService $mapperService = null;

	/** {@inheritDoc} */
	public function apply(array &$target, MapperService $mapperService): void
	{
		$this->mapperService = $mapperService;
		$recordModel = $mapperService->getRecord();

		$isAdvanceInvoice = RodzajFaktury::Zal === $target['//Fa/RodzajFaktury'];

		if ($isAdvanceInvoice) {
			$this->buildAdvanceInvoiceOrder($target, $recordModel);
		} else {
			$this->buildStandardInvoiceLines($target, $recordModel);
		}
	}

	/**
	 * Build standard invoice lines (FaWiersz) from inventory data.
	 *
	 * @param array                &$target
	 * @param \Vtiger_Record_Model $recordModel
	 *
	 * @return void
	 */
	private function buildStandardInvoiceLines(array &$target, \Vtiger_Record_Model $recordModel): void
	{
		$rowNode = [];

		if ('FCorectingInvoice' === $recordModel->getModuleName()) {
			$originalInvoiceRecord = \Vtiger_Record_Model::getInstanceById(
				$this->mapperService->mapField('yf_finvoiceid', Type::STRING)
			);

			$i = 1;
			foreach ($originalInvoiceRecord->getInventoryData() as $inventory) {
				$name = $this->mapperService->mapField('//Fa/FaWiersz/P_7', Type::STRING, $inventory);
				$unit = $this->mapperService->mapField('//Fa/FaWiersz/P_8A', Type::STRING, $inventory);
				$qty = $this->mapperService->mapField('//Fa/FaWiersz/P_8B', Type::FLOAT, $inventory);
				$price = $this->mapperService->mapField('//Fa/FaWiersz/P_9A', Type::FLOAT, $inventory);
				$discount = $this->mapperService->mapField('//Fa/FaWiersz/P_10', Type::FLOAT, $inventory);
				if (str_contains($inventory['discountparam'] ?? '', 'markup')) {
					$price = round($price + $discount);
					$discount = 0.0;
				}

				$net = $this->mapperService->mapField('//Fa/FaWiersz/P_11', Type::FLOAT, $inventory);
				$gross = $this->mapperService->mapField('//Fa/FaWiersz/P_11A', Type::FLOAT, $inventory);
				$vat = $this->mapperService->mapField('//Fa/FaWiersz/P_11Vat', Type::FLOAT, $inventory);
				$vatRate = $this->calcVatRate($inventory, $this->mapperService->getRecord());
				$currencyRate = $this->mapperService->mapField('//Fa/FaWiersz/KursWaluty', Type::FLOAT, $inventory);

				$rowNode[] = new FaWiersz(
					nrWierszaFa: new NrWierszaFa($i++),
					uu_id: new Optional(), /* Not implemented yet */
					p_6A: new Optional(), /* Not implemented yet */
					p_7: !(empty($name)) ? new P_7($name) : new Optional(),
					indeks: new Optional(), /* Not implemented yet */
					gtin: new Optional(), /* Not implemented yet */
					pkwiu: new Optional(), /* Not implemented yet */
					cn: new Optional(), /* Not implemented yet */
					pkob: new Optional(), /* Not implemented yet */
					p_8A: !(empty($unit)) ? new P_8A($unit) : new Optional(),
					p_8B: !(empty($qty)) ? new P_8B($qty) : new Optional(),
					p_9A: !(empty($price)) ? new P_9A($price) : new Optional(),
					p_9B: new Optional(), /* Not implemented yet */
					p_10: $discount ? new P_10(round($discount, 2)) : new Optional(),
					p_11: !(empty($net)) ? new P_11($net) : new Optional(),
					p_11A: !(empty($gross)) ? new P_11A($gross) : new Optional(),
					p_11Vat: !(empty($vat)) ? new P_11Vat($vat) : new Optional(),
					p_12: $vatRate,
					p_12_XII: new Optional(), /* Not implemented yet */
					p_12_Zal_15: new Optional(), /* Not implemented yet */
					kwotaAkcyzy: new Optional(), /* Not implemented yet */
					gtu: new Optional(), /* Not implemented yet */
					procedura: new Optional(), /* Not implemented yet */
					kursWaluty: $currencyRate ? new KursWaluty($currencyRate) : new Optional(),
					stanPrzed: StanPrzed::Default,
				);
			}
		}

		$i = 1;
		foreach ($recordModel->getInventoryData() as $inventory) {
			$name = $this->mapperService->mapField('//Fa/FaWiersz/P_7', Type::STRING, $inventory);
			$unit = $this->mapperService->mapField('//Fa/FaWiersz/P_8A', Type::STRING, $inventory);
			$qty = $this->mapperService->mapField('//Fa/FaWiersz/P_8B', Type::FLOAT, $inventory);
			$price = $this->mapperService->mapField('//Fa/FaWiersz/P_9A', Type::FLOAT, $inventory);
			$discount = $this->mapperService->mapField('//Fa/FaWiersz/P_10', Type::FLOAT, $inventory);
			if (str_contains($inventory['discountparam'] ?? '', 'markup')) {
				$price = round($price + $discount);
				$discount = 0.0;
			}
			$net = $this->mapperService->mapField('//Fa/FaWiersz/P_11', Type::FLOAT, $inventory);
			$gross = $this->mapperService->mapField('//Fa/FaWiersz/P_11A', Type::FLOAT, $inventory);
			$vat = $this->mapperService->mapField('//Fa/FaWiersz/P_11Vat', Type::FLOAT, $inventory);
			$vatRate = $this->calcVatRate($inventory, $this->mapperService->getRecord());
			$currencyRate = $this->mapperService->mapField('//Fa/FaWiersz/KursWaluty', Type::FLOAT, $inventory);

			$rowNode[] = new FaWiersz(
				nrWierszaFa: new NrWierszaFa($i++),
				uu_id: new Optional(), /* Not implemented yet */
				p_6A: new Optional(), /* Not implemented yet */
				p_7: !(empty($name)) ? new P_7($name) : new Optional(),
				indeks: new Optional(), /* Not implemented yet */
				gtin: new Optional(), /* Not implemented yet */
				pkwiu: new Optional(), /* Not implemented yet */
				cn: new Optional(), /* Not implemented yet */
				pkob: new Optional(), /* Not implemented yet */
				p_8A: !(empty($unit)) ? new P_8A($unit) : new Optional(),
				p_8B: !(empty($qty)) ? new P_8B($qty) : new Optional(),
				p_9A: !(empty($price)) ? new P_9A($price) : new Optional(),
				p_9B: new Optional(),
				p_10: $discount ? new P_10(round($discount, 2)) : new Optional(),
				p_11: !(empty($net)) ? new P_11($net) : new Optional(),
				p_11A: !(empty($gross)) ? new P_11A($gross) : new Optional(),
				p_11Vat: !(empty($vat)) ? new P_11Vat($vat) : new Optional(),
				p_12: $vatRate,
				p_12_XII: new Optional(), /* Not implemented yet */
				p_12_Zal_15: new Optional(), /* Not implemented yet */
				kwotaAkcyzy: new Optional(), /* Not implemented yet */
				gtu: new Optional(), /* Not implemented yet */
				procedura: new Optional(), /* Not implemented yet */
				kursWaluty: $currencyRate ? new KursWaluty($currencyRate) : new Optional(),
				stanPrzed: new Optional(),
			);
		}

		$target['faWiersz'] = $rowNode;
	}

	/**
	 * Build advance invoice order (Zamowienie with ZamowienieWiersz) from inventory data.
	 *
	 * @param array                &$target
	 * @param \Vtiger_Record_Model $recordModel
	 *
	 * @return void
	 */
	private function buildAdvanceInvoiceOrder(array &$target, \Vtiger_Record_Model $recordModel): void
	{
		$orderItems = [];
		$i = 1;

		foreach ($recordModel->getInventoryData() as $inventory) {
			$name = $this->mapperService->mapField('//Fa/Zamowienie/ZamowienieWiersz/P_7Z', Type::STRING, $inventory);
			$unit = $this->mapperService->mapField('//Fa/Zamowienie/ZamowienieWiersz/P_8AZ', Type::STRING, $inventory);
			$qty = $this->mapperService->mapField('//Fa/Zamowienie/ZamowienieWiersz/P_8BZ', Type::FLOAT, $inventory);
			$price = $this->mapperService->mapField('//Fa/Zamowienie/ZamowienieWiersz/P_9AZ', Type::FLOAT, $inventory);
			$net = $this->mapperService->mapField('//Fa/Zamowienie/ZamowienieWiersz/P_11NettoZ', Type::FLOAT, $inventory);
			$vat = $this->mapperService->mapField('//Fa/Zamowienie/ZamowienieWiersz/P_11VatZ', Type::FLOAT, $inventory);
			$vatRate = $this->calcAdvanceVatRate($inventory, $recordModel);

			$orderItems[] = new ZamowienieWiersz(
				nrWierszaZam: new NrWierszaZam($i++),
				uu_idZ: new Optional(),
				p_7Z: !(empty($name)) ? new P_7Z($name) : new Optional(),
				indeksZ: new Optional(),
				gtinZ: new Optional(),
				pkwiuZ: new Optional(),
				cnZ: new Optional(),
				pkobZ: new Optional(),
				p_8AZ: !(empty($unit)) ? new P_8AZ($unit) : new Optional(),
				p_8BZ: !(empty($qty)) ? new P_8BZ($qty) : new Optional(),
				p_9AZ: !(empty($price)) ? new P_9AZ($price) : new Optional(),
				p_11NettoZ: !(empty($net)) ? new P_11NettoZ($net) : new Optional(),
				p_11VatZ: !(empty($vat)) ? new P_11VatZ($vat) : new Optional(),
				p_12Z: $vatRate,
				p_12Z_XII: new Optional(),
				p_12Z_Zal_15: new Optional(),
				gtuZ: new Optional(),
				proceduraZ: new Optional(),
				kwotaAkcyzyZ: new Optional(),
				stanPrzedZ: new Optional(),
			);
		}

		$orderValue = $this->mapperService->mapField('//Fa/Zamowienie/WartoscZamowienia', Type::FLOAT);
		$orderNode = new Zamowienie(
			wartoscZamowienia: new WartoscZamowienia($orderValue),
			zamowienieWiersz: $orderItems,
		);

		$target['fa_zamowienie'] = $orderNode;
	}

	private function calcDiscount(array $inventory): P_10
	{
		$value = 0.00;

		$total = $this->mapperService->mapField('yf_total', Type::FLOAT, $inventory);
		$net = $this->mapperService->mapField('//Fa/FaWiersz/P_11', Type::FLOAT, $inventory);

		if ($total !== $net) {
			$value = $total - $net;
		}

		return new P_10(round($value, 2));
	}

	private function calcVatRate(array $inventory, \Vtiger_Record_Model $recordModel): P_12
	{
		$globalTaxes = \Vtiger_Inventory_Model::getGlobalTaxes();

		$taxPercent = $this->mapperService->mapField('//Fa/FaWiersz/P_12', Type::INTEGER, $inventory);
		$vatRate = match ($taxPercent) {
			23 => P_12::Tax23,
			22 => P_12::Tax22,
			8 => P_12::Tax8,
			7 => P_12::Tax7,
			5 => P_12::Tax5,
			4 => P_12::Tax4,
			3 => P_12::Tax3,
			default => null,
		};

		if (null !== $taxPercent && '' !== $taxPercent && null === $vatRate) {
			$taxParam = $this->mapperService->mapField('yf_taxparam', Type::STRING, $inventory);
			if ($recordModel->get('reverse_charge')) {
				$vatRate = P_12::Oo;
			} elseif (0 === $taxPercent) {
				$taxParam = json_decode($taxParam, true);
				$taxName = (isset($taxParam['globalId']))
					? $globalTaxes[$taxParam['globalId']]['name']
					: '';

				$invoiceType = $this->mapperService->mapField('yf_finvoice_type', Type::STRING);
				if ('ZW' === strtoupper($taxName)) {
					$vatRate = P_12::Zw;
				} elseif ('NP' === strtoupper($taxName)) {
					$vatRate = ('PLL_DOMESTIC_INVOICE' !== $invoiceType)
						? P_12::NpI
						: P_12::NpII;
				} else {
					$vatRate = match ($invoiceType) {
						'PLL_DOMESTIC_INVOICE' => P_12::Tax0Kr,
						'PLL_IC_INVOICE' => P_12::Tax0Wdt,
						'PLL_FOREIGN_INVOICE' => P_12::Tax0Ex,
						default => P_12::Tax0Kr,
					};
				}
			}
		}

		return $vatRate;
	}

	/**
	 * Calculate VAT rate for advance invoice order items.
	 *
	 * @param array                $inventory
	 * @param \Vtiger_Record_Model $recordModel
	 *
	 * @return P_12Z
	 */
	private function calcAdvanceVatRate(array $inventory, \Vtiger_Record_Model $recordModel): P_12Z
	{
		$globalTaxes = \Vtiger_Inventory_Model::getGlobalTaxes();

		$taxPercent = $this->mapperService->mapField('//Fa/Zamowienie/ZamowienieWiersz/P_12Z', Type::INTEGER, $inventory);
		$vatRate = match ($taxPercent) {
			23 => P_12Z::Tax23,
			22 => P_12Z::Tax22,
			8 => P_12Z::Tax8,
			7 => P_12Z::Tax7,
			5 => P_12Z::Tax5,
			4 => P_12Z::Tax4,
			3 => P_12Z::Tax3,
			default => null,
		};

		if (null !== $taxPercent && '' !== $taxPercent && null === $vatRate) {
			$taxParam = $this->mapperService->mapField('yf_taxparam', Type::STRING, $inventory);
			if ($recordModel->get('reverse_charge')) {
				$vatRate = P_12Z::Oo;
			} elseif (0 === $taxPercent) {
				$taxParam = json_decode($taxParam, true);
				$taxName = (isset($taxParam['globalId']))
					? $globalTaxes[$taxParam['globalId']]['name']
					: '';

				$invoiceType = $this->mapperService->mapField('yf_finvoice_type', Type::STRING);
				if ('ZW' === strtoupper($taxName)) {
					$vatRate = P_12Z::Zw;
				} elseif ('NP' === strtoupper($taxName)) {
					$vatRate = ('PLL_DOMESTIC_INVOICE' !== $invoiceType)
						? P_12Z::NpI
						: P_12Z::NpII;
				} else {
					$vatRate = match ($invoiceType) {
						'PLL_DOMESTIC_INVOICE' => P_12Z::Tax0Kr,
						'PLL_IC_INVOICE' => P_12Z::Tax0Wdt,
						'PLL_FOREIGN_INVOICE' => P_12Z::Tax0Ex,
						default => P_12Z::Tax0Kr,
					};
				}
			}
		}

		return $vatRate;
	}
}
