<?php

/**
 * KSeF Connector Service class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Service\Connector;

use App\Encryption;
use App\Integrations\KSeF\Exception\KSeFClientException;
use App\Integrations\KSeF\Exception\KSeFConfigurationException;
use App\Integrations\KSeF\Model\Certificate\Certificate;
use App\Integrations\KSeF\Model\Connector\Connection;
use App\Integrations\KSeF\Model\Connector\Setting;
use App\Integrations\KSeF\Model\Enum\ImportType;
use App\Integrations\KSeF\Repository\CertificateRepository;
use N1ebieski\KSEFClient\ClientBuilder;
use N1ebieski\KSEFClient\Contracts\HttpClient\ResponseInterface;
use N1ebieski\KSEFClient\DTOs\Requests\Sessions\Faktura;
use N1ebieski\KSEFClient\Requests\Certificates\Query\QueryRequest;
use N1ebieski\KSEFClient\Requests\Sessions\Invoices\List\ListRequest;
use N1ebieski\KSEFClient\Requests\Sessions\Online\Send\SendRequest;
use N1ebieski\KSEFClient\Resources\ClientResource;
use N1ebieski\KSEFClient\ValueObjects\CertificateSerialNumber;
use N1ebieski\KSEFClient\ValueObjects\EncryptionKey;
use N1ebieski\KSEFClient\ValueObjects\Requests\Certificates\CertificateType;
use N1ebieski\KSEFClient\ValueObjects\Requests\ReferenceNumber;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\FormCode;

/**
 * KSeF Connector Service class.
 */
final class ConnectorService
{
	/** @var Connection|null */
	private ?Connection $connection = null;

	/** @var EncryptionKey|null */
	private ?EncryptionKey $encryptionKey;

	/**
	 * Create a connection with loaded setting and built client.
	 *
	 * @param Setting|int   $setting       Setting object or setting record ID
	 * @param EncryptionKey $encryptionKey
	 *
	 * @throws KSeFConfigurationException
	 *
	 * @return self
	 */
	public function createConnection(int|Setting $setting, EncryptionKey $encryptionKey): self
	{
		$this->encryptionKey = $encryptionKey;

		$this->connection = new Connection();

		$setting = (\is_int($setting))
			? $this->loadSetting($setting)
			: $setting;

		$this->connection->setSetting($setting);
		$client = $this->buildClient($setting);
		$this->connection->setClient($client);

		return $this;
	}

	/**
	 * Get connection.
	 *
	 * @return Connection
	 */
	public function getConnection(): Connection
	{
		return $this->connection;
	}

	/**
	 * Load setting from database with decrypted sensitive data.
	 *
	 * @param int $recordId
	 *
	 * @throws KSeFConfigurationException
	 *
	 * @return Setting
	 */
	public function loadSetting(int $recordId): Setting
	{
		return (new CertificateRepository())->findById($recordId);
	}

	/**
	 * Check if connection is valid.
	 *
	 * @return bool
	 */
	public function isConnected(): bool
	{
		return $this->connection->isConnected();
	}

	/**
	 * Open session.
	 *
	 * @return ResponseInterface
	 */
	public function openSession(): ResponseInterface
	{
		$response = $this->connection->getClient()->sessions()->online()->open([
			'formCode' => FormCode::Fa3->value,
		]);

		if (201 !== $response?->status()) {
			throw new KSeFClientException($response->body());
		}

		return $response;
	}

	/**
	 * Close session.
	 *
	 * @param string $referenceNumber From open session response
	 *
	 * @return ResponseInterface
	 */
	public function closeSession(string $referenceNumber): ResponseInterface
	{
		$response = $this->connection->getClient()->sessions()->online()->close([
			'referenceNumber' => $referenceNumber,
		]);

		if (204 !== $response?->status()) {
			throw new KSeFClientException(\sprintf($response->body() . '|referenceNumber => %s|status => %s', $referenceNumber, $response?->status()));
		}

		return $response;
	}

	/**
	 * Close batch session.
	 *
	 * @param string $referenceNumber From open session response
	 *
	 * @return ResponseInterface
	 */
	public function closeBatchSession(string $referenceNumber): ResponseInterface
	{
		$response = $this->connection->getClient()->sessions()->batch()->close([
			'referenceNumber' => $referenceNumber,
		]);

		if (204 !== $response?->status()) {
			throw new KSeFClientException(\sprintf($response->body() . '|referenceNumber => %s|status => %s', $referenceNumber, $response?->status()));
		}

		return $response;
	}

	/**
	 * Send invoice to KSeF API.
	 *
	 * @param string  $referenceNumber From open session response
	 * @param Faktura $invoice         Invoice DTO
	 * @param bool    $offline         Whether to send the invoice in offline mode
	 *
	 * @return ResponseInterface
	 */
	public function send(
		Faktura $invoice,
		string $referenceNumber,
		bool $offline = false,
	): ResponseInterface {
		$request = new SendRequest(new ReferenceNumber($referenceNumber), $invoice, $offline);

		$response = $this->connection->getClient()->sessions()->online()->send($request);

		if (202 !== $response?->status()) {
			throw new KSeFClientException(\sprintf($response->body() . '|referenceNumber => %s|status => %s', $referenceNumber, $response?->status()));
		}

		return $response;
	}

	/**
	 * Send invoices in batch to KSeF API.
	 *
	 * @param array $invoices An array of invoice DTOs
	 *
	 * @return ResponseInterface
	 */
	public function sendBatch(array $invoices): ResponseInterface
	{
		$response = $this->connection->getClient()->sessions()->batch()->openAndSend([
			'formCode' => FormCode::Fa3->value,
			'faktury' => $invoices,
		]);

		if (201 !== $response?->status()) {
			throw new KSeFClientException(\sprintf($response->body() . '|status => %s', $response?->status()));
		}

		return $response;
	}

	/**
	 * Get invoice status from KSeF API.
	 *
	 * @param string $sessionReferenceNumber From open session response
	 * @param string $invoiceReferenceNumber From invoice send response
	 *
	 * @return ResponseInterface
	 */
	public function status(
		string $sessionReferenceNumber,
		string $invoiceReferenceNumber,
	): ResponseInterface {
		$response = $this->connection->getClient()->sessions()->invoices()->status([
			'referenceNumber' => $sessionReferenceNumber,
			'invoiceReferenceNumber' => $invoiceReferenceNumber
		]);

		if (200 !== $response->object()?->status?->code) {
			throw new KSeFClientException(\sprintf($response->body() . '|referenceNumber => %s|invoiceReferenceNumber => %s|status => %s', $sessionReferenceNumber, $invoiceReferenceNumber, $response?->status()));
		}

		return $response;
	}

	/**
	 * Get invoices status in batch from KSeF API.
	 *
	 * @param string $sessionReferenceNumber From open session response
	 *
	 * @return ResponseInterface
	 */
	public function statusBatch(string $sessionReferenceNumber): ResponseInterface
	{
		$response = $this->connection->getClient()->sessions()->invoices()->list(
			new ListRequest(new ReferenceNumber($sessionReferenceNumber)),
		);
		if (200 !== $response?->status()) {
			throw new KSeFClientException(\sprintf($response->body() . '|referenceNumber => %s|status => %s', $sessionReferenceNumber, $response?->status()));
		}

		return $response;
	}

	/**
	 * Request an import of invoices from KSeF API.
	 *
	 * @param \DateTime  $from Date from which to import invoices
	 * @param \DateTime  $to   Date to which to import invoices
	 * @param ImportType $type
	 *
	 * @return ResponseInterface
	 */
	public function requestImport(
		\DateTime $from = new \DateTime('-1 day'),
		\DateTime $to = new \DateTime(),
		ImportType $type = ImportType::COST,
	): ResponseInterface {
		$filters = [
			'subjectType' => $type->value,
			'dateRange' => [
				'dateType' => 'PermanentStorage',
				'from' => $from,
				'to' => $to,
			],
		];
		$response = $this->connection->getClient()->invoices()->exports()->init([
			'filters' => $filters,
		]);

		if (201 !== $response?->status()) {
			throw new KSeFClientException(\sprintf($response->body() . '|status => %s', $response?->status()));
		}

		return $response;
	}

	/**
	 * Download an import file of invoices package from KSeF API.
	 *
	 * @param string $referenceNumber
	 *
	 * @return ResponseInterface
	 */
	public function downloadImport(
		string $referenceNumber,
	): ResponseInterface {
		$response = $this->connection->getClient()->invoices()->exports()->status(['referenceNumber' => $referenceNumber]);

		if (200 !== $response?->status()) {
			throw new KSeFClientException(\sprintf($response->body() . '|referenceNumber => %s|status => %s', $referenceNumber, $response?->status()));
		}

		return $response;
	}

	/**
	 * Download details for current certificate from KSeF API.
	 *
	 * @param ?Certificate $certificate
	 *
	 * @return ResponseInterface
	 */
	public function requestCertificateDetails(?Certificate $certificate = null): ResponseInterface
	{
		return $this->connection->getClient()->certificates()->query(
			new QueryRequest(
				type: $certificate ? CertificateType::tryFrom($certificate->getType()->value) : null,
				certificateSerialNumber: $certificate ? new CertificateSerialNumber($certificate?->getSerialNumber()) : null,
			)
		);
	}

	/**
	 * Build KSeF API client from setting.
	 *
	 * @param Setting $setting
	 *
	 * @return ClientResource
	 */
	private function buildClient(Setting $setting): ClientResource
	{
		$encryption = Encryption::getInstance();

		return (new ClientBuilder())
			->withApiUrl($setting->getApiUrl())
			->withCertificate(
				$setting->getCertificate()->getContent(),
				$setting->getPrivateKey(),
				$encryption->decrypt($setting->getPassphrase()),
			)
//			->withLogger(KSeF::getLogger())
			->withIdentifier($setting->getIdentifier())
			->withEncryptionKey($this->encryptionKey)
			->build();
	}
}
