<?php

/**
 * Import Invoices - download package.
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Service\Cron;

use App\CronHandler;
use App\Integrations\KSeF\Client\BatchInvoiceClient;
use App\Integrations\KSeF\Model\Enum\Status;
use App\Integrations\KSeF\Repository\DownloadQueueRepository;
use App\Integrations\KSeF\Service\Reader\XmlDeserializerService;
use App\Integrations\KSeF\Service\Reader\ZipReaderService;

/**
 * Step 2: request and download zip data.
 * Step 3: unpack zip data.
 * Step 4: deserialize XML data to records.
 */
class ImportDownload extends CronHandler
{
	/** @var int */
	private const LIMIT = 10;

	/** {@inheritdoc} */
	public function process()
	{
		$downloadQueueRepository = new DownloadQueueRepository();
		$batchInvoiceClient = new BatchInvoiceClient();

		$queue = $downloadQueueRepository->getByStatus(Status::NEW, self::LIMIT);
		foreach ($queue as $item) {
			$batchInvoiceClient->downloadBatchImport($item->getSettingsId(), $item);
		}
		unset($queue, $batchInvoiceClient);

		$reader = new ZipReaderService();
		$xmlDeserializerService = new XmlDeserializerService();
		foreach ($downloadQueueRepository->getByStatus(Status::IN_PROGRESS, self::LIMIT) as $item) {
			$results = [];
			$xmlJobs = $reader->read($item);
			foreach ($xmlJobs as $xmlJob) {
				$results[] = $xmlDeserializerService->deserialize($xmlJob, $item->getSettingsId());
			}

			$newStatus = \in_array(false, $results, true) ? Status::ERROR : Status::DONE;
			$downloadQueueRepository->update($item->setStatus($newStatus->value));
		}
	}
}
