<?php

/**
 * KSeF logger class file.
 *
 * @package   KSeF
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

declare(strict_types=1);

namespace App\Integrations\KSeF\Service\Logger;

use App\Db;
use App\Log\LoggerPsrAdapter;
use App\TextUtils;

class KSeFLogger extends LoggerPsrAdapter
{
	public const TABLE_NAME = 'l_#__ksef';

	public function log($level, $message, array $context = []): void
	{
		$context['category'] ??= 'KSeF';
		parent::log($level, $message, $context);
		$dbLevel = self::mapLevel((string) $level);

		if ('error' !== $dbLevel && 'info' !== $dbLevel) {
			return;
		}

		Db::getInstance('log')->createCommand()
			->insert(self::TABLE_NAME, [
				'time' => date('Y-m-d H:i:s'),
				'error' => (int) ('error' === $dbLevel),
				'message' => TextUtils::textTruncate($this->getShortMessage($message), 255),
				'params' => TextUtils::textTruncate($this->getParamsAsString($message, $context), 65535),
				'trace' => TextUtils::textTruncate($this->getTrace($context), 65535),
			])->execute();
	}

	public function getParamsAsString(string $message, array $context): string
	{
		$params = $context['params'] ?? [];
		if (str_contains($message, '|')) {
			$params = explode('|', $message);
			array_shift($params);
			$params = array_filter($params);
		}

		return implode("\n", $params);
	}

	/**
	 * @param string $message
	 *
	 * @return string
	 */
	public function getShortMessage(string $message): string
	{
		if (str_contains($message, '|')) {
			$params = explode('|', $message);
			$message = array_shift($params);
		}

		return $message;
	}

	public function getTrace(array $context): string
	{
		/** @var array{ e?: \Throwable } $context */
		$trace = $context['e']?->getTraceAsString() ?? '';

		return 	rtrim(str_replace(ROOT_DIRECTORY . \DIRECTORY_SEPARATOR, '', $trace), PHP_EOL);
	}
}
