<?php
/**
 * KSeF Address Value Resolver class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Service\Mapper\Resolver;

use App\Integrations\KSeF\Model\Mapper\Pair;
use App\Json;

class AddressValueResolver implements ValueResolverInterface
{
	public function supports(Pair $pair, array|\Vtiger_Record_Model|null $data): bool
	{
		return Json::isJson($pair->getField());
	}

	public function resolve(
		Pair $pair,
		array|\Vtiger_Record_Model|null $data,
		ValueResolverRegistry $registry
	): string {
		$keys = ['street', 'buildingNumber', 'localNumber', 'city', 'postCode', 'cityRegion',
			'voivodeship', 'commune', 'country', 'county', 'pobox', 'alternative'];

		extract(array_merge(array_fill_keys($keys, null), Json::decode($pair->getField()) ?: []));

		if ($alternative) {
			$altPair = clone $pair;
			$altPair->setField($alternative);
			$alternativeAddress = trim((string) $registry->resolve($altPair, $data));
			if ($alternativeAddress) {
				return $alternativeAddress;
			}
		}

		$parts = [];

		foreach ($keys as $key) {
			$fieldName = ${$key};
			if ($fieldName) {
				$newPair = clone $pair;
				$newPair->setField($fieldName);
				${$key} = trim((string) $registry->resolve($newPair, $data));
			}
		}

		$streetPart = trim(($street ?? '') . ' ' . ($buildingNumber ?? ''));
		if ($localNumber) {
			$streetPart = rtrim($streetPart) . '/' . $localNumber;
		}
		if ('' !== $streetPart) {
			$parts[] = $streetPart;
		}

		$postalCity = trim(($postCode ?? '') . ' ' . ($city ?? ''));
		if ('' !== $postalCity) {
			$parts[] = $postalCity;
		}

		if ($commune) {
			$parts[] = 'gmina ' . $commune;
		}

		if ($county) {
			$parts[] = 'powiat ' . $county;
		}

		if ($cityRegion) {
			$parts[] = 'region ' . $cityRegion;
		}

		if ($voivodeship) {
			$parts[] = 'woj. ' . $voivodeship;
		}

		if ($country) {
			$parts[] = $country;
		}

		if ($pobox) {
			$parts[] = 'skrytka pocztowa ' . $pobox;
		}

		return implode(', ', $parts);
	}
}
