<?php

/**
 * KSeF Invoice Cost Builder class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Service\Reader\RecordBuilder;

use App\Integrations\KSeF\Exception\KSeFMapperException;
use App\Integrations\KSeF\Exception\KSeFXmlException;
use App\Integrations\KSeF\Service\Mapper\MapperService;
use App\Integrations\KSeF\Service\Reader\RecordBuilder\Rules\AdvanceInvoiceRule;
use App\Integrations\KSeF\Service\Reader\RecordBuilder\Rules\AnnotationsRule;
use App\Integrations\KSeF\Service\Reader\RecordBuilder\Rules\BuyerFieldsRule;
use App\Integrations\KSeF\Service\Reader\RecordBuilder\Rules\CompanyRule;
use App\Integrations\KSeF\Service\Reader\RecordBuilder\Rules\CorrectionRule;
use App\Integrations\KSeF\Service\Reader\RecordBuilder\Rules\FooterRule;
use App\Integrations\KSeF\Service\Reader\RecordBuilder\Rules\HeaderRule;
use App\Integrations\KSeF\Service\Reader\RecordBuilder\Rules\LinesRule;
use App\Integrations\KSeF\Service\Reader\RecordBuilder\Rules\MarginProcedureRule;
use App\Integrations\KSeF\Service\Reader\RecordBuilder\Rules\PartialAdvanceRule;
use App\Integrations\KSeF\Service\Reader\RecordBuilder\Rules\PaymentRule;
use App\Integrations\KSeF\Service\Reader\RecordBuilder\Rules\SellerFieldsRule;
use App\Integrations\KSeF\Service\Reader\RecordBuilder\Rules\SettlementInvoiceRule;
use App\Integrations\KSeF\Service\Reader\RecordBuilder\Rules\SubjectsRule;
use App\Integrations\KSeF\Service\Reader\RecordBuilder\Rules\SummaryRule;
use App\QueryGenerator;

/**
 * KSeF Invoice Cost Builder class.
 *
 * Builds FInvoiceCost records from KSeF XML using rule-based architecture.
 */
final class InvoiceCostBuilder extends AbstractRecordBuilder
{
	/**
	 * {@inheritDoc}
	 */
	protected ?\SimpleXMLElement $xml = null;

	public function __construct(string $moduleName, int $companyId)
	{
		$this->initializeRules();
		$this->mapperService = (new MapperService())->loadMapper($moduleName);
		$this->companyId = $companyId;
	}

	/**
	 * Build invoice record from KSeF XML.
	 *
	 * @param \SimpleXMLElement $xml         The XML element to read from
	 * @param string|null       $xmlFileName Optional XML filename for KSeF number extraction
	 *
	 * @return \Vtiger_Record_Model The populated invoice record
	 */
	public function build(\SimpleXMLElement $xml, ?string $xmlFileName = null): \Vtiger_Record_Model
	{
		$this->xml = $xml;
		$moduleName = $this->getMapperService()->getModuleName();

		$invoiceRecord = \Vtiger_Record_Model::getCleanInstance($moduleName);

		$this->context = [
			'moduleName' => $moduleName,
			'xmlFileName' => $xmlFileName,
			'currencyCode' => $this->getOptionalString('//Fa/KodWaluty') ?: 'PLN',
		];

		$this->applyRules($xml, $invoiceRecord);

		return $invoiceRecord;
	}

	/**
	 * Check if invoice with same parameters already exists.
	 *
	 * @param \SimpleXMLElement $xml
	 *
	 * @throws KSeFMapperException
	 * @throws KSeFXmlException
	 *
	 * @return bool True if exists, False otherwise
	 */
	public function checkIfInvoiceExists(\SimpleXMLElement $xml): bool
	{
		$moduleName = $this->getMapperService()->getModuleName();

		$sellerVatId = $this->getOptionalString('//Podmiot1/DaneIdentyfikacyjne/NIP', $xml);
		$sellerVatField = $this->getMapperService()->getMapper()->get('//Podmiot1/DaneIdentyfikacyjne/NIP')->getField();

		$invoiceNumber = $this->getRequiredString('//Fa/P_2', $xml);
		$invoiceNumberField = $this->getMapperService()->getMapper()->get('//Fa/P_2')->getField();

		$companyField = $this->getMapperService()->getMapper()->get('yf_company')->getField();

		$queryGenerator = new QueryGenerator($moduleName);
		$queryGenerator->permissions = false;
		$queryGenerator->setFields(['id'])
			->addCondition($invoiceNumberField, $invoiceNumber, 'e')
			->addCondition($sellerVatField, $sellerVatId, 'e')
			->addCondition($companyField, $this->companyId, 'eid');

		return $queryGenerator->createQuery()->exists();
	}

	/**
	 * Initialize all rules for building invoice records.
	 *
	 * @return void
	 */
	private function initializeRules(): void
	{
		$this->addRule(new CompanyRule($this));
		$this->addRule(new HeaderRule($this));
		$this->addRule(new SubjectsRule($this));
		$this->addRule(new SellerFieldsRule($this));
		$this->addRule(new BuyerFieldsRule($this));
		$this->addRule(new FooterRule($this));
		$this->addRule(new SummaryRule($this));
		$this->addRule(new CorrectionRule($this));
		$this->addRule(new AdvanceInvoiceRule($this));
		$this->addRule(new LinesRule($this));
		$this->addRule(new PaymentRule($this));
		$this->addRule(new AnnotationsRule($this));
		$this->addRule(new MarginProcedureRule($this));
		$this->addRule(new PartialAdvanceRule($this));
		$this->addRule(new SettlementInvoiceRule($this));
	}
}
