<?php

/**
 * KSeF Reader Lines Rule class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Service\Reader\RecordBuilder\Rules;

use App\Db\Query;
use App\Integrations\KSeF\Service\Reader\RecordBuilder\AbstractRecordBuilder;
use App\Json;

/**
 * KSeF Reader Lines Rule class.
 *
 * Handles building invoice line items (FaWiersz) as inventory data.
 */
final class LinesRule implements RuleInterface
{
	private AbstractRecordBuilder $builder;
	private array $serviceCache = [];
	private array $globalTaxesCache = [];

	/** {@inheritDoc} */
	public function __construct(AbstractRecordBuilder $builder)
	{
		$this->builder = $builder;
	}

	/** {@inheritDoc} */
	public function apply(
		\Vtiger_Record_Model $record,
		array $context = [],
	): void {
		$lines = [];

		$advanceRule = $this->getAdvanceInvoiceRule();
		$isAdvanceInvoice = $advanceRule && $advanceRule->isAdvanceInvoice();

		$lineNodes = $isAdvanceInvoice
			? $this->builder->getXml()->xpath('//Fa/Zamowienie/ZamowienieWiersz')
			: $this->builder->getXml()->xpath('//Fa/FaWiersz');

		if (empty($lineNodes)) {
			return;
		}

		$currencyCode = $context['currencyCode'] ?? 'PLN';
		$currencyId = $this->getCurrencyId($currencyCode);

		$correctionRule = $this->getCorrectionRule();

		foreach ($lineNodes as $lineNode) {
			if ($correctionRule && $correctionRule->shouldExcludeLine($lineNode)) {
				continue;
			}

			$lineData = $this->buildAsInventoryRow($lineNode, $currencyId, $isAdvanceInvoice);
			$lines[] = $lineData;
		}

		$record->initInventoryData($lines, false);
	}

	/**
	 * Build invoice line as inventory row data (for FInvoiceCost and similar modules).
	 *
	 * @param \SimpleXMLElement $lineXml
	 * @param int               $currencyId
	 * @param bool              $isAdvanceInvoice
	 *
	 * @return array
	 */
	private function buildAsInventoryRow(\SimpleXMLElement $lineXml, int $currencyId = 1, bool $isAdvanceInvoice = false): array
	{
		$rowData = [];
		$rowData['currency'] = $currencyId;
		$rowData['discountmode'] = 1;
		$rowData['taxmode'] = 1;

		$p7Key = $isAdvanceInvoice ? 'P_7Z' : 'P_7';
		// unit
		$p8aKey = $isAdvanceInvoice ? 'P_8AZ' : 'P_8A';
		// qty
		$p8bKey = $isAdvanceInvoice ? 'P_8BZ' : 'P_8B';
		// net or gross unit price
		$p9aKey = $isAdvanceInvoice ? 'P_9AZ' : 'P_9A';
		$p8bGross = $isAdvanceInvoice ? 'P_9BZ' : 'P_9B';
		// tax
		$p12Key = $isAdvanceInvoice ? 'P_12Z' : 'P_12';
		// discount
		$p10Key = $isAdvanceInvoice ? 'P_10Z' : 'P_10';

		$p7 = $this->builder->getRequiredString($p7Key, $lineXml);
		$serviceId = $this->findOrCreateService($p7);
		$rowData['name'] = $serviceId;

		$p8a = $this->builder->getOptionalString($p8aKey, $lineXml);
		if ($p8a) {
			$rowData['unit'] = $p8a;
		}

		$p8b = $this->builder->getOptionalDecimal($p8bKey, $lineXml);
		if (null !== $p8b) {
			$rowData['qty'] = $p8b;
		}

		$p12 = $this->builder->getOptionalString($p12Key, $lineXml);
		if ($p12) {
			$taxConfig = $this->getTaxConfigForP12($p12);
			if ($taxConfig) {
				$rowData['taxparam'] = json_encode($taxConfig);
			}
		}

		$p9a = $this->builder->getOptionalDecimal($p9aKey, $lineXml);
		if (null !== $p9a) {
			$rowData['price'] = $p9a;
		} elseif (null !== ($unitPriceB = $this->builder->getOptionalDecimal($p8bGross, $lineXml))) {
			$unitPriceB = (float) $unitPriceB;
			$unitPrice = $unitPriceB / (1 + ((float) $p12 / 100));
			$rowData['price'] = round($unitPrice, 2);
		}

		$p10 = $this->builder->getOptionalDecimal($p10Key, $lineXml);
		if (null !== $p10) {
			$rowData['discount'] = $p10;
			$rowData['discountparam'] = Json::encode([
				'aggregationType' => 'individual',
				'individualDiscountType' => 'amount',
				'individualDiscount' => $p10
			]);
		}
		$uuId = $this->builder->getOptionalString('UU_ID', $lineXml);
		if ($uuId) {
			$rowData['comment1'] = 'UU_ID: ' . $uuId;
		}

		$nrWiersza = $this->builder->getOptionalString('NrWierszaFa', $lineXml);
		if ($nrWiersza) {
			$rowData['seq'] = (int) $nrWiersza;
		}

		return $rowData;
	}

	private function getCurrencyId(string $currencyCode): int
	{
		$id = (new Query())
			->select(['id'])
			->from('vtiger_currency_info')
			->where(['currency_code' => $currencyCode])
			->scalar();

		return (int) ($id ?: 1);
	}

	/**
	 * Get global taxes from database.
	 *
	 * @return array
	 */
	private function getGlobalTaxes(): array
	{
		if (!empty($this->globalTaxesCache)) {
			return $this->globalTaxesCache;
		}

		$this->globalTaxesCache = (new Query())
			->select(['id', 'name', 'value'])
			->from('a_#__taxes_global')
			->where(['status' => 0])
			->indexBy('id')
			->all();

		return $this->globalTaxesCache;
	}

	/**
	 * Get tax configuration for P_12 value from KSeF XML.
	 *
	 * Maps P_12 values to global tax IDs from a_yf_taxes_global table.
	 *
	 * @param string $p12Value P_12 value from XML (e.g., "23", "zw", "np", "0 kr")
	 *
	 * @return array|null Tax configuration array or null
	 */
	private function getTaxConfigForP12(string $p12Value): ?array
	{
		$globalTaxes = $this->getGlobalTaxes();
		$p12Lower = strtolower(trim($p12Value));

		foreach ($globalTaxes as $taxId => $tax) {
			$taxNameLower = strtolower($tax['name']);
			$taxValue = (float) $tax['value'];

			if ('zw' === $p12Lower && 'zw' === $taxNameLower) {
				return [
					'aggregationType' => 'global',
					'globalTax' => 0,
					'globalId' => $taxId,
				];
			}

			if ('np' === $p12Lower && 'np' === $taxNameLower) {
				return [
					'aggregationType' => 'global',
					'globalTax' => 0,
					'globalId' => $taxId,
				];
			}

			if ('oo' === $p12Lower && 'oo' === $taxNameLower) {
				return [
					'aggregationType' => 'global',
					'globalTax' => 0,
					'globalId' => $taxId,
				];
			}

			if (str_contains($p12Lower, 'kr') && str_contains($taxNameLower, 'kr')) {
				return [
					'aggregationType' => 'global',
					'globalTax' => 0,
					'globalId' => $taxId,
				];
			}

			if (is_numeric($p12Value)) {
				$p12Numeric = (float) $p12Value;
				if (abs($taxValue - $p12Numeric) < 0.01) {
					return [
						'aggregationType' => 'global',
						'globalTax' => $taxValue,
						'globalId' => $taxId,
					];
				}
			}
		}

		if (is_numeric($p12Value)) {
			return [
				'aggregationType' => 'individual',
				'individualTax' => (float) $p12Value,
			];
		}

		return null;
	}

	/**
	 * Find or create service by name.
	 *
	 * @param string $serviceName
	 *
	 * @return int Service ID
	 */
	private function findOrCreateService(string $serviceName): int
	{
		if (isset($this->serviceCache[$serviceName])) {
			return $this->serviceCache[$serviceName];
		}

		$serviceId = (new Query())
			->select(['serviceid'])
			->from('vtiger_service')
			->innerJoin('vtiger_crmentity', 'vtiger_service.serviceid = vtiger_crmentity.crmid')
			->where(['vtiger_crmentity.deleted' => 0])
			->andWhere(['vtiger_service.servicename' => $serviceName])
			->scalar();

		if ($serviceId) {
			$this->serviceCache[$serviceName] = (int) $serviceId;
			return (int) $serviceId;
		}

		$serviceRecord = \Vtiger_Record_Model::getCleanInstance('Services');
		$serviceRecord->set('servicename', $serviceName);
		$serviceRecord->set('service_usageunit', '');
		$serviceRecord->set('discontinued', 0);
		$serviceRecord->save();

		$serviceId = $serviceRecord->getId();
		$this->serviceCache[$serviceName] = $serviceId;

		return $serviceId;
	}

	/**
	 * Get CorrectionRule from builder if available.
	 *
	 * @return CorrectionRule|null
	 */
	private function getCorrectionRule(): ?CorrectionRule
	{
		if (empty($this->builder->getRules())) {
			return null;
		}

		foreach ($this->builder->getRules() as $rule) {
			if ($rule instanceof CorrectionRule) {
				return $rule;
			}
		}

		return null;
	}

	/**
	 * Get AdvanceInvoiceRule from builder if available.
	 *
	 * @return AdvanceInvoiceRule|null
	 */
	private function getAdvanceInvoiceRule(): ?AdvanceInvoiceRule
	{
		if (empty($this->builder->getRules())) {
			return null;
		}

		foreach ($this->builder->getRules() as $rule) {
			if ($rule instanceof AdvanceInvoiceRule) {
				return $rule;
			}
		}

		return null;
	}
}
