<?php
/**
 * KSeF Reader Margin Procedure Rule class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Service\Reader\RecordBuilder\Rules;

use App\Integrations\KSeF\Service\Reader\RecordBuilder\AbstractRecordBuilder;

/**
 * KSeF Reader Margin Procedure Rule class.
 *
 * Handles margin procedure field mapping.
 */
final class MarginProcedureRule implements RuleInterface
{
	private AbstractRecordBuilder $builder;

	/** {@inheritDoc} */
	public function __construct(AbstractRecordBuilder $builder)
	{
		$this->builder = $builder;
	}

	/** {@inheritDoc} */
	public function apply(
		\Vtiger_Record_Model $record,
		array $context = [],
	): void {
		$marginProcedure = $this->getMarginProcedure();
		if ($marginProcedure) {
			$this->builder->setFieldIfMapped($record, '//Fa/Adnotacje/PMarzy', $marginProcedure);
		}
	}

	/**
	 * Get margin procedure from XML and map to picklist value.
	 * 
	 * @return string|null The picklist value for margin procedure or null if not applicable.
	 */
	private function getMarginProcedure(): ?string
	{
		$xml = $this->builder->getXml();
		$marginNode = $xml->xpath('//Fa/Adnotacje/PMarzy');
		if (empty($marginNode)) {
			return null;
		}

		if (!empty($xml->xpath('//Fa/Adnotacje/PMarzy/P_PMarzy_2'))) {
			return 'PLL_TRAVEL_AGENCIES';
		}
		if (!empty($xml->xpath('//Fa/Adnotacje/PMarzy/P_PMarzy_3_1'))) {
			return 'PLL_SECOND_HAND_GOODS';
		}
		if (!empty($xml->xpath('//Fa/Adnotacje/PMarzy/P_PMarzy_3_2'))) {
			return 'PLL_WORKS_OF_ART';
		}
		if (!empty($xml->xpath('//Fa/Adnotacje/PMarzy/P_PMarzy_3_3'))) {
			return 'PLL_COLLECTIBLES_AND_ANTIQUES';
		}

		return null;
	}
}
