<?php

/**
 * KSeF Settlement Invoice Rule class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Service\Reader\RecordBuilder\Rules;

use App\Integrations\KSeF\Service\Reader\RecordBuilder\AbstractRecordBuilder;
use App\QueryGenerator;

/**
 * KSeF Settlement Invoice Rule class.
 */
final class SettlementInvoiceRule implements RuleInterface
{
	private AbstractRecordBuilder $builder;

	/** {@inheritDoc} */
	public function __construct(AbstractRecordBuilder $builder)
	{
		$this->builder = $builder;
	}

	/** {@inheritDoc} */
	public function apply(
		\Vtiger_Record_Model $record,
		array $context = [],
	): void {
		$this->createAdvanceInvoiceRelations($record);
	}

	/**
	 * Create relations to advance invoices after settlement invoice is saved.
	 *
	 * This must be called after the invoice has been saved and has an ID.
	 *
	 * @param \Vtiger_Record_Model $recordModel The saved settlement invoice record
	 */
	public function createAdvanceInvoiceRelations(\Vtiger_Record_Model $recordModel): void
	{
		$advanceInvoiceNodes = $this->builder->getXml()->xpath('//Fa/FakturaZaliczkowa');
		if (empty($advanceInvoiceNodes)) {
			return;
		}

		$sellerVatId = $this->builder->getOptionalString('//Podmiot1/DaneIdentyfikacyjne/NIP');

		$ids = [];
		foreach ($advanceInvoiceNodes as $advanceNode) {
			$advanceInvoiceId = $this->findAdvanceInvoice($advanceNode, $sellerVatId);
			if ($advanceInvoiceId) {
				$ids[] = $advanceInvoiceId;
			}
		}

		if ($ids) {
			$recordModel->ext['relations'][] = [
				'relatedModule' => $recordModel->getModuleName(),
				'relatedRecords' => $ids,
			];
		}
	}

	/**
	 * Find advance invoice by KSeF number or invoice number + seller VAT.
	 *
	 * @param \SimpleXMLElement $advanceNode FakturaZaliczkowa node
	 * @param string|null       $sellerVatId Seller VAT ID from Podmiot1
	 *
	 * @return int|null Advance invoice ID
	 */
	private function findAdvanceInvoice(\SimpleXMLElement $advanceNode, ?string $sellerVatId): ?int
	{
		$ksefNumber = $this->builder->getOptionalString('NrKSeFFaZaliczkowej', $advanceNode);
		if ($ksefNumber) {
			$invoiceId = $this->findInvoiceByKsefNumber($ksefNumber);
			if ($invoiceId) {
				return $invoiceId;
			}
		}

		$invoiceNumber = $this->builder->getOptionalString('NrFaZaliczkowej', $advanceNode);
		if ($invoiceNumber && $sellerVatId) {
			return $this->findInvoiceByNumberAndVat($invoiceNumber, $sellerVatId);
		}

		return null;
	}

	/**
	 * Find invoice by KSeF number.
	 *
	 * @param string $ksefNumber
	 *
	 * @return int|null
	 */
	private function findInvoiceByKsefNumber(string $ksefNumber): ?int
	{
		$fieldName = $this->builder->getMapperService()->getMapper()->get('yf_ksef_number')?->getField();
		if (!$fieldName) {
			return null;
		}

		$queryGenerator = new QueryGenerator('FInvoiceCost');
		$queryGenerator->permissions = false;
		$queryGenerator->setFields(['id']);
		$queryGenerator->addCondition($fieldName, $ksefNumber, 'e');

		return $queryGenerator->createQuery()->scalar() ?: null;
	}

	/**
	 * Find invoice by number and seller VAT ID.
	 *
	 * @param string $invoiceNumber
	 * @param string $sellerVatId
	 *
	 * @return int|null
	 */
	private function findInvoiceByNumberAndVat(string $invoiceNumber, string $sellerVatId): ?int
	{
		$sellerVatFieldName = $this->builder->getMapperService()->getMapper()->get('//Podmiot1/DaneIdentyfikacyjne/NIP')?->getField();

		$queryGenerator = new QueryGenerator('FInvoiceCost');
		$queryGenerator->permissions = false;
		$queryGenerator->setFields(['id']);
		$queryGenerator->addCondition('subject', $invoiceNumber, 'e');

		if ($sellerVatFieldName) {
			$queryGenerator->addCondition($sellerVatFieldName, $sellerVatId, 'e');
		}

		return $queryGenerator->createQuery()->scalar() ?: null;
	}
}
