<?php

/**
 * KSeF Reader Subjects Rule class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Service\Reader\RecordBuilder\Rules;

use App\Fields\Country;
use App\Integrations\KSeF\Service\Reader\RecordBuilder\AbstractRecordBuilder;
use App\QueryGenerator;

/**
 * KSeF Reader Subjects Rule class.
 */
final class SubjectsRule implements RuleInterface
{
	private AbstractRecordBuilder $builder;

	/** {@inheritDoc} */
	public function __construct(AbstractRecordBuilder $builder)
	{
		$this->builder = $builder;
	}

	/** {@inheritDoc} */
	public function apply(
		\Vtiger_Record_Model $record,
		array $context = [],
	): void {
		$moduleName = $record->getModuleName();

		if ('FInvoiceCost' === $moduleName) {
			$this->setSellerDataOnInvoice($record);
			//			$this->setMultiCompanyFromBuyer($record);
		}

		// Podmiot3 (third party): Not yet implemented
		// TODO: Add support for Podmiot3 when needed
	}

	/**
	 * Set MultiCompany from Podmiot2 (buyer).
	 *
	 * Creates or finds MultiCompany record and sets it in invoice 'company' reference field.
	 *
	 * @param \Vtiger_Record_Model $record The invoice record
	 */
	private function setMultiCompanyFromBuyer(\Vtiger_Record_Model $record): void
	{
		$multiCompanyId = $this->findOrCreateMultiCompany();
		if ($multiCompanyId) {
			$record->set('companyid', $multiCompanyId);
		}
	}

	/**
	 * Find or create MultiCompany from Podmiot2 (buyer) data.
	 *
	 * @return int|null MultiCompany ID if created/found, null if no VAT ID in XML
	 */
	private function findOrCreateMultiCompany(): ?int
	{
		$vatId = $this->builder->getOptionalString('//Podmiot2/DaneIdentyfikacyjne/NIP');

		if (!$vatId) {
			return null;
		}

		$queryGenerator = new QueryGenerator('MultiCompany');
		$queryGenerator->permissions = false;
		$queryGenerator->setFields(['id']);
		$queryGenerator->addCondition('vat', $vatId, 'e');
		$recordId = $queryGenerator->createQuery()->scalar();

		return (int) $recordId;
	}

	/**
	 * Set Podmiot1 (seller) data directly on FInvoiceCost record.
	 *
	 * Maps seller data from XML to invoice fields using MapperService.
	 *
	 * @param \Vtiger_Record_Model $record The invoice record
	 */
	private function setSellerDataOnInvoice(\Vtiger_Record_Model $record): void
	{
		$vatId = $this->builder->getOptionalString('//Podmiot1/DaneIdentyfikacyjne/NIP');
		if ($vatId) {
			$this->builder->setFieldIfMapped($record, '//Podmiot1/DaneIdentyfikacyjne/NIP', $vatId);
		}

		$name = $this->builder->getOptionalString('//Podmiot1/DaneIdentyfikacyjne/Nazwa');
		if ($name) {
			$this->builder->setFieldIfMapped($record, '//Podmiot1/DaneIdentyfikacyjne/Nazwa', $name);
		}

		$countryCode = $this->builder->getOptionalString('//Podmiot1/Adres/KodKraju');
		if ($countryCode && ($countryName = Country::getCountryName($countryCode))) {
			$this->builder->setFieldIfMapped($record, '//Podmiot1/Adres/KodKraju', $countryName);
		}

		$addressL1 = $this->builder->getOptionalString('//Podmiot1/Adres/AdresL1');
		$addressL2 = $this->builder->getOptionalString('//Podmiot1/Adres/AdresL2');

		$address = trim($addressL1 . ' ' . $addressL2);
		if ($address) {
			$this->builder->setFieldIfMapped($record, '//Podmiot1/Adres/AdresL1', $address);
		}
	}
}
