<?php

/**
 * KSeF ZipReader Service class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Service\Reader;

use App\Integrations\KSeF\Exception\KSeFXmlException;
use App\Integrations\KSeF\KSeF;
use App\Integrations\KSeF\Model\Client\DownloadQueue;
use App\Integrations\KSeF\Model\Enum\Status;
use App\Integrations\KSeF\Model\Reader\XmlJob;
use App\Integrations\KSeF\Repository\DownloadQueueRepository;

/**
 * KSeF ZipReader Service class.
 */
final class ZipReaderService
{
	/**
	 * Read and process ZIP files from KSeF storage.
	 *
	 * @param DownloadQueue $downloadQueue Specific DownloadQueue object to process
	 *
	 * @return XmlJob[] XML job objects array
	 */
	public function read(DownloadQueue $downloadQueue): array
	{
		$path = KSeF::ZIP_PATH;
		$downloadQueueRepository = new DownloadQueueRepository();
		$xmlJobs = [];

		if (!is_dir($path)) {
			$downloadQueueRepository->update($downloadQueue->setStatus(Status::ERROR->value));
			KSeF::getLogger()->error('ZIP path does not exist', ['params' => ['path' => $path]]);
		} else {
			$zipFile = $this->getZipFile($path, $downloadQueue->getReferenceNumber() . '.zip');
			if (!empty($zipFile)) {
				$xmlJobs = $this->processZipFile(
					$zipFile,
					$downloadQueue->getModule(),
				);
				$downloadQueueRepository->update($downloadQueue->setStatus(Status::DONE->value));
			} else {
				$downloadQueueRepository->update($downloadQueue->setStatus(Status::ERROR->value));
				KSeF::getLogger()->error('ZIP file does not exist', ['params' => ['path' => $path, 'reference' => $downloadQueue->getReferenceNumber()]]);
			}
		}

		return $xmlJobs;
	}

	/**
	 * Get ZIP file from directory.
	 *
	 * @param string $path     Directory path
	 * @param string $fileName Specific filename
	 *
	 * @return string Full file path
	 */
	private function getZipFile(string $path, string $fileName): string
	{
		$fullPath = $path . $fileName;

		return file_exists($fullPath) && 'zip' === pathinfo($fullPath, PATHINFO_EXTENSION) ? $fullPath : '';
	}

	/**
	 * Process single ZIP file - extract XMLs and save to database.
	 *
	 * @param string      $zipFilePath Full path to ZIP file
	 * @param string|null $module      Module name
	 *
	 * @return XmlJob[] XML job objects array
	 */
	private function processZipFile(string $zipFilePath, ?string $module): array
	{
		if (!file_exists($zipFilePath) || empty(filesize($zipFilePath))) {
			KSeF::getLogger()->error('ZIP file is empty or unreadable', ['params' => ['path' => $zipFilePath]]);
			$this->deleteZip($zipFilePath);

			return [];
		}

		$xmlJobs = [];
		$zip = new \ZipArchive();
		$result = $zip->open($zipFilePath);

		if (true !== $result) {
			KSeF::getLogger()->error('Failed to open ZIP file', ['params' => ['file' => $zipFilePath, 'error_code' => $result]]);
			$this->deleteZip($zipFilePath);
		} else {
			$filename = '';
			try {
				for ($i = 0; $i < $zip->numFiles; ++$i) {
					$stat = $zip->statIndex($i);
					$filename = $stat['name'];

					if ('xml' === pathinfo($filename, PATHINFO_EXTENSION)) {
						$xmlContent = $zip->getFromIndex($i);

						if (false !== $xmlContent) {
							$xmlJob = (new XmlJob())
								->setXmlName($filename)
								->setXmlPayload($xmlContent)
								->setModule($module);
							$xmlJobs[] = $xmlJob;
						} else {
							throw new KSeFXmlException('Failed to extract XML from ZIP');
						}
					}
				}
				$zip->close();
				$this->deleteZip($zipFilePath);
			} catch (\Throwable $t) {
				KSeF::getLogger()->error($t->getMessage(), ['e' => $t, 'params' => ['zip' => $zipFilePath, 'xml' => $filename]]);
				$zip->close();
				$this->deleteZip($zipFilePath);

				throw $t;
			}
		}

		return $xmlJobs;
	}

	/**
	 * Delete ZIP file from filesystem.
	 *
	 * @param string $zipFilePath Full path to ZIP file
	 *
	 * @return void
	 */
	private function deleteZip(string $zipFilePath): void
	{
		try {
			if (file_exists($zipFilePath)) {
				unlink($zipFilePath);
			}
		} catch (\Throwable $t) {
			KSeF::getLogger()->error($t->getMessage(), ['e' => $t, 'params' => ['file' => $zipFilePath]]);
		}
	}
}
