<?php

/**
 * Provider for Azure AD SSO.
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 * @author Antoni Kiszka <a.kiszka@yetiforce.com>
 */

namespace App\Integrations\UserAuth;

use App\Integrations\Services;
use TheNetworg\OAuth2\Client\Provider\Azure;

/**
 * Azure OAuth provider - class.
 */
class AzureAD extends \App\Authenticator\SSOProvider
{
	/** @var string Session state key */
	private const SESSION_STATE_KEY = 'OAuth2.SSO.state';

	/** {@inheritdoc} */
	public $btnClass = 'btn-primary';
	/** {@inheritdoc} */
	protected $label = 'Azure AD';
	/** {@inheritdoc} */
	protected $icon = 'mdi mdi-microsoft-azure u-fs-xlg';

	/** {@inheritdoc} */
	protected $editFieldNames = ['redirect_uri_id', 'tenant_id', 'client_id', 'client_secret', 'base_user'];
	/** {@inheritdoc} */
	protected $encryptedFields = ['client_secret'];

	/**
	 * OAuth2 provider.
	 *
	 * @var \TheNetworg\OAuth2\Client\Provider\Azure
	 */
	protected $oauthProvider;

	/** {@inheritDoc} */
	public function __construct()
	{
		parent::__construct();
		$config = $this->get('config');
		if ($secret = $config['client_secret'] ?? '') {
			$encryption = \App\Encryptions\Settings::getInstance();
			if ($encryption->isActive()) {
				$secret = $encryption->decrypt($secret);
				if (false === $secret) {
					throw new \App\Exceptions\AppException('ERR_IMPOSSIBLE_DECRYPT');
				}
				$config['client_secret'] = $secret;
				$this->set('config', $config);
			}
		}
	}

	/** {@inheritdoc} */
	public function getAdditionalInfo(): ?string
	{
		return \App\Language::translate('LBL_AZURE_INFO', 'Settings:UserAuth');
	}

	/** {@inheritdoc} */
	public function startAuthorization(): string
	{
		$provider = $this->getProvider();
		$authorizationUrl = $provider->getAuthorizationUrl(['scope' => $provider->scope]);
		// remember the state for later verification
		$state = $provider->getState();
		$hash = sha1($state);
		\App\Session::init();
		\App\Session::set(static::SESSION_STATE_KEY . "{$hash}", $state);
		return $authorizationUrl;
	}

	/** {@inheritdoc} */
	public function finishAuthorization(string $code, string $state): int
	{
		$provider = $this->getProvider();
		$hash = sha1($state);
		\App\Session::init();
		$stateKey = static::SESSION_STATE_KEY . "{$hash}";
		$dbUserId = 0;
		if (!\App\Session::has($stateKey) || $state !== \App\Session::get($stateKey)) {
			throw new \App\Exceptions\Unauthorized('Invalid state', 401);
		}
		\App\Session::delete($stateKey);

		$token = $provider->getAccessToken('authorization_code', [
			'scope' => $provider->scope,
			'code' => $code
		]);

		$userData = $provider->getResourceOwner($token)->toArray();
		$email = $userData['email'] ?? '';
		if (!$email) {
			$userInfo = $provider->get($provider->getRootMicrosoftGraphUri($token) . '/v1.0/me', $token);
			$email = $userInfo['mail'] ?? $userInfo['userPrincipalName'] ?? '';
		}

		if (empty($email) || !\App\Validator::email($email)) {
			\App\Log::warning('No email address returned from Azure AD', 'UserAuthentication');
		} elseif (!$userId = \App\User::getUserIdByEmail($email)) {
			\App\Log::info("User not exists: {$email}", 'UserAuthentication');
			$userbaseId = $this->get('base_user');
			if ($userbaseId && \App\User::isExists($userbaseId, false)) {
				try {
					$dbUserId = $this->createUser($token, $userbaseId);
				} catch (\Throwable $th) {
					\App\Log::error($th->__toString());
				}
			}
		} elseif (!($userModel = \App\User::getUserModel($userId))->isActive()) {
			$userName = $userModel->getDetail('user_name');
			\App\Log::warning("Inactive user: {$userName}:" . $this->getName(), 'UserAuthentication');
		} else {
			$dbUserId = $userModel->getId();
		}

		return $dbUserId;
	}

	/** {@inheritdoc} */
	public function getFieldInstanceByName(string $name): ?\Vtiger_Field_Model
	{
		$moduleName = 'Settings:UserAuth';
		$field = ['uitype' => 1, 'column' => $name, 'name' => $name, 'displaytype' => 1, 'typeofdata' => 'V~M', 'presence' => 0, 'isEditableReadOnly' => false, 'maximumlength' => '100'];
		switch ($name) {
			case 'tenant_id':
				$field['label'] = 'FL_AZURE_TENANT_ID';
				$field['purifyType'] = \App\Purifier::ALNUM2;
				$field['fieldvalue'] = $this->has($name) ? $this->get($name) : '';
				break;
			case 'client_id':
				$field['label'] = 'FL_AZURE_CLIENT_ID';
				$field['purifyType'] = \App\Purifier::TEXT;
				$field['fieldvalue'] = $this->has($name) ? $this->get($name) : '';
				break;
			case 'client_secret':
				$field['uitype'] = 99;
				$field['label'] = 'FL_AZURE_CLIENT_SECRET';
				$field['purifyType'] = 'raw';
				$field['fieldvalue'] = $this->has($name) ? $this->get($name) : '';
				break;
			case 'base_user':
				$field['uitype'] = 16;
				$field['label'] = 'FL_AZURE_BASE_USER';
				$field['purifyType'] = \App\Purifier::ALNUM;
				$field['fieldvalue'] = $this->has($name) ? $this->get($name) : '';
				$field['typeofdata'] = 'V~O';
				$field['tooltip'] = 'LBL_AZURE_BASE_USER_TOOLTIP';
				$field['picklistValues'] = [];
				foreach (\Users_Record_Model::getAllNonAdminUsers(false) as $user) {
					$field['picklistValues'][$user->getId()] = $user->getName();
				}
				break;
			case 'redirect_uri_id':
				$field = [
					'name' => $name,
					'label' => 'FL_REDIRECT_URI_ID',
					'uitype' => 16,
					'typeofdata' => 'I~M',
					'maximumlength' => '2147483647',
					'tooltip' => 'LBL_REDIRECT_URI_ID_DESC',
					'purifyType' => \App\Purifier::INTEGER,
					'fieldvalue' => $this->has($name) ? $this->get($name) : '',
					'picklistValues' => array_map(fn ($service) => $service['name'], Services::getByType(Services::OAUTH))
				];
				break;
			default:
				$field = [];
				break;
		}

		return $field ? \Vtiger_Field_Model::init($moduleName, $field, $name) : null;
	}

	/** {@inheritDoc} */
	public function getAuthorizationHost(): string
	{
		$result = parse_url((new Azure())->getBaseAuthorizationUrl());
		return $result['scheme'] . '://' . $result['host'];
	}

	/**
	 * Get the OAuth2 provider or create it if it doesn't exist.
	 * This method is used to avoid creating the provider before it's needed.
	 *
	 * @return Azure
	 */
	protected function getProvider(): Azure
	{
		if (!empty($this->oauthProvider)) {
			return $this->oauthProvider;
		}
		$this->oauthProvider = new Azure([
			'clientId' => $this->get('client_id'),
			'clientSecret' => $this->get('client_secret'),
			'tenant' => $this->get('tenant_id'),
			'redirectUri' => $this->getRedirectUri(),
			'defaultEndPointVersion' => '2.0'
		]);

		$baseGraphUri = $this->oauthProvider->getRootMicrosoftGraphUri(null);
		$this->oauthProvider->scope = 'openid profile email ' . $baseGraphUri . '/User.Read';

		return $this->oauthProvider;
	}

	/**
	 * Create user.
	 *
	 * @param mixed $token
	 * @param int   $baseUserId
	 *
	 * @return int Newly created user ID
	 */
	private function createUser($token, int $baseUserId): int
	{
		$provider = $this->getProvider();
		$userInfo = $provider->get($provider->getRootMicrosoftGraphUri($token) . '/v1.0/me', $token);
		if (empty($userInfo['mail']) || empty($userInfo['givenName']) || empty($userInfo['surname'])) {
			\App\Log::warning('Create User skipped - no data provider:' . $this->getName(), 'UserAuthentication');
			return 0;
		}
		$baseUser = \Users_Record_Model::getInstanceById($baseUserId, 'Users');
		$resultUser = \Users_Record_Model::getCleanInstance('Users');

		foreach ($baseUser->getModule()->getFields() as $fieldModel) {
			$value = $baseUser->get($fieldModel->getName());
			if ($fieldModel->isDuplicable() && (!$fieldModel->isReferenceField() || \App\Record::isExists($value))
			&& !\in_array($fieldModel->getName(), ['id', 'user_name', 'first_name', 'last_name', 'description', 'primary_phone', 'primary_phone_extra', 'secondary_email', 'authy_methods', 'authy_secret_totp', 'login_method'])) {
				$resultUser->set($fieldModel->getName(), $value);
			}
		}

		$email = $userInfo['mail'];
		$resultUser->set('id', '');
		$resultUser->set('user_name', $email);
		$resultUser->set('email1', $email);
		$resultUser->set('user_password', \App\Encryption::generateUserPassword(20));
		$resultUser->set('description', 'User auto-generated by Azure AD SSO. Template user: ' . $baseUser->getName());
		$resultUser->set('login_method', 'PLL_PASSWORD');
		$resultUser->set('first_name', $userInfo['givenName']);
		$resultUser->set('last_name', $userInfo['surname']);

		\App\User::setCurrentUserId(\App\User::getActiveAdminId());
		$resultUser->save();
		\App\User::setCurrentUserId(0, false);

		return $resultUser->getId();
	}
}
