<?php
/**
 * File connector to find routing. Connector based on service Project-OSRM.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 *
 * @see       http://project-osrm.org
 * @see       https://github.com/Project-OSRM/osrm-backend
 */

namespace App\Map\Routing;

/**
 * Class connector for service Project-OSRM to get routing.
 */
class Osrm extends Base
{
	/**
	 * API URL.
	 */
	private const API_URL = 'https://routing.openstreetmap.de/routed-car';

	/**
	 * @inheritdoc
	 */
	protected string $label = 'LBL_ROUTING_OSRM';

	/**
	 * @inheritdoc
	 */
	protected string $docUrl = 'https://project-osrm.org/docs/v5.24.0/api/#route-service';

	/**
	 * @inheritdoc
	 */
	protected array $formFields = [];

	/** {@inheritdoc} */
	public function calculate(): void
	{
		if (!\App\RequestUtil::isNetConnection()) {
			return;
		}
		$url = self::API_URL . '/route/v1/car/' . implode(';', $this->parsePoints()) . '?' . http_build_query([
			'geometries' => 'geojson',
			'steps' => 'true',
		]);
		\App\Log::beginProfile("GET|OsrmRouting::calculate|{$url}", __NAMESPACE__);
		$response = \App\RequestHttp::getClient()->get($url, [
			'timeout' => 120,
			'http_errors' => false,
		]);
		\App\Log::endProfile("GET|OsrmRouting::calculate|{$url}", __NAMESPACE__);
		if (200 === $response->getStatusCode()) {
			$body = \App\Json::decode($response->getBody());
		} else {
			throw new \App\Exceptions\AppException('Error with connection |' . $response->getReasonPhrase() . '|' . $response->getBody(), 500);
		}
		$this->parseResponse($body);
	}

	/** {@inheritdoc} */
	public function parsePoints(): array
	{
		$tracks = [
			$this->start['lon'] . ',' . $this->start['lat'],
		];
		if (!empty($this->indirectPoints)) {
			foreach ($this->indirectPoints as $tempLon) {
				$tracks[] = $tempLon['lon'] . ',' . $tempLon['lat'];
			}
		}
		$tracks[] = $this->end['lon'] . ',' . $this->end['lat'];
		return $tracks;
	}

	/**
	 * Parse response from API.
	 *
	 * @param array $body
	 *
	 * @return void
	 */
	private function parseResponse(array $body): void
	{
		$coordinates = [];
		if (!empty($body['routes'])) {
			foreach ($body['routes'] as $route) {
				$this->distance += $route['distance'] / 1000;
				if ($this->returnDetails['duration']) {
					$this->travelTime += $route['duration'];
				}
				if ($this->returnDetails['polyline']) {
					$coordinates = array_merge($coordinates, $route['geometry']['coordinates']);
				}
			}
		}
		$this->geoJson = [
			'type' => 'LineString',
			'coordinates' => $coordinates,
		];
	}
}
