<?php
/**
 * Expression model
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Leszek Koziatek <l.koziatek@yetiforce.com>
 * @author Łukasz Krawczyk <l.krawczyk@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Report\Model;

use App\Report\Enums\FunctionType;
use App\Report\Provider\AliasProvider;

/** Expression class */
final class Expression
{
	public function __construct(
		private int $id,
		private Query $query,
		private string $field,
		private ?FunctionType $function,
		private bool $groupBy,
		private ?string $orderBy,
		private int $sort
	) {
	}

	/** Get module name */
	public function getModuleName(): string
	{
		return $this->query->getModuleName();
	}

	/** Get expression id */
	public function getId(): int
	{
		return $this->id;
	}

	/** Get parent Query */
	public function getQuery(): Query
	{
		return $this->query;
	}

	/** Get field expression */
	public function getField(): string
	{
		return $this->field;
	}


	/** Get field name for function */
	public function getFunctionField(): string
	{
		return FunctionType::COUNT === $this->function
			? sprintf('%s(distinct %s)', $this->function->value, $this->field)
			: sprintf('%s(%s)', $this->function->value, $this->field);
	}

	/** Get field name for formula */
	public function getFormulaField(): array
	{
		return json_decode($this->field);
	}

	/** Get function type */
	public function getFunction(): ?FunctionType
	{
		return $this->function;
	}

	/** Check if expression is grouped */
	public function isGroupBy(): bool
	{
		return $this->groupBy;
	}

	/** Get order */
	public function getOrderBy(): ?string
	{
		return $this->orderBy;
	}

	/** Get sort value */
	public function getSort(): int
	{
		return $this->sort;
	}

	/** Get alias for expression */
	public function getAlias(): string
	{
		if (true === $this->isFormula()) {
			return AliasProvider::provideFormulaAlias($this->getFunction(), $this->getFormulaField());
		}

		if (true === str_contains($this->field, ':')) {
			return AliasProvider::provide($this->field, $this->getFunction()?->value);
		}

		return null === $this->getFunction()
			? $this->field
			: sprintf('%s_%s', $this->field, $this->function->value);
	}

	/** Check if expression contains formula */
	public function isFormula(): bool
	{
		json_decode($this->field);

		return json_last_error() === JSON_ERROR_NONE && null !== $this->function;
	}

	/** Check if expression is aggregatable */
	public function isAggregate(): bool
	{
		return $this->function?->isAggregate() ?? false;
	}

	public function isSortable(): bool
	{
		return null !== $this->orderBy && false === $this->isFormula();
	}
}
