<?php

/**
 * Service provides Query Generator ready to return data.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Łukasz Krawczyk <l.krawczyk@yetiforce.com>
 * @author Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 * @author Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Report\Provider;

use App\Config;
use App\Exceptions\AppException;
use App\Report\Builder\QueryGeneratorBuilder;
use App\Report\Builder\TemplateBuilder;
use App\Report\Decorator\ReportDataDecorator;
use App\Report\Enums\FunctionType;
use App\Report\Model\Expression;
use App\Report\Model\Query;
use App\Report\Model\Template;
use App\Report\Report;
use App\Report\Repository\TemplateRepository;
use App\Report\Sanitizer\ReportSanitizer;
use App\Report\Sanitizer\XlsReportFieldSanitizer;

/** QueryGeneratorReportProvider class */
final class QueryGeneratorReportProvider implements ReportProvider
{
	public function __construct(
		private readonly TemplateRepository $templateRepository,
		private readonly QueryGeneratorBuilder $queryGeneratorBuilder,
		private readonly ReportDataDecorator $decorator,
		private readonly ReportSanitizer $sanitizer,
		public readonly TemplateBuilder $builder,
	) {}

	public function getQueryGeneratorBuilder(): QueryGeneratorBuilder
	{
		return $this->queryGeneratorBuilder;
	}

	public function getBuilder(): TemplateBuilder
	{
		return $this->builder;
	}

	/**
	 * Provide data from report.
	 *
	 * @param int  $id
	 * @param int  $userId
	 * @param bool $isPreviewMode
	 *
	 * @throws AppException
	 */
	public function provide(int $id, int $userId, bool $isPreviewMode = false): array
	{
		$template = $this->templateRepository->find($id);
		$data = $this->generateByTemplate($template, $userId, $isPreviewMode);

		return $this->sanitizer->sanitize($template->getQuery(), $data);
	}

	public function provideToXls(int $id, int $userId, bool $isPreviewMode = false): array
	{
		$template = $this->templateRepository->find($id);
		$data = $this->generateByTemplate($template, $userId, $isPreviewMode);
		$sanitizer = Report::get(XlsReportFieldSanitizer::class);

		return $sanitizer->sanitize($template->getQuery(), $data);
	}

	public function provideByData(string $moduleName, array $expressions, array $filters, int $userId = 0, bool $isPreviewMode = false): array
	{
		$query = $this->builder->buildQuery($moduleName);
		$template = $this->builder->build($query, $expressions, $filters);
		$data = $this->generateByTemplate($template, $userId, $isPreviewMode);

		return $this->sanitizer->sanitize($template->getQuery(), $data);
	}

	private function generateByTemplate(Template $template, int $userId, bool $isPreviewMode = false): array
	{
		$query = $template->getQuery();
		$queryGenerator = $this->queryGeneratorBuilder->build($query, $userId);

		if (true === $isPreviewMode) {
			$queryGenerator->setLimit(Config::module('ReportTemplate', 'PREVIEW_ROW_LIMIT', 10));
		}

		$result = array_map(
			fn (array $item): array => $this->decorator->decorate($item, $query),
			$queryGenerator->createQuery()->all(),
		);

		if (true === $isPreviewMode) {
			$result = $this->roundForPreview($query, $result);
		}

		return $result;
	}

	private function roundForPreview(Query $query, array $data): array
	{
		$formulaAliases = array_map(
			static fn (Expression $expression): string => $expression->getAlias(),
			array_filter(
				$query->getExpressions(),
				static fn (Expression $expression): bool => true === $expression->isFormula()
					&& \in_array($expression->getFunction(), [FunctionType::PROD, FunctionType::QUO]),
			),
		);

		$result = [];
		foreach ($data as $key => $rowValue) {
			foreach ($rowValue as $alias => $value) {
				$result[$key][$alias] = (true === \in_array($alias, $formulaAliases))
					? round($value, 2)
					: $value;
			}
		}

		return $result;
	}
}
