<?php

/**
 * Main file that includes basic operations on relations.
 *
 * @package   Relation
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 */

use App\Db;
use App\Db\Query;
use App\QueryGenerator;
use App\Relation\RelationAbstraction;

/**
 * Campaigns_GetCampaignsRecords_Relation class.
 */
class Campaigns_GetCampaignsRecords_Relation extends RelationAbstraction
{
	/**
	 * @var string Name of the table that stores relations.
	 */
	public const TABLE_NAME = 'vtiger_campaign_records';

	/** {@inheritdoc} */
	public function getRelationType(): int
	{
		return Vtiger_Relation_Model::RELATION_M2M;
	}

	/** {@inheritdoc} */
	public function getQuery()
	{
		$queryGenerator = $this->relationModel->getQueryGenerator();
		$queryGenerator->addJoin(['INNER JOIN', self::TABLE_NAME, self::TABLE_NAME . '.crmid=' . $queryGenerator->getColumnName('id')])
			->addNativeCondition([self::TABLE_NAME . '.campaignid' => $this->relationModel->get('parentRecord')->getId()]);
	}

	public function getQueryForReport(QueryGenerator $queryGenerator, array $relField): void
	{
		$tableName = static::TABLE_NAME;
		$query = $queryGenerator->getQueryRelatedField($relField);
		$relTableName = $queryGenerator->resolveRelTableName($relField);
		$relatedTableIndex = $queryGenerator->resolveRelTableIndex($relField);
		$relAlias = $query->getTableName();
		$relColumnNameIndex = \sprintf('%s.%s', $relAlias, $relatedTableIndex);

		$queryGenerator->addJoin(
			[
				'LEFT JOIN',
				$tableName,
				\sprintf('%s.campaignid = %s', $tableName, $queryGenerator->getColumnName('id')),
			],
		);

		$queryGenerator->addJoin([
			'LEFT JOIN',
			\sprintf('%s %s', $relTableName, $relAlias),
			\sprintf(
				'%s.crmid = %s',
				$tableName,
				$relColumnNameIndex
			),
		]);
		$queryGenerator->setDistinct('id');
	}

	/** {@inheritdoc} */
	public function delete(int $sourceRecordId, int $destinationRecordId): bool
	{
		return (bool) Db::getInstance()->createCommand()
			->delete(self::TABLE_NAME, ['crmid' => $destinationRecordId, 'campaignid' => $sourceRecordId])
			->execute();
	}

	/** {@inheritdoc} */
	public function create(int $sourceRecordId, int $destinationRecordId): bool
	{
		$result = false;
		$data = ['campaignid' => $sourceRecordId, 'crmid' => $destinationRecordId];
		if (!(new Query())->from(self::TABLE_NAME)->where($data)->exists()) {
			$data['campaignrelstatusid'] = 0;
			$result = (bool) Db::getInstance()->createCommand()->insert(self::TABLE_NAME, $data)->execute();
		}
		return $result;
	}

	/** {@inheritdoc} */
	public function transfer(int $relatedRecordId, int $fromRecordId, int $toRecordId): bool
	{
		return (bool) Db::getInstance()->createCommand()->update(self::TABLE_NAME, ['campaignid' => $toRecordId], [
			'crmid' => $relatedRecordId, 'campaignid' => $fromRecordId,
		])->execute();
	}
}
