<?php

/**
 * Accept action class file. Proceed to.
 *
 * @package Action
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
declare(strict_types=1);

use App\Controller\Action;
use App\Exceptions\NoPermitted;
use App\Integrations\KSeF\KSeF;
use App\Integrations\KSeF\Model\Client\InvoiceQueue;
use App\Integrations\KSeF\Model\Enum\Status;
use App\Integrations\KSeF\Repository\CertificateRepository;
use App\Integrations\KSeF\Repository\QueueRepository;
use App\Integrations\KSeF\Service\Invoice\InvoiceService;
use App\Language;
use App\Request;

/**
 * FInvoice Accept Action class.
 */
class FInvoice_SendToKSeF_Action extends Action
{
	private Vtiger_Record_Model $recordModel;

	/** {@inheritdoc} */
	public function checkPermission(Request $request)
	{
		$this->recordModel = Vtiger_Record_Model::getInstanceById($request->getInteger('record'), $request->getModule());
		$service = (new InvoiceService($this->recordModel));
		if (!$this->recordModel->isPermitted('KSeF') || !$this->recordModel->isEditable()
			|| (!$service->isAllowedToSend() && !$service->isAllowedToReSend())) {
			throw new NoPermitted('LBL_PERMISSION_DENIED', 406);
		}
	}

	/** {@inheritdoc} */
	public function process(Request $request)
	{
		$moduleName = $request->getModule();
		$response = new Vtiger_Response();
		$queueRepository = new QueueRepository();
		$message = 'LBL_ACCEPT_INVOICE_ADDED_TO_QUEUE';
		$result = false;

		try {
			$settings = (new CertificateRepository())->findByCompanyId($this->recordModel->get('companyid'));
			$invoiceQueue = (new InvoiceQueue())
				->setInvoiceRecordId($this->recordModel->getId())
				->setStatus(Status::NEW->value)
				->setSettingsId($settings->getId());

			$result = $queueRepository->new($invoiceQueue);
		} catch (Throwable $e) {
			$message = Language::translate('ERR_OCCURRED_ERROR');
			KSeF::getLogger()->error($e->getMessage(), ['e' => $e, 'params' => ['record' => $request->getInteger('record')]]);
		}
		$response->setResult([
			'success' => $result,
			'notify' => Language::translate($message, $moduleName),
			'result' => ['_reload' => $result],
			'skipReload' => !$result,
		]);

		$response->emit();
	}
}
