<?php

/**
 * FInvoice KSeF handler file.
 *
 * @package Handler
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Michał Stancelewski <m.stancelewski@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
declare(strict_types=1);

use App\EventHandler;
use App\Integrations\KSeF\Model\Client\InvoiceQueue;
use App\Integrations\KSeF\Model\Enum\Status;
use App\Integrations\KSeF\Service\Invoice\InvoiceService;
use App\Language;

/**
 * FInvoice KSeF handler class.
 */
class FInvoice_KSeF_Handler
{
	/**
	 * InvoiceQueueAfterUpdate handler.
	 * Updates FInvoice record with KSeF queue data.
	 *
	 * @param EventHandler $eventHandler
	 *
	 * @return void
	 */
	public function invoiceQueueAfterUpdate(EventHandler $eventHandler): void
	{
		/** @var InvoiceQueue $queueObject */
		$queueObject = $eventHandler->getParams()['queueObject'];

		if ($queueObject && $queueObject->getInvoiceRecordId()) {
			$this->updateRecord($queueObject->getInvoiceRecordId(), $queueObject);
		}
	}

	/**
	 * DetailViewBefore handler function.
	 *
	 * @param EventHandler $eventHandler
	 */
	public function detailViewBefore(EventHandler $eventHandler): void
	{
		$recordModel = $eventHandler->getRecordModel();
		$invoiceService = new InvoiceService($recordModel);
		if ($recordModel->isPermitted('KSeF')) {
			$link = [];
			if ($invoiceService->isAllowedToSend()) {
				$link = $this->getLinkData($recordModel);
			} elseif ($invoiceService->isAllowedToReSend()) {
				$link = $this->getLinkData($recordModel);
				$link['linkclass'] = 'btn-warning btn-sm js-record-action';
				$link['linklabel'] = 'LBL_RESEND_TO_KSEF';
			}
			if ($link) {
				$links = $eventHandler->getParam('viewLinks');
				$links['DETAIL_VIEW_CUSTOM_GROUP_BUTTONS']['KSeF'][] = Vtiger_Link_Model::getInstanceFromValues($link);
				$eventHandler->addParams('viewLinks', $links);
			}
		}
	}

	private function getLinkData(Vtiger_Record_Model $recordModel): array
	{
		return [
			'linktype' => 'DETAIL_VIEW_ADDITIONAL',
			'linklabel' => 'LBL_SEND_TO_KSEF',
			'dataUrl' => 'index.php?module=' . $recordModel->getModuleName() . '&action=SendToKSeF&record=' . $recordModel->getId(),
			'linkdata' => ['confirm' => Language::translate('LBL_ADD_TO_KSEF_QUEUE_CONFIRM')],
			'linkclass' => 'btn-success btn-sm js-record-action',
			'showLabel' => 1,
		];
	}

	/**
	 * Update record function.
	 *
	 * @param int          $recordId    Record ID to update
	 * @param InvoiceQueue $queueObject Queue object with data
	 *
	 * @return void
	 */
	private function updateRecord(int $recordId, InvoiceQueue $queueObject): void
	{
		$recordModel = Vtiger_Record_Model::getInstanceById($recordId, 'FInvoice');

		$statusEnum = Status::tryFrom($queueObject->getStatus());
		$statusLabel = $statusEnum ? $statusEnum->getLabel() : '';

		$statusField = $recordModel->getField('ksef_status');
		$numberField = $recordModel->getField('ksef_number');
		$dateField = $recordModel->getField('ksef_date');

		$recordModel->set($statusField->getName(), $statusLabel);
		$recordModel->set($numberField->getName(), $queueObject->getKsefNumber());
		$recordModel->set($dateField->getName(), $queueObject->getDateAcquisition());

		$dataForSave = [];
		$dataForSave[$statusField->getTableName()][$statusField->getColumnName()] = $statusLabel;
		$dataForSave[$numberField->getTableName()][$numberField->getColumnName()] = $queueObject->getKsefNumber();
		$dataForSave[$dateField->getTableName()][$dateField->getColumnName()] = $queueObject->getDateAcquisition();
		$recordModel->setDataForSave($dataForSave);

		$recordModel->save();
	}
}
