<?php

/* +***********************************************************************************
* The contents of this file are subject to the vtiger CRM Public License Version 1.0
* ("License"); You may not use this file except in compliance with the License
* The Original Code is:  vtiger CRM Open Source
* The Initial Developer of the Original Code is vtiger.
* Portions created by vtiger are Copyright (C) vtiger.
* All Rights Reserved.
* Contributor(s): YetiForce S.A.
* *********************************************************************************** */

use App\Db\Query;
use App\Exceptions\NoPermitted;
use App\Json;
use App\Log;
use App\QueryGenerator;
use App\RecordSearch;

class Leads_Module_Model extends Vtiger_Module_Model
{
	/**
	 * Function to get list view query for popup window.
	 *
	 * @param string         $sourceModule   Parent module
	 * @param string         $field          parent fieldname
	 * @param string         $record         parent id
	 * @param QueryGenerator $queryGenerator
	 */
	public function getQueryByModuleField($sourceModule, $field, $record, QueryGenerator $queryGenerator)
	{
		if (!empty($record) && \in_array($sourceModule, ['Campaigns', 'Products', 'Services'])) {
			switch ($sourceModule) {
				case 'Campaigns':
					$tableName = 'vtiger_campaign_records';
					$fieldName = 'crmid';
					$relatedFieldName = 'campaignid';
					break;
				case 'Products':
					$tableName = 'vtiger_seproductsrel';
					$fieldName = 'crmid';
					$relatedFieldName = 'productid';
					break;
				default:
					break;
			}

			if ('Services' === $sourceModule) {
				$subQuery = (new Query())
					->select(['relcrmid'])
					->from('vtiger_crmentityrel')
					->where(['crmid' => $record]);
				$secondSubQuery = (new Query())
					->select(['crmid'])
					->from('vtiger_crmentityrel')
					->where(['relcrmid' => $record]);
				$condition = ['and', ['not in', 'vtiger_leaddetails.leadid', $subQuery], ['not in', 'vtiger_leaddetails.leadid', $secondSubQuery]];
			} else {
				$condition = ['not in', 'vtiger_leaddetails.leadid', (new Query())->select([$fieldName])->from($tableName)->where([$relatedFieldName => $record])];
			}
			$queryGenerator->addNativeCondition($condition);
		}
	}

	/**
	 * Function to search accounts.
	 *
	 * @param Vtiger_Record_Model $recordModel
	 *
	 * @throws NoPermitted
	 *
	 * @return bool
	 */
	public function searchAccountsToConvert(Vtiger_Record_Model $recordModel)
	{
		Log::trace('Start ' . __METHOD__);
		if ($recordModel) {
			$mappingFields = Vtiger_Processes_Model::getConfig('marketing', 'conversion', 'mapping');
			$mappingFields = Json::decode($mappingFields);
			$query = (new Query())->select(['vtiger_account.accountid'])
				->from('vtiger_account')
				->innerJoin('vtiger_crmentity', 'vtiger_crmentity.crmid = vtiger_account.accountid')
				->where(['vtiger_crmentity.deleted' => 0]);
			$joinTable = ['vtiger_account', 'vtiger_crmentity'];
			$moduleModel = Vtiger_Module_Model::getInstance('Accounts');
			$focus = $moduleModel->getEntityInstance();
			foreach ($mappingFields as $mappingField) {
				foreach ($mappingField as $leadFieldName => $accountFieldName) {
					$fieldModel = $moduleModel->getFieldByName($accountFieldName);
					if (!$fieldModel) {
						throw new NoPermitted('LBL_PERMISSION_DENIED');
					}
					$tableName = $fieldModel->get('table');
					if (!\in_array($tableName, $joinTable)) {
						$query->innerJoin($tableName, "{$tableName}.{$focus->tab_name_index[$tableName]} = vtiger_account.accountid");
						$joinTable[] = $tableName;
					}
					$query->andWhere(["{$tableName}.{$fieldModel->getColumnName()}" => $recordModel->get($leadFieldName)]);
				}
			}
			$query->limit(2);
			$dataReader = $query->createCommand()->query();
			$numberRows = $dataReader->count();
			if ($numberRows > 1) {
				$dataReader->close();
				Log::trace('End ' . __METHOD__);

				return false;
			}
			if (1 === $numberRows) {
				Log::trace('End ' . __METHOD__);

				return (int) $dataReader->readColumn(0);
			}
		}
		Log::trace('End ' . __METHOD__);

		return true;
	}

	/**
	 * Function that returns status that allow to convert Lead.
	 *
	 * @return array array of statuses
	 */
	public static function getConversionAvaibleStatuses()
	{
		$leadConfig = Settings_MarketingProcesses_Module_Model::getConfig('lead');

		return $leadConfig['convert_status'];
	}

	/**
	 * Function that checks if lead record can be converted.
	 *
	 * @param string $status - lead status
	 *
	 * @return bool whether allowed to convert
	 */
	public static function checkIfAllowedToConvert($status)
	{
		$leadConfig = Settings_MarketingProcesses_Module_Model::getConfig('lead');

		if (empty($leadConfig['convert_status'])) {
			return true;
		}
		return \in_array($status, $leadConfig['convert_status']);
	}

	/**
	 * The function adds restrictions to the functionality of searching for records.
	 *
	 * @param Query             $query
	 * @param RecordSearch|null $recordSearch
	 *
	 * @return void
	 */
	public function searchRecordCondition(Query $query, ?RecordSearch $recordSearch = null): void
	{
		if ($recordSearch->moduleName === $this->getName()) {
			$query->innerJoin('vtiger_leaddetails', 'csl.crmid = vtiger_leaddetails.leadid');
			$query->andWhere(['vtiger_leaddetails.converted' => 0]);
		} else {
			$query->andWhere(['not in', 'csl.crmid', (new Query())->select(['leadid'])->from('vtiger_leaddetails')->where(['converted' => 1])]);
		}
	}
}
