<?php
/**
 * Project module model class.
 *
 * @package   Module
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

use App\Db\Query;
use App\Record;
use yii\db\Expression;

class Project_Module_Model extends Vtiger_Module_Model
{
	/** {@inheritdoc} */
	public function getSideBarLinks($linkParams)
	{
		$links = parent::getSideBarLinks($linkParams);
		$links['SIDEBARLINK'][] = Vtiger_Link_Model::getInstanceFromValues([
			'linktype' => 'SIDEBARLINK',
			'linklabel' => 'LBL_VIEW_GANTT',
			'linkurl' => 'index.php?module=Project&view=Gantt',
			'linkicon' => 'fas fa-briefcase',
		]);
		return $links;
	}

	/**
	 * Update progress and estimated work time in project.
	 *
	 * @param int $id
	 *
	 * @return void
	 */
	public static function updateProgress(int $id): void
	{
		if (!Record::isExists($id)) {
			return;
		}

		$recordModel = Vtiger_Record_Model::getInstanceById($id);
		$progressField = $recordModel->getField('progress');
		$estimatedField = $recordModel->getField('estimated_work_time');
		if ((!$progressField || !$progressField->isActiveField()) && (!$estimatedField || !$estimatedField->isActiveField())) {
			return;
		}

		$estimatedWorkTime = 0;
		$progressInHours = 0;
		$projectIds = static::getChildren($id, [$id]);

		foreach (array_chunk($projectIds, 20) as $partProjectIds) {
			$row = (new Query())
				->select([
					'estimated_work_time' => new Expression('SUM(vtiger_projecttask.estimated_work_time)'),
					'progress_in_hours' => new Expression('SUM(vtiger_projecttask.estimated_work_time * vtiger_projecttask.projecttaskprogress / 100)')
				])
				->from('vtiger_projecttask')
				->innerJoin('vtiger_crmentity', 'vtiger_projecttask.projecttaskid = vtiger_crmentity.crmid')
				->where(['vtiger_projecttask.projectid' => $partProjectIds])
				->andWhere(['vtiger_crmentity.deleted' => [0, 2]])
				->one();

			$estimatedWorkTime += (float) ($row['estimated_work_time'] ?? 0);
			$progressInHours += (float) ($row['progress_in_hours'] ?? 0);
		}

		$projectProgress = $estimatedWorkTime ? round(($progressInHours * 100) / $estimatedWorkTime) : 0;

		$recordModel->set($progressField->getName(), $projectProgress);
		$recordModel->setDataForSave([$progressField->getTableName() => [$progressField->getColumnName() => $projectProgress]]);

		$recordModel->set($estimatedField->getName(), $estimatedWorkTime);
		$recordModel->setDataForSave([$estimatedField->getTableName() => [$estimatedField->getColumnName() => $estimatedWorkTime]]);

		if (false !== $recordModel->getPreviousValue($progressField->getName()) || false !== $recordModel->getPreviousValue($estimatedField->getName())) {
			$recordModel->save();

			$parentProjectId = $recordModel->get('parentid');
			if (!empty($parentProjectId)) {
				static::updateProgress($parentProjectId);
			}
		}
	}

	/**
	 * Get child element IDs by hierarchy.
	 *
	 * @param int   $id
	 * @param array $projectIds
	 *
	 * @return array
	 */
	private static function getChildren(int $id, array $projectIds = []): array
	{
		$ids = (new Query())
			->select(['vtiger_project.projectid'])
			->from('vtiger_project')
			->innerJoin('vtiger_crmentity', 'vtiger_project.projectid = vtiger_crmentity.crmid')
			->where(['vtiger_project.parentid' => $id])
			->andWhere(['vtiger_crmentity.deleted' => [0, 2]])
			->column();

		$ids = array_diff($ids, $projectIds);
		$projectIds = array_merge($projectIds, $ids);

		foreach ($ids as $projectId) {
			$projectIds = static::getChildren($projectId, $projectIds);
		}

		return $projectIds;
	}
}
