<?php

/**
 * Settings KSeF save action file.
 *
 * @package Settings.Action
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
declare(strict_types=1);

use App\Anonymization;
use App\Exceptions\IllegalValue;
use App\Language;
use App\Log;
use App\Request;

/**
 * Settings KSeF save ajax action class.
 */
class Settings_KSeF_SaveAjax_Action extends Settings_Vtiger_Save_Action
{
	/** {@inheritdoc} */
	public function __construct()
	{
		parent::__construct();
		$this->exposeMethod('save');
		$this->exposeMethod('checkConnection');
		$this->exposeMethod('preSaveValidation');
	}

	/**
	 * PreSave validation function.
	 *
	 * @param Request $request
	 *
	 * @return void
	 */
	public function preSaveValidation(Request $request)
	{
		$recordModel = $this->getRecordModelFromRequest($request);
		$response = new Vtiger_Response();
		$response->setResult($recordModel->validate());
		$response->emit();
	}

	/**
	 * Save function.
	 *
	 * @param Request $request
	 *
	 * @return void
	 */
	public function save(Request $request): void
	{
		$response = new Vtiger_Response();

		try {
			$recordModel = $this->getRecordModelFromRequest($request);

			$recordModel->save();
			$recordId = (int) $recordModel->getId();
			$prev = $recordModel->getPreviousValue();
			$post = $recordId ? array_intersect_key($recordModel->getData(), $prev) : $recordModel->getData();

			$anonymization = (new Anonymization())->setModuleName($request->getModule());
			$prev = $anonymization->setData($prev)->anonymize()->getData();
			$post = $anonymization->setValue('*****')->setData($post)->anonymize()->getData();
			Settings_Vtiger_Tracker_Model::addDetail($prev, $post);
			$response->setResult(true);
		} catch (Throwable $e) {
			$response->setError($e->getMessage());
			Log::error($e->__toString());
		}

		$response->emit();
	}

	/**
	 * Check connection function.
	 *
	 * @param Request $request
	 *
	 * @return void
	 */
	public function checkConnection(Request $request): void
	{
		try {
			$recordModel = Settings_KSeF_Record_Model::getInstanceById($request->getInteger('record'));
			$recordModel->updateStatus();
			$recordModel->save();

			$result = [
				'success' => true,
				'message' => Language::translate('LBL_CHECK_CONNECTION_SUCCESS', $request->getModule(false)),
				'reloadList' => true,
			];
		} catch (Exception $e) {
			$result = [
				'success' => false,
				'message' => $e->getMessage(),
			];
		}

		$response = new Vtiger_Response();
		$response->setResult($result);
		$response->emit();
	}

	/**
	 * Function to get the record model based on the request parameters.
	 *
	 * @param Request $request
	 *
	 * @throws IllegalValue
	 *
	 * @return Settings_KSeF_Record_Model
	 */
	protected function getRecordModelFromRequest(Request $request): Settings_KSeF_Record_Model
	{
		if ($request->isEmpty('record')) {
			$recordModel = Settings_KSeF_Record_Model::getCleanInstance();
		} else {
			$recordModel = Settings_KSeF_Record_Model::getInstanceById($request->getInteger('record'));
		}
		$recordModel->setDataFromRequest($request);

		return $recordModel;
	}
}
